;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/typical/kernel"))


;;;; Logical Combinations

;;; These functions are used to implement "finders" for type
;;;  definitions which union or intersect other types.  These
;;;  finders return those types which are ANALYTICALLY subsumed 
;;;  by or subsume a given intersection or union.  In more formal
;;;  terms, these procedures guarantee the completeness of the
;;;  type lattice.
;;; There are three ways that a given intersection or union can
;;;  be known to subsume another type:
;;;      * Inheritance -- One of the types is directly related
;;;          to another in a way that is preserved under the 
;;;          relevant type combination.  For instance, A&B is
;;;          always under every superset of A and every superset
;;;          of B.
;;;      * Merge -- One type may be directly related to BOTH
;;;          elements of the combination in such a way as to make
;;;          it related to the combination.  For instance, every
;;;          intersection of subsets of A with subsets of B is a
;;;          subset of A&B.
;;;      * Joins -- One type may be indirectly related to both A and
;;;          B such as to make a relation to A combined with B.  For
;;;          instance, if A and B have supersets C and D, then C&D
;;;          is a superset of A&B.
;;; The first of these is automatically handled in definitions
;;;  (so that A&B has generalizations A and B) using the transitive
;;;  definition of subsumption.  The second is handled by 
;;;  FIND-ACCIDENTAL-MERGES, which combs through the lattice out from
;;;  the nodes being combined, looking for accidental collusions of 
;;;  familiar subsets.  The third (and most complicated) is handled 
;;;  by FIND-ACCIDENTAL-JOINS and combs the lattice out from the
;;;  supersets or subsets of the nodes being combined.

(define (find-accidental-joins node1 node2 down up join-type left right)
  (let ((marked-nodes ()) (findings (list node1 node2)))
    ;; Marked nodes are known joins.
    (define (marked? node) (memq node marked-nodes))
    (define (mark-node! node)
      (set! marked-nodes (cons node marked-nodes)))
    ;; Joins are nodes all of whose UP's are marked.
    (define (new-join? n)
      (and (not (marked? n))
	   (eq? (first (td-specification n)) join-type)
	   (marked? (left n)) (marked? (right n))))
    (define (check-node! node)
      (define (check-sub-node! n)
	(if (new-join? n)
	 (begin (set! findings (cons n findings)) (check-node! n))))
      (mark-node! node)
      (for-each check-sub-node! (down node))
      #T)
    (maptree check-node! (list node1 node2) up)
    findings))


;;;; Intersections and Unions

;;; ----------------------------------------------------------------
;;;;  TYPE-INTERSECTION combinator (primitive)
;;; This is what you'd expect; an object satisfies the intersection of
;;; two types if it satisfies both the types.  Inferences are
;;; complicated; for details, read the TR.

;;; Stuff for extracting the component types of an intersection.
(define (type-intersection-left x) (second (td-specification x)))
(define (type-intersection-right x) (third (td-specification x)))

;;; This has to handle three-valued satisfaction functions.
(define (type-intersection-function type1 type2)
  (define (in-intersection? x)
    (let ((result1 (satisfies? x type1)))
      (if (undefined? result1)
	  (if (satisfies? x type2) (%undefined) #F)
	  (if (false? result1) #F
	      (let ((result2 (satisfies? x type2)))
		(if (undefined? result2) (%undefined) result2))))))
    in-intersection?)

(define (find-intersection-generalizations type1 type2) 
  (find-accidental-joins type1 type2 td-specializations td-generalizations
			 type-intersection type-intersection-left type-intersection-right))

(define (find-intersection-specializations node1 node2)
  (let ((common ()))
    (maptree (lambda (node) 
	       (cond ((td-<<? node node2) (set! common (cons node common))
		      #F)
		     (else #T)))
	     (list node1) td-specializations)			
    common))

;;; The actual combinator definition:
(define type-intersection
  (type-generator type-intersection-function
		  find-intersection-generalizations
		  find-intersection-specializations))

;;; Printing intersections:
(define (print-conjunction type)
  (let ((spec (td-specification type))) (printout (second spec) and-string (third spec))))
(declare-combinator-printer type-intersection print-conjunction)


;;; ----------------------------------------------------------------
;;;;   TYPE-UNION combinator (primitive)
;;; Again, straightforward.  An object satisfies the union of two
;;; types if it satisfies either one.

;;; Extracting parts of type specifications:
(define (type-union-left x) (second (td-specification x)))
(define (type-union-right x) (third (td-specification x)))

;;; Again, handling three-valued satisfaction information.
(define (type-union-function type1 type2)
  (define (in-union? x)
    (let ((result1 (satisfies? x type1)))
      (cond ((undefined? result1)
	     (let ((result2 (satisfies? x type2)))
	       (if (undefined? result2) (%undefined)
		   (if (false? result2) (%undefined) result2))))
	    ((false? result1)
	     (let ((result2 (satisfies? x type2)))
	       (if (undefined? result2) (%undefined) result2)))
	    (else #T))))
    in-union?)

(define (find-union-generalizations node1 node2)
  ;; Return #T for merges (a homy version of SUBSUMED-BY?)
  (let ((common ()))
    (maptree (lambda (node) 
	       (if (td-<<? node2 node)
		   (begin (set! common (cons node common)) #F)
		   #T))
	     (list node1) td-generalizations)			
    (if (null? common) (list lattice-top) common)))

(define (find-union-specializations type1 type2)
  (find-accidental-joins type1 type2 td-generalizations td-specializations
			 type-union type-union-left type-union-right))

;;; The actual combinator:
(define type-union
  (type-generator type-union-function
		  find-union-generalizations
		  find-union-specializations))

;;; Printing out disjunctions:
(define (print-disjunction type)
  (let ((spec (td-specification type))) (printout (second spec) or-string (third spec))))
(declare-combinator-printer type-union print-disjunction)



;;;; Complement types

;;; This stores the complement of a type for use as a `pre-combinator'
;;; cache. 
(define complement-cache (td-property 'defined-complement))

;;;----------------------------------------------------------------
;;;;  COMPLEMENT combinator
;;; This implements inverse complements as well
;;;  as forward complements; when COMPLEMENT is
;;;  called on a complement, the type it complements
;;;  is returned.

(define type-complement
  (type-generator
    (lambda (type) (type-complement-function type))
    (lambda (type)
      (let ((fringe (get-fringe type td-specializations complement-cache)))
	(if (null? fringe) (list lattice-top) fringe)))
    (lambda (type) (get-fringe type td-generalizations complement-cache))))

(define (complement? type)
  (eq? (first (td-specification type)) type-complement))

;;;; This is what is called to set up disjointness and inverse complements.
(define (complement type)
  (let ((cache-contents (complement-cache type)))
    (if (defined? cache-contents) cache-contents
	(let ((not-type (type-complement type)))
	  (make-disjoint! type not-type)
	  ((modifier complement-cache) not-type type)
	  ((modifier complement-cache) type not-type)
	  not-type))))

(define (type-complement-function type) 
  (let ((predicate (td-predicate type)))
    (define (in-complement? x)
      (let ((result (satisfies? x type)))
	(if (undefined? result) result
	    (if (false? result) #T #F))))
    in-complement?))

;;; Printing complements (I wish I could think of a better format..)
(define (print-type-complement type)
  (printout "Not(" (second  (td-specification type)) ")"))
(declare-combinator-printer type-complement print-type-complement)

