;;; -*- Mode: Lisp; Syntax: Common-lisp; Package: (PSEUDOSCHEME :USE (LISP) :SHADOW (+ - * / ASSOC CASE COMPILE DO ERROR EVAL LAMBDA LET LET* LOOP MAKE-STRING MAP MEMBER #+(OR (NOT Explorer) IEEE-FLOATING-POINT) NIL PEEK-CHAR PRINT READ READ-CHAR T WRITE)) -*-

;;; This file is the source code for "Pseudoscheme", version 163.  There
;;; is a separate user's guide.  Please read it!  If for some reason you
;;; don't have it, send electronic mail to:
;;;         INFO-CLSCHEME-REQUEST@MC.LCS.MIT.EDU
;;;
;;; This program was written by Jonathan Rees, starting in late summer
;;; 1985, with sporadic work in between, including a total rewrite in
;;; summer of '86, and further development at various times in 1987.
;;;
;;; Many thanks for help from Alan Bawden (MIT), Eric Benson (Lucid),
;;; Michael Blair (MIT), Dan Cerys (Texas Instruments), Eric Kennedy (S-cubed),
;;; Kent Pitman (Symbolics), John Ramsdell (MITRE), and Thomas A. Russ (MIT).
;;;
;;; Please, this is a quick hack, not a model of clean code.  Don't
;;; entertain any illusions.  It's in need of major overhaul.
;;;
;;; In Lucid CL, do (delete-package 'pseudoscheme) after compiling if you
;;; want to load it into the same Lisp as the one in which it was compiled.
;;; 
;;; Edit history:
;;;
;;;  12-20-87  JAR  - 161: fixed a bug whereby macros couldn't be used in
;;;		      the file in which they were defined.
;;;		      Fixed error reporting for out-of-context unquote and
;;;		      unquote-splicing.
;;;  12-19-87  JAR  - 160: Flushed the "incorrect speed hacks".
;;;		      Export HEUR, B, O, D, X for number/string conversion.
;;;  10-22-87 Cerys - 159: Added Explorer-specific (Rel3+ only) support for
;;;                    Zmacs, (scheme), and readtables.
;;;		       Use -*-Mode:Scheme-*- now.
;;;  10-14-87  JAR  - 158: added Russ fix conditionalized for slime rel 6.
;;;  10-12-87  JAR  - 157: Fixed (letrec ((foo (lambda x ...))) ...) problem.
;;;		      Added Bawden/Russ fix for slime lisp syntax.
;;;   9-27-87  JAR  - Hacked BEGIN to allow define-macro's inside of them.
;;;		      Unexported .lambda-body and .functional since they
;;;		      were interfering with user variables of those names.
;;;   9-16-87  JAR  - Added a dummy WRITE that special-cases () and #t.
;;;                   Defined QUIT.
;;;		      Some command loop and editor eval/print hacking for
;;;		      Slime.  Face lift to the (scheme) entry function.
;;;   9-16-87 Ziggy - Ver.150: Implemented TI-REL2 directive and supplied
;;;                   pathname for (REQUIRE "Common Lisp for Exploder" ...)
;;;   9-5-87   JAR  - :input and :output instead of :in and :out.
;;;		      Fixed bug in list->string.
;;;   5-28-87  JAR  - fixed bugs in PROCEDURE?.
;;;   3-27-87  JAR  - Added defstruct bug workaround for exploder.
;;;   3-15-87  JAR  - #+DEC (shadow '(sequence type)) should make VAX LISP
;;;		      compiler stop emitting bogus warning message.
;;;   3-13-87  JAR  - Formal parameter type erasure works now (this is a
;;;		      hack for Gifford's group).
;;;   3-12-87  JAR  - Added new macro proposal (syntax tables, PREPROCESS,
;;;		      and friends)
;;;		    - Various changes to the preprocessor
;;;		    - -*-Mode:Scheme;-*- should work on 3600's now
;;;		    - EXACT? is now same as RATIONALP; INEXACT? is FLOATP;
;;;		      EXACT->INEXACT is FLOAT; INEXACT->EXACT is RATIONALIZE
;;;   2-19-87  JAR  - Added :IF-EXISTS :NEW-VERSION in calls to OPEN.
;;;		      Fixed a bug in COMPILE which showed up in VAX LISP
;;;		      but not in SCL.
;;;   2-17-87  JAR  - Added COMPILE.
;;;   2-13-87  JAR  - Separated the scheme-readtable, conditionally
;;;		      on not being on an exploder.  
;;;		    - Added #D.
;;;		    - Lisp macros (and special forms, sort of) can
;;;		      now be used in Scheme code.
;;;   2-11-87  JAR  - Added -*-Syntax:Scheme;-*- hacking for Symbolics.
;;;		    - Almost made the scheme-readtable be separate from
;;;		      CL readtable, but decided not to because that might
;;;		      break the exploder (and maybe Xerox or Lucid?)
;;;		      versions.
;;;     11-86       - The preprocessor comes to life, including alpha
;;;		      conversion and hairy LETREC analysis.
;;;      7-86       - CLSCH dies and Pseudoscheme is born out of the ashes.
;;;      8-85	    - CLSCH written, a first attempt at a Scheme
;;;		      "compatibility package" for Common Lisp.

;;; Bug:
;;;    (eqv? "" "")  and  (eqv? '#() '#())  =>  #f
;;; but the Scheme spec will be changed next time around to allow this.

;;; The Explorer rel1 and rel2 systems don't have a Common Lisp package.
;;; There is a separate file which provides one, which you'll need to
;;; obtain in order to run this program.

#+Explorer
(eval-when (lisp:eval load lisp:compile)
  (pushnew 
    (if (lisp:member :IEEE-FLOATING-POINT *features*)
	 :TI-REL3
	 :TI-REL2)
    *features*))

#+TI-REL2
(require "Common Lisp for Exploder" "ZERMATT:>6821>CL")

;;; Package setup 

(lisp:in-package 'pseudoscheme :use '(lisp))   	; Use underlying LISP support

(shadow '(
          + - * /
          assoc
	  case
	  compile
	  do
	  error
	  eval
          lambda
	  let
	  let*
	  loop		    ;Symbolics Rel 7 (beta) loses
          make-string
          map
          member
	  #-TI-REL2 nil    ;Shadowing NIL wedges REL2 Exploders
	  peek-char
	  print
          read
          read-char
	  #+DEC sequence
	  t
	  #+DEC type
          write
          ))

(export '(
	  ;; CL stuff:
	  in-package
	  use-package
	  export         
	  import
	  provide
	  require
	  shadow
	  trace
	  untrace
	  compile-file
	  describe
	  ed
	  ;; Random features:
	  declare-pseudoscheme-package
	  scheme
	  ;; Random symbols:
	  b o d x heur
	  ))

(eval-when (lisp:eval load lisp:compile)
#-TI-REL2
(defconstant nil  lisp:nil)		;Exploder -- Cannot shadow NIL
(defconstant t	  lisp:t)
(defconstant else lisp:t)

(defparameter *preprocess?* nil)  ;set to t after bootstrap is complete
)
#|

Implementing Scheme variables well is not at all easy.  Here is
a description of the hair that Pseudoscheme employs to try to make
optimal use of Common Lisp value and function cells.

For each global variable, we keep track of a "reference history"
and its current "definition method".

The reference history is some subset of the following keywords:
  :variable        x
  :setq            (setq x ...)
  :set-globally    (set-global-value 'x ...)
  :function	   (x ...)   at a point when definition status
			     was  :function
  :macro	   (x ...)   at a point when definition status
			     was  :macro

The definition method is one of:
  :variable   (setq x ...) / (defun x (&rest args) (apply x args))
  :constant   (defconstant x ...)
  :function   (defun x ...)	   / (setq x #'x)
  :macro      (defmacro x ...)	   / (setq x #'x[closed])

This information is used by the DEFINE and SET! macros, and by
the preprocessor, to decide how to generate code in various
situations.

1. When we see a call (x ...), we may generate either
   (x ...) or (funcall x ...).
      Correctness considerations:
        -- (funcall x ...) is always correct.
        -- (x ...) is correct iff method isn't :constant.
      Efficiency considerations:
        -- (funcall x ...) is preferable if status is :variable.
        -- (x ...) is preferable otherwise.
   Therefore, we generate (funcall x ...) if method is :variable
   or :constant, and (x ...) otherwise.

2. When we see an assignment (set! x ...), we may generate either
   (setq x ...) or (set-global-value 'x ...).
      Correctness considerations:
        -- We are quite screwed if status is :macro.
        -- setq is correct only if status is (and/or will be) :variable.
	-- set-global-value is always correct.
      Efficiency considerations:
        -- Want (setq x ...) if at all possible, since set-global-value
	   is an expensive out-of-line call.
   Therefore we generate (setq x ...) if method is :variable or
   undetermined, and (set-global-value 'x ...) otherwise.

3. When we see (define x ...), we can generate any of the four methods
   enumerated above.
      Correctness considerations:
        -- :variable is always correct,
	-- :function is correct iff [rhs is lambda and] :setq isn't in
	   history,
        -- :macro is correct iff [rhs is lambda and] :function and :setq
	   aren't in history.
      Efficiency considerations:
        -- :variable is never preferable (unless :setq is in history,
	   in which case it's forced).
	-- :macro is preferable when it's correct, the function
	   body is small, and automagic open-coding has been enabled.

A reference to a global variable x could in principle be transformed
by the preprocessor into #'x, if its definition status isn't :macro.
However this probably won't buy us any performance, so the preprocessor
always leaves references alone.

It is possible to get screwed by this mechanism, if you compile a file
under one set of assumptions and then load it where an incompatible set
is in effect.  This would be hard to fix without sacrificing performance.
C'est la guerre.

|#

(eval-when (lisp:eval load lisp:compile)

(defun get-method (var)
  (get var 'method))

(defun use-method (var method)
  (cond
   ((and (eq method :function)
	 (lisp:member :setq (get var 'history)))
    (cerror "proceed as if everything was all right"
	    "(DEFINE ~S ...) is being implemented as (DEFUN ~S ...), but~
	   ~%some previously generated code assumed that it was OK to~
	   ~%implement (SET! ~S ...) as (SETQ ~S ...)."
	    var var var var))
   ((and (eq method :macro)
	 (lisp:member :function (get var 'history)))
    (cerror "proceed as if everything was all right"
	    "(DEFINE ~S ...) is being implemented as (DEFMACRO ~S ...), but~
	   ~%some code containing a call (~S ...) was compiled before this~
	   ~%definition was seen."
	    var var var)
    (setf (get var 'history) (remove :function (get var 'history)))))
  (setf (get var 'method) method))

(defun record-variable-usage (var usage)
  (pushnew usage (get var 'history)))

(use-method 't   :variable)
(use-method 'nil :variable)

;;; Generally useful utility:

(defun concatenate-symbol (&rest things)
  (values (intern (apply #'concatenate 'string
                         (mapcar #'(lisp:lambda (thing)
                                     (if (numberp thing)
                                         (with-output-to-string (s) (princ thing s))
                                         (string thing)))
                                 things)))))
);nehw-lave

;;; Scheme names database

(eval-when (lisp:eval load lisp:compile)
(defmacro entry-type (entry) `(car ,entry))
(defmacro entry-name (entry) `(cadr ,entry))
(defmacro entry-dests (entry) `(caddr ,entry))
(defmacro entry-provision (entry) `(cadddr ,entry))

(defun inherit? (entry)
  (and (eq (entry-type entry) 'proc)
       (lisp:member (entry-provision entry) '(cl ~cl))))

(defun renaming? (entry)
  (lisp:let ((prov (entry-provision entry)))
    (and (eq (entry-type entry) 'proc)
	 (consp prov)
	 (lisp:member (car prov) '(=cl ~cl)))))
);nehw-lave

(defun set-up-database-entry (entry)
  (lisp:let ((name (entry-name entry)))
    (setf (get name 'scheme-info) entry)
    (cond ((inherit? entry)
	   ;; Inheriting from CL -- just set the value cell.
	   (if (and (boundp name)
		    (not (eq (symbol-value name)
			     (symbol-function name))))
	       (cerror "clobber the value that's already there"
		       "package pollution! -- ~s is bound"
		       name))
	   (use-method name :function))
	  ((renaming? entry)
	   (lisp:let ((src (cadr (entry-provision entry))))
	     (if (and (boundp name)
		      (not (eq (symbol-value name)
			       (symbol-function src))))
		 (cerror "clobber value that's already there"
			 "package pollution! -- ~s is bound"
			 name))
	     (if (and (fboundp name)
		      (not (macro-function name)))
		 (cerror "clobber function that's already there"
			 "package pollution! -- ~s is defined"
			 name))
	     (use-method name :macro))))
    (entry-name entry)))

;;; DEFMACRO is the most reliable way to get efficient compilation.
;;; PROCLAIM INLINE may lose.

(defmacro define-scheme-database (&body db)
  `(progn 'lisp:compile
          (defparameter scheme-names
	    (mapcar #'set-up-database-entry ',db))
          ;; Work around @#%$$% 3600 bug.
	  ,@(lisp:let ((names (mapcar #'(lisp:lambda (entry)
					  (entry-name entry))
				      db)))
	      (if (< (length names) #+Symbolics 200 #-Symbolics 1000000)
		  `((export ',names))
		  `((export ',(subseq names 0 200))
		    (export ',(subseq names 200)))))
          ,@(mapcan #'(lisp:lambda (entry)
			(lisp:let ((name (entry-name entry)))
			  (cond ((inherit? entry)
				 `((defparameter ,name #',name)))
				((renaming? entry)
				 (lisp:let ((src
					     (cadr (entry-provision entry))))
				   `((defmacro ,name (&rest x) `(,',src ,@x))
				     (defparameter ,name #',src))))
				(t '()))))
		    db)))

;;; Entries within each major category are in alphabetical order.
;;; "ps" stands for "pseudoscheme".
;;; "DESTS" is a list of the dialects to which this feature belongs:
;;;    E = essential subset of Revised^3 Scheme
;;;    R = non-essential Revised^3 Scheme
;;;    T = Yale Scheme
;;;    M = MIT Scheme
;;;    S = used in Abelson & Sussman

(define-scheme-database
;;(TYPE   NAME                         (DESTS) PROVIDED-BY)
  (random else			       (E T S) ps)
  (value nil			       (R T S) #-TI-REL2 ps #+TI-REL2 CL)
  (value t			       (R T S) ps)
  (value the-empty-stream              (    S) ps)
  (value user-initial-environment      (    S) ps)
  (syntax and                          (R T  ) CL)
  (syntax begin                        (E    ) ps)
  (syntax case                         (R T  ) ps)
  (syntax cond                         (E T S) CL)
  (syntax cons-stream                  (    S) ps)
  (syntax define                       (E T S) ps)
  (syntax define-macro                 (  T  ) ps)
  (syntax delay                        (R T S) ps)
  (syntax do                           (R T  ) ps)
  (syntax if                           (E T S) CL)
  (syntax lambda                       (E T S) ps)
  (syntax let                          (E    ) ps)
  (syntax let*                         (R T  ) ps)
  (syntax letrec                       (E    ) ps)
  (syntax or                           (R T  ) CL)
  (syntax quasiquote                   (R    ) ps)
  (syntax quote                        (E T S) CL)
  (syntax sequence                     (    S) (= begin))
  (syntax set!                         (E    ) ps)
  (syntax unquote		       (     ) ps)
  (syntax unquote-splicing	       (     ) ps)
  (proc *                              (R T  ) (=CL lisp:*))
  (proc +                              (R T  ) (=CL lisp:+))
  (proc -1+                            (  T S) (=CL 1-))
  (proc -                              (R T  ) (=CL lisp:-))
  (proc /                              (E T S) (=CL lisp:/))
  (proc 1+                             (  T S) CL)
  (proc <=                             (E T S) CL)
  (proc <                              (E T S) CL)
  (proc =                              (E T S) CL)
  (proc >=                             (E T S) CL)
  (proc >                              (E T S) CL)
  (proc abs                            (E T S) CL)
  (proc acos                           (R T  ) CL)
  (proc angle			       (R    ) (=CL phase))
  (proc append                         (E T  ) CL)
  (proc apply                          (E T  ) ~CL)
  (proc asin                           (R T  ) CL)
  (proc assoc                          (E   S) ps)
  (proc assq                           (E T S) ps)
  (proc assv                           (E    ) (=CL lisp:assoc))
  (proc atan                           (R    ) CL)
  (proc atom?                          (  T S) (=CL atom))  ;Yuck
  (proc boolean?                       (E T  ) ps)
  (proc caaaar                         (E T S) CL)
  (proc caaadr                         (E T S) CL)
  (proc caaar                          (E T S) CL)
  (proc caadar                         (E T S) CL)
  (proc caaddr                         (E T S) CL)
  (proc caadr                          (E T S) CL)
  (proc caar                           (E T S) CL)
  (proc cadaar                         (E T S) CL)
  (proc cadadr                         (E T S) CL)
  (proc cadar                          (E T S) CL)
  (proc caddar                         (E T S) CL)
  (proc cadddr                         (E T S) CL)
  (proc caddr                          (E T S) CL)
  (proc cadr                           (E T S) CL)
  (proc call-with-current-continuation (E    ) ps)
  (proc call-with-input-file           (E    ) ps)
  (proc call-with-output-file          (E    ) ps)
  (proc car                            (E T S) CL)
  (proc cdaaar                         (E T S) CL)
  (proc cdaadr                         (E T S) CL)
  (proc cdaar                          (E T S) CL)
  (proc cdadar                         (E T S) CL)
  (proc cdaddr                         (E T S) CL)
  (proc cdadr                          (E T S) CL)
  (proc cdar                           (E T S) CL)
  (proc cddaar                         (E T S) CL)
  (proc cddadr                         (E T S) CL)
  (proc cddar                          (E T S) CL)
  (proc cdddar                         (E T S) CL)
  (proc cddddr                         (E T S) CL)
  (proc cdddr                          (E T S) CL)
  (proc cddr                           (E T S) CL)
  (proc cdr                            (E T S) CL)
  (proc ceiling                        (R    ) ~CL)
  (proc char->integer                  (E    ) (=CL char-code))
  (proc char-alphabetic?               (R    ) (=CL alpha-char-p))
  (proc char-ci<=?                     (R    ) (=CL char-not-greaterp))
  (proc char-ci<?                      (R    ) (=CL char-lessp))
  (proc char-ci=?                      (R    ) (=CL char-equal))
  (proc char-ci>=?                     (R    ) (=CL char-not-lessp))
  (proc char-ci>?                      (R    ) (=CL char-greaterp))
  (proc char-downcase                  (R    ) CL)
  (proc char-lower-case?               (R    ) (=CL lower-case-p))
  (proc char-numeric?                  (R    ) (=CL digit-char-p))
  (proc char-ready?		       (R    ) ?)
  (proc char-upcase                    (R    ) CL)
  (proc char-upper-case?               (R    ) (=CL upper-case-p))
  (proc char-whitespace?               (R    ) ps)
  (proc char<=?                        (E    ) (=CL char<=))
  (proc char<?                         (E    ) (=CL char<))
  (proc char=?                         (E    ) (=CL char=))
  (proc char>=?                        (E    ) (=CL char>=))
  (proc char>?                         (E    ) (=CL char>))
  (proc char?                          (E T  ) (=CL characterp))
  (proc close-input-port	       (R    ) (=CL close))
  (proc close-output-port	       (R    ) (=CL close))
  (proc compile			       (     ) ps)
  (proc complex?                       (E    ) (=CL numberp))
  (proc cons                           (E T S) CL)
  (proc cons*                          (  T  ) (=CL list*))  ;For quasiquote
  (proc cos                            (R T S) CL)
  (proc current-input-port             (E    ) ps)
  (proc current-output-port            (E    ) ps)
  (proc denominator		       (R    ) CL)
  (proc display                        (R T  ) ps)
  (proc empty-stream?                  (    S) (=CL null))
  (proc eof-object?                    (E    ) ps)
  (proc eq?                            (E T S) (=CL eq))
  (proc equal?                         (E    ) (=CL equal))
  (proc eqv?                           (E    ) (=CL eql))
  (proc error                          (  T S) ps)
  (proc eval                           (  T S) ps)
  (proc even?                          (E T  ) (=CL evenp))
  (proc exact?                         (E    ) (=CL rationalp))
  (proc exact->inexact                 (R    ) (=CL float))
  (proc expt                           (R T S) CL)
  (proc exp                            (R T S) CL)
  (proc floor                          (R    ) ~CL)
  (proc for-each                       (R    ) (~CL mapc))
  (proc force                          (R   S) ps)
  (proc gcd                            (R   S) CL)
  (proc head                           (    S) (=CL car))
  (proc imag-part		       (R    ) (=CL imagpart))
  (proc inexact?                       (E    ) (=CL floatp))
  (proc inexact->exact                 (R    ) (=CL rationalize))
  (proc input-port?                    (E    ) ps)
  (proc integer->char                  (E    ) (=CL code-char))
  (proc integer?                       (E T  ) (=CL integerp))
  (proc last-pair                      (R    ) (=CL last))
  (proc lcm                            (R    ) CL)
  (proc length                         (E T S) CL)
  (proc list                           (E T S) CL)
  (proc list->string                   (E T  ) ps)
  (proc list->vector                   (E T  ) ps)
  (proc list-ref                       (R    ) ps)
  (proc list-tail                      (R    ) ps)
  (proc load                           (E T  ) ~CL)
  (proc log                            (R T  ) CL)
  (proc magnitude		       (R    ) (=CL abs))
  (proc make-polar		       (R    ) ps)
  (proc make-rectangular	       (R    ) (=CL complex))
  (proc make-string                    (E T S) ps)
  (proc make-vector                    (E T S) ps)
  (proc map                            (R T  ) (~CL mapcar))
  (proc max                            (E T S) CL)
  (proc member                         (E T S) ps)
  (proc memq                           (E T S) ps)
  (proc memv                           (E    ) (=CL lisp:member))
  (proc min                            (E T S) CL)
  (proc modulo                         (R    ) (=CL mod))
  (proc negative?                      (E    ) (=CL minusp))
  (proc newline                        (E T  ) (=CL terpri))
  (proc not                            (E T S) CL)
  (proc null?                          (E T S) (=CL null))
  (proc number->string                 (R    ) ps)
  (proc number?                        (E T S) (=CL numberp))
  (proc numerator		       (R    ) CL)
  (proc odd?                           (E T  ) (=CL oddp))
  (proc open-input-file		       (R    ) ps)
  (proc open-output-file	       (R    ) ps)
  (proc output-port?                   (E    ) ps)
  (proc pair?                          (E T  ) (=CL consp))
  (proc positive?                      (E T  ) (=CL plusp))
  (proc pp			       (  T M) ps)
  (proc procedure?		       (E    ) ps)
  (proc quit			       (    M) ps)
  (proc quotient                       (E T S) ps)
  (proc random                         (    M) CL)
  (proc rational?                      (E T  ) (=CL rationalp))
  (proc rationalize                    (R    ) CL)
  (proc read-char                      (E T  ) ps)
  (proc read                           (E T  ) ps)
  (proc real?                          (E T  ) ps)
  (proc real-part		       (R    ) (=CL realpart))
  (proc remainder                      (E T S) (=CL rem))
  (proc reverse                        (R T  ) CL)
  (proc round                          (R    ) ~CL)
  (proc set-car!                       (E    ) ps)
  (proc set-cdr!                       (E    ) ps)
  (proc sin                            (R   S) CL)
  (proc sqrt                           (R T S) CL)
  (proc string->list                   (E T  ) ps)
  (proc string->number                 (R T  ) ps)
  (proc string->symbol                 (R T  ) ps)
  (proc string-append                  (E T  ) ps)
  (proc string-ci<=?                   (R    ) (=CL string-not-greaterp))
  (proc string-ci<?                    (R    ) (=CL string-lessp))
  (proc string-ci=?                    (R    ) (=CL string-equal))
  (proc string-ci>=?                   (R    ) (=CL string-not-lessp))
  (proc string-ci>?                    (R    ) (=CL string-greaterp))
  (proc string-copy		       (R    ) (=CL copy-seq))
  (proc string-fill!		       (R    ) (=CL fill))
  (proc string-length                  (E T  ) (=CL length))
  (proc string-ref                     (E    ) (=CL char))
  (proc string-set!		       (R    ) ps)
  (proc string<=?                      (E    ) (=CL string<=))
  (proc string<?                       (E    ) (=CL string<))
  (proc string=?                       (E    ) (=CL string=))
  (proc string>=?                      (E    ) (=CL string>=))
  (proc string>?                       (E    ) (=CL string>))
  (proc string?                        (E T  ) (=CL stringp))
  (proc substring                      (E    ) (=CL subseq))
  (proc symbol->string                 (E T  ) (=CL symbol-name))
  (proc symbol?                        (E T  ) ps)
  (proc tail                           (    S) ps)
  (proc tan                            (R T  ) CL)
  (proc transcript-on		       (R T  ) (=CL dribble))
  (proc transcript-off		       (R T  ) (=CL dribble))
  (proc truncate                       (R    ) ~CL)
  (proc vector->list                   (E T  ) ps)
  (proc vector-fill!		       (R    ) (=CL fill))
  (proc vector-length                  (E T  ) (=CL length))
  (proc vector-ref                     (E   S) (=CL svref))
  (proc vector-set!                    (E   S) ps)
  (proc vector?                        (E T  ) ps)
  (proc vector                         (E T  ) CL)  ;(vector) returns wrong empty vector
  (proc with-input-from-file           (R    ) ps)
  (proc with-output-to-file            (R    ) ps)
  (proc write-char                     (E    ) CL)
  (proc write                          (R    ) ps)
  (proc zero?                          (E T  ) (=CL zerop))

;The macro proposal:
;  (proc preprocess		       (     ) ps)
;  (proc add-keyword		       (     ) ps)
;  (proc remove-keyword		       (     ) ps)
;  (proc scheme-syntax-table	       (     ) ps)
  )

;;; Code manipulation utilities.

(eval-when (lisp:eval load lisp:compile)

;;; Prepare for future type syntax extension

(defun make-formal (name type) (if type (list name type) name))
(defun formal-name (formal) (if (consp formal) (car formal) formal))
(defun formal-type (formal) (if (consp formal) (cadr formal) nil))

(defun named? (formal) formal)  ;?

(defun rest? (formals)
  (and (not (null formals))
       (or (not (consp formals))
           (eq (car formals) '&rest)
           (eq (car formals) '&body))))

(defun rest-formal (formals)
  (if (consp formals) (cadr formals) formals))

(defun convert-formals (formals mode &optional (optional-or-key nil))
  (cond ((rest? formals)
         (lisp:let* ((p (rest-formal formals))
		     (name (if (named? p)
			       (formal-name p)
			       'ignore)))
           (ecase mode
             ((apply)
              (list (if (named? p) name ''())))
             ((no-rest)
              '())
             ((scheme-lambda)
              name)
             ((defmacro)   ;very effective hack, for proper indentation
              `(,(if (string= (symbol-name name) "BODY") '&body '&rest) ,name))
             ((cl-lambda)
              `(&rest ,name)))))
        ((null formals)
         (ecase mode
           ((apply) '('()))
           ((no-rest scheme-lambda defmacro cl-lambda) '())))
        (t
         (lisp:let ((formal (car formals)))
           (cond ((lisp:member formal '(&optional &key))
                  (ecase mode
                    ((defmacro cl-lambda) formals)
                    ((apply no-rest scheme-lambda)
                     ;; Gross approximation!!!
                     (convert-formals (cdr formals) mode formal))))
                 (t
                  (cons (formal-name formal)
                        (convert-formals (cdr formals) mode optional-or-key))))))))

(defun flatten-formals (formals)
  (cond ((null formals) '())
	((rest? formals) (list (rest-formal formals)))
	(t (cons (car formals) (flatten-formals (cdr formals))))))

(defun lambda-expression? (e)
  (and (consp e) (eq (car e) '.lambda)))

(defun to-lambda-expression (e)
  (if (and (consp e) (eq (car e) 'lambda))
      (macroexpand-1 e)
      e))

(defun definition? (e)
  (and (consp e) (eq (car e) 'define)))

(defun undefinify (body pass)
  (lisp:let ((?lambda (ecase pass ((1) 'lambda) ((2) '.lambda)))
	     (?letrec (ecase pass ((1) 'letrec) ((2) '.letrec))))
    (lisp:do ((e body (cdr e))
	      (d '()
		 (lisp:let ((pat (cadr (car e)))
			    (bod (cddr (car e))))
		   (cons (if (consp pat)
			     `(,(car pat)
			       (,?lambda ,(cdr pat) ,@bod))
			     (cdr (car e)))
			 d))))
	     ((or (null (cdr e))
		  (not (definition? (car e))))
	      (if (null d)
		  body
		  `((,?letrec ,(reverse d) ,@e)))))))

(defun beginify (forms pass)
  (lisp:let ((?begin (ecase pass ((1) 'begin) ((2) '.begin))))
    (if (null (cdr forms)) (car forms) `(,?begin ,@forms))))

(defun *---*? (var)
  (lisp:let ((s (symbol-name var)))
    (and (>= (length s) 3)
	 (char= (aref s 0) #\*)
	 (char= (aref s (- (length s) 1)) #\*))))

(defun maybe-preprocess (e)
  (if *preprocess?* (preprocess-top e) e))

);nehw-lave

;;; Pass 2 of Scheme->CL translation is accomplished by a bunch of CL
;;; macros.

;;;  (define-pseudo-macro (foo formals) body)
;;;    ==>  (progn (defmacro foo (&whole form)
;;; 		     (if *preprocess?*
;;; 		         (preprocess-top form)
;;; 		         `(.foo ,@(cdr form))))
;;; 	           (defmacro .foo (formals)
;;; 		     body))

(defmacro define-pseudo-macro (pat &body body)
  (lisp:let* ((name (car pat))
	      (aux-name (lisp:let ((*package* (find-package 'pseudoscheme)))
			  (concatenate-symbol "." name)))
	      (new-formals (convert-formals (cdr pat) 'defmacro)))
    `(progn 'lisp:compile
            (defmacro ,aux-name ,new-formals
	      ,@body)
	    (export ',aux-name)       ;For prettier output
	    ;; We could use &whole, but the following makes
	    ;; control-shift-A work better.
            (defmacro ,name ,new-formals
	      (lisp:let ((tail (list* ,@(convert-formals (cdr pat) 'apply))))
		(if *preprocess?*
		    (preprocess-top `(,',name ,@tail))
		    `(,',aux-name ,@tail)))))))

;;; First, the trivial ones.

(defmacro .and (&rest conjuncts)
  `(and ,@conjuncts))

(export '(.and))
(export '(.or))
(export '(.if))
(export '(.cond))

;;; Kludge to allow (begin (define-macro (foo ...) ...) ... (foo ...) ...).

(defmacro begin (first-form &rest other-forms)
  (if (null other-forms)
      (preprocess-top first-form)
      `(.begin ,(preprocess-top first-form)
	       (begin ,@other-forms))))

(defmacro .begin (&rest forms)
  `(progn 'lisp:compile ,@forms))

(export '.begin)


(defmacro .call (&rest args)
  `(funcall ,@args))
(defmacro .call[tail] (&rest args)
  `(funcall ,@args))
(export '(.call .call[tail]))

(define-pseudo-macro (case test-object . body)
  `(lisp:case ,test-object
     ,@(mapcar #'(lisp:lambda (clause)
                   (if (eq (car clause) 'else)
                       `(otherwise ,@(cdr clause))
                       clause))
               body)))

(defmacro .cond (&rest clauses)   ;Not shadowed, yet.
  `(cond ,@clauses))

(define-pseudo-macro (cons-stream hd tl)
  `(cons ,hd (delay ,tl)))

(define-pseudo-macro (delay e)
  `(make-delay :thunk-or-value #'(lisp:lambda () ,e)))

(defmacro do (specs end &body body)
  (lisp:let ((loop (gentemp "DO-")))
    `(letrec ((,loop
	       (lambda ,(mapcar #'car specs)
		 (cond ,end
		       (else ,@body
			     (,loop
			      ,@(mapcar #'(lisp:lambda (y)
					    (if (null (cddr y))
						(car y)
						(caddr y)))
					specs)))))))
       (,loop ,@(mapcar #'cadr specs)))))
(export '(do))

(defmacro .if (test consequent &optional alternate)  ;Not shadowed, yet.
  `(if ,test ,consequent ,alternate))

(defmacro .or (&rest disjuncts)
  `(or ,@disjuncts))

(define-pseudo-macro (sequence . forms)
  `(.begin ,@forms))

;;;; Binding forms: LAMBDA, LET, LET*, SET!

;;; .FUNCTIONAL is used by pass-1 output.
;;; [The name was suggested by Mike Beckerle.]

(defmacro .functional (formals &body body)
  (if (null formals)
      (beginify body 2)
      `(macrolet ,(mapcar #'(lisp:lambda (formal)
			      (lisp:let ((var (formal-name formal)))
				`(,var (&rest args)
				  `(.call ,',var ,@args))))
			  formals)
	 ,@body)))
;(export '(.functional))  -- bad idea

;;; We can't just define a LISP:LAMBDA macro, since the host Common Lisp
;;; might define one of its own that shouldn't be clobberred.  (This is
;;; the case on the 3600.)

(define-pseudo-macro (lambda formals . body)
  `#'(lisp:lambda ,(convert-formals formals 'cl-lambda)
       (.lambda-body ,formals ,@body)))

;;; Needed by top-level DEFINE; makes bodies of lambda-expressions
;;; prettier for GRINDEF etc.

(defmacro .lambda-body (formals &body body)
  #-Lispm
  (declare (ignore formals))
  (lisp:let* ((body (undefinify body 2))
	      #+Lispm
	      (rest-var (if (atom formals) formals (cdr (last formals)))))
    #-Lispm
    (beginify body 2)
    #+Lispm   ;Lisp Machines suck
    (if rest-var
        `(lisp:let ((,rest-var (copy-list ,rest-var)))
            ,@body)
        (beginify body 2))))
;(export '(.lambda-body)) -- bad idea

(define-pseudo-macro (let specs . body)
  (cond ((or (null specs) (consp specs))
	 ;; Normal case
	 `(lisp:let ,(mapcar #'(lisp:lambda (spec)
			         (list (formal-name (car spec)) (cadr spec)))
			     specs)
	    ,@(undefinify body 2)))
	(t
	 ;; Named LET
	 (lisp:let ((tag specs)
		    (specs (car body))
		    (body (cdr body)))
	   `(,(if *preprocess?* '.letrec '.letrec[tagbody])
	     ;; If preprocessor is turned off, be optimistic.
	     ((,tag (.lambda ,(mapcar #'car specs)
		      ,@body)))
	     (,tag ,@(mapcar #'cadr specs)))))))

(define-pseudo-macro (let* specs . body)
  (cond ((null specs) (beginify (undefinify body 2) 2))
	(t `(.let (,(car specs))
	      (.let* ,(cdr specs) ,@body)))))

;;; Note that SET! on lambda-bound variables won't work in
;;; unpreprocessed programs.

(define-pseudo-macro (set! var val)
  (lisp:let ((method (get-method var)))
    (cond ((or (null method) (eq method :variable))
	   ;; Be optimistic.
	   (record-variable-usage var :setq)
	   `(setq ,var ,val))
	  (t
	   (when (eq method :macro)
	     (cerror "proceed as if everything was OK"
		     "(DEFINE ~S ...) was implemented as (DEFMACRO ~S ...),~
		    ~%but (SET! ~S ...) was encountered"
		     var var var))
	   (record-variable-usage var :set-globally)
	   `(set-global-value ',var ,val)))))

(defun set-global-value (var val)
  (record-variable-usage var :set-globally)
  (setf (symbol-value var) val)
  (if (procedure? val)
      (setf (symbol-function var) val)
      (setf (symbol-function var)
	    #'(lisp:lambda (&rest args)
		(apply var args))))
  t)

;;;; LETREC

;;; There are three kinds of LETREC.  In the absence of further
;;; information we assume the most general (least "efficient").

;;; 1. Most general case

(define-pseudo-macro (letrec specs . body)
  `(.let ,(mapcar #'(lisp:lambda (spec) `(,(car spec) ':undefined))
		  specs)
     (.functional ,(mapcar #'car specs)
        ,@(mapcar #'(lisp:lambda (spec)
		      `(setq ,(formal-name (car spec)) ,(cadr spec)))
		  specs)
        ,@(undefinify body 2))))

;;; 2. Somewhat less general; SET! and non-functional references not
;;; allowed.

(defmacro .letrec[labels] (specs &body body)
  `(labels ,(mapcar #'(lisp:lambda (spec)
			(lisp:let ((var (formal-name (car spec)))
				   (val (to-lambda-expression (cadr spec))))
			  (cond ((lambda-expression? val)
				 `(,var ,(convert-formals (cadr val) 'cl-lambda)
				   (.lambda-body ,@(cdr val))))
				(t
				 (cerror "ignore it"
					 "losing LETREC spec - ~s"
					 spec)
				 `(,var (&rest foo) foo)))))
		    specs)
     ,@body))

;;; 3. This is the interesting case: the LETREC is really a loop.  This
;;; only works if all calls to all the procedures are at the same
;;; tail-recursion level.

(defmacro .letrec[tagbody] (specs &body body)
  #-TAIL-RECURSION-WINS
  (lisp:let* ((number-of-registers
		(apply #'max (mapcar #'(lisp:lambda (spec)
					 (length (cadr (cadr spec))))
				     specs)))
	      (registers
		(lisp:do ((i 0 (lisp:+ i 1))
			  (regs '() (cons (gentemp "REG-") regs)))
			 ((= i number-of-registers) (reverse regs))))
	      (block-name (gentemp "LETREC-")))
    `(lisp:block ,block-name
       (lisp:let ,registers
	 (macrolet ,(mapcar #'(lisp:lambda (spec)
				(lisp:let ((tag (formal-name (car spec)))
					   (formals (cadr (cadr spec))))
				  `(,tag ,formals
				    `(progn (setq ,@(list ,@(mapcan #'(lisp:lambda (reg var)
									`(',reg ,var))
								    registers
								    formals)))
					    (go ,',tag)))))
			    specs)
	   (tagbody
	     (return-from ,block-name ,(beginify (undefinify body 2) 2))
	     ,@(mapcan #'(lisp:lambda (spec)
			   (list (formal-name (car spec))
				 `(return-from ,block-name
				    (lisp:let ,(mapcar #'list (cadr (cadr spec)) registers)
				      ,@(undefinify (cddr (cadr spec)) 2)))))
		       specs))))))
  #+TAIL-RECURSION-WINS
  `(.letrec[labels] ,specs ,@body))

(export '(.letrec[labels] .letrec[tagbody]))

;;;; Definers

(define-pseudo-macro (define pat . body)
  (if (consp pat)
      `(.define ,(car pat) (.lambda ,(cdr pat) ,@body))
      (lisp:let* ((var pat)
		  (val (to-lambda-expression (car body)))
		  (h (get var 'history)))
	(cond ((eq (get-method var) :syntactic-keyword)
	       `(progn 'lisp:compile
		       (defvar ,var)
		       (setq ,var ,val)
		       ',var))
	      ((or (lisp:member :setq h)
		   (*---*? var))
	       `(define[variable] ,var ,val))
	      ((lambda-expression? val)
	       `(define[function] ,var ,val))
	      (t
	       `(define[variable] ,var ,val))))))

(defmacro define[function] (var (-ignore- formals . body))
  #-Lispm (declare (ignore -ignore-))  #+Lispm -ignore-
  (use-method var :function)
  `(progn 'lisp:compile
	  (use-method ',var :function)
	  ;; Must use DEFVAR to inhibit "not declared special" warnings.
	  (defvar ,var)
	  (defun ,var ,(convert-formals formals 'cl-lambda)
	    (.lambda-body ,formals ,@body))
	  (setq ,var #',var)
	  ',var))

(defmacro define[variable] (var val)
  (use-method var :variable)
  `(progn 'lisp:compile
	  (use-method ',var :variable)
	  (defvar ,var)
	  (setq ,var ,val)
	  (defun ,var (&rest args)
	    (apply ,var args))))

(defmacro define[constant] (var val)
  (use-method var :constant)
  `(progn 'lisp:compile
	  (use-method ',var :constant)
	  (defconstant ,var ,val)))

(defmacro define[subst] (pat &body body)
  (if (consp pat)
      `(define[subst] ,(car pat) (.lambda ,(cdr pat) ,@body))
      (lisp:let ((var pat)
		 (val (to-lambda-expression (car body))))
	(use-method var :macro)
	(if (lambda-expression? val)
	    (lisp:let* ((formals (cadr val))
			(formal-names (mapcar #'formal-name (flatten-formals formals)))
			(body-exp (if (null (cdddr val))
				      (caddr val)
				      `(progn ,@(cddr val))))
			(aux-var (concatenate-symbol var '[closed])))
	      `(progn 'lisp:compile
		      (use-method ',var :macro)
		      (defmacro ,var ,(convert-formals formals 'defmacro)
			,(if (null formal-names)
			     `',body-exp
			     `(sublis (mapcar #'cons
					      ',formal-names
					      (list ,@formal-names))
				      ',body-exp)))
		      (defvar ,var)
		      (defun ,aux-var ,(convert-formals formals 'cl-lambda)
			(.lambda-body ,formals ,@(cddr val)))
		      (setq ,var #',aux-var)
		      ',var))
	    `(progn 'lisp:compile
		    (use-method ',var :macro)
		    (defmacro ,var (&rest args)
		      `(,',val ,@args))
		    (defparameter ,var ,val))))))

;;; Nonstandard

(define-pseudo-macro (define-macro pat . body)
  (lisp:let* ((name (car pat))
	      (formals (cdr pat))
	      (aux-name (concatenate-symbol '.expand- name)))
    `(progn 'lisp:compile
	    (defmacro ,name ,(convert-formals formals 'defmacro)
	      (preprocess-top (list* ',name ,@(convert-formals formals 'apply))))
	    (defmacro ,aux-name ,(convert-formals formals 'defmacro)
	      ,@body)
	    (eval-when (lisp:eval lisp:load lisp:compile)
	      (add-preprocessor!
	       *syntax-table* ',name
	       (.lambda (e env k)
		 (preprocess-internal (macroexpand-1 (cons ',aux-name (cdr e)))
				      env k))))
	    ',name)))

;;;; QUASIQUOTE

(define-pseudo-macro (quasiquote x)
  (expand-quasiquote x 0))

(define-pseudo-macro (unquote x)
  (cerror "act as if the comma wasn't there at all"
	  "comma not inside backquote form - ,~S" x)
  x)

(define-pseudo-macro (unquote-splicing x)
  (cerror "act as if the ,@ wasn't there at all"
	  "\",@\" not inside backquote form - ,@~S" x)
  x)

(define (expand-quasiquote x level)
  (multiple-value-bind (mode arg)
      (descend-quasiquote x level)
    (finalize-quasiquote mode arg)))

(define (finalize-quasiquote mode arg)
  (cond ((eq mode 'quote) `',arg)
	((eq mode 'unquote) arg)
	((eq mode 'unquote-splicing)
	 (cerror "act as if () had been seen instead of ,@<form>"
		 ",@ in illegal context - ,@~s"
		 arg))
	(else (cons mode arg))))

;;; The two return values, mode and arg, are interpreted as follows:
;;;    mode    arg          meaning
;;;    QUOTE   x            'x
;;;    UNQUOTE x            x
;;;    LIST    (x1 x2 ...)  (LIST x1 x2 ...)
;;;    CONS*   (x1 x2 ...)  (CONS* x1 x2 ...)
;;;    APPEND  (x1 x2 ...)  (APPEND x1 x2 ...)

(define (descend-quasiquote x level)
  (cond ((vector? x)
	 (descend-quasiquote-vector x level))
	((atom x)
	 (values 'quote x))
	((interesting-to-quasiquote? x 'quasiquote)
	 (descend-quasiquote-pair x (1+ level)))
	((interesting-to-quasiquote? x 'unquote)
	 (cond ((= level 0)
		(values 'unquote (cadr x)))
	       (else
		;; BUG: ,,@ doesn't work.  I think this is the spot
		;; where it would have to be hacked in.
		(descend-quasiquote-pair x (- level 1)))))
	((interesting-to-quasiquote? x 'unquote-splicing)
	 (cond ((= level 0)
		(values 'unquote-splicing (cadr x)))
	       (else
		(descend-quasiquote-pair x (- level 1)))))
        (else
	 (descend-quasiquote-pair x level))))

(define (descend-quasiquote-pair x level)
  (multiple-value-bind (car-mode car-arg)
      (descend-quasiquote (car x) level)
    (multiple-value-bind (cdr-mode cdr-arg)
	(descend-quasiquote (cdr x) level)
      (cond ((and (eq car-mode 'quote) (eq cdr-mode 'quote))
	     (values 'quote x))
	    ((eq car-mode 'unquote-splicing)
	     ;; (,@mumble ...)
	     (cond ((and (eq cdr-mode 'quote) (null cdr-arg))
		    (values 'unquote
			    car-arg))
		   ((eq cdr-mode 'append)
		    (values 'append
			    (cons car-arg cdr-arg)))
		   (else
		    (values 'append
			    (list car-arg (finalize-quasiquote cdr-mode cdr-arg))))))
	    ((and (eq cdr-mode 'quote) (null cdr-arg))
	     (values 'list
		     (list (finalize-quasiquote car-mode car-arg))))
	    ((or (eq cdr-mode 'list) (eq cdr-mode 'cons*))
	     (values cdr-mode
		     (cons (finalize-quasiquote car-mode car-arg)
			   cdr-arg)))
	    (else
	     (values 'cons*
		     (list (finalize-quasiquote car-mode car-arg)
			   (finalize-quasiquote cdr-mode cdr-arg))))))))

;;;   #(a b c)     ==>  '#(a b c)
;;;   #(a ,b c)    ==>  (vector 'a b 'c)
;;;   #(a ,@b ,c)  ==>  (list->vector (append '(a) b (list c)))
;;; Isn't there some elegant way to do this?
;;; YES.  Parameterize descend-quasiquote-pair.

(define (descend-quasiquote-vector x level)
  (let loop ((i (- (vector-length x) 1))
	     (modes '())
	     (args '())
	     (flag nil))
    (cond ((>= i 0)
	   (multiple-value-bind (mode arg)
	       (descend-quasiquote (vector-ref x i) level)
	     ;; mode = quote, unquote, unquote-splicing, list, cons*
	     (loop (- i 1)
		   (cons mode modes)
		   (cons arg args)
		   (or flag (not (eq mode 'quote))))))
	  ((not flag)
	   (values x 'quote))
	  ((lisp:member 'unquote-splicing modes)
	   (values 'list->vector
		   (list (cons 'append
			       (mapcar (lambda (mode arg)
					 (cond ((eq mode 'unquote-splicing) arg)
					       ;; Not good.
					       ((eq mode 'quote)
						(list 'quote (list arg)))
					       (else
						(list 'list
						      (finalize-quasiquote mode arg)))))
				       modes
				       args)))))
	  (else
	   (values 'vector
		   (mapcar #'finalize-quasiquote modes args))))))

(define (interesting-to-quasiquote? x marker)
  (and (consp x)
       (eq (car x) marker)
       (consp (cdr x))
       (null (cddr x))))

;;;; Procedures

(defmacro define-pseudo (pat &body body)
  `(define ,pat ,@body))

(defmacro define-pseudo[subst] (pat &body body)
  `(define[subst] ,pat ,@body))

(define-pseudo[subst] (assq obj list)
  (lisp:assoc obj list :test #'eq))

(define-pseudo[subst] (assoc obj list)
  (lisp:assoc obj list :test #'equal))

(define-pseudo[subst] (boolean? obj)
  (or (eq obj t) (eq obj nil)))

(define-pseudo[subst] (call-with-current-continuation proc)
  (block current-continuation
    (.call proc #'(lisp:lambda (val)
		    (return-from current-continuation val)))))

(define-pseudo[subst] (call-with-input-file string proc)
  (with-open-file (port string :direction :input)
    (.call proc port)))

(define-pseudo[subst] (call-with-output-file string proc)
  (with-open-file (port string :direction :output
			       :if-exists :new-version)
    (.call proc port)))

(define-pseudo (char-whitespace? char)
  (or (char= char #\space)
      (not (graphic-char-p char))))

(define-pseudo (compile symbol &optional lambda-exp)  ;kludge
  (let* ((e (macroexpand (preprocess-top lambda-exp)))
	 (lambda-exp (if (and (consp e) (eq (car e) 'function)) (cadr e) e)))
    (cond ((null symbol)
	   (lisp:compile nil lambda-exp))
	  (t
	   (prog1 (if lambda-exp
		      (lisp:compile symbol lambda-exp)	;clobbers function cell
		      (lisp:compile symbol))
		  (if (and symbol (symbolp symbol))
		      (setf (symbol-value symbol) (symbol-function symbol))))))))

(define-pseudo[subst] (current-input-port)
  *standard-input*)

(define-pseudo[subst] (current-output-port)
  *standard-output*)

(define-pseudo[subst] (eof-object? obj)
  (eq obj 'eof-object))

(define-pseudo (error . items)
  (apply #'cerror
	 "Return from the call to the ERROR procedure"
	 (apply #'concatenate
		'string
		"~a"
		(mapcar #'(lisp:lambda (item)
			    #-Symbolics (declare (ignore item))
			    #+Symbolics item
			    "~%~s")
			(cdr items)))
	 items))

#+LispM
(setf (get 'error :error-reporter) t)  ;Thanks to KMP

(define-pseudo (eval obj env)
  env
  (lisp:eval (preprocess-top obj)))

(defstruct (delay (:print-function print-delay) (:predicate delay?))
  (forced-yet? nil)
  thunk-or-value)

(defun print-delay (obj stream escape?)
  #-Symbolics (declare (ignore escape?))
  #+Symbolics escape?
  (if (delay-forced-yet? obj)
      (format stream "#{Forced ~S}" (delay-thunk-or-value obj))
      (format stream "#{Delayed}")))

(define-pseudo (display obj &optional (port *standard-output*))
  (lisp:let ((*standard-output* port))
    (really-display obj)))

;;; Temporary kludge...

(define (really-display obj)
  (cond ((null? obj) (princ "()"))
	((eq? obj t) (princ "#T"))
	(else (princ obj))))


(define-pseudo (force obj)
  (cond ((delay? obj)
         (lisp:let ((tv (delay-thunk-or-value obj)))
           (cond ((delay-forced-yet? obj) tv)
                 (t (lisp:let ((val (funcall tv)))
                      (setf (delay-thunk-or-value obj) val)
                      (setf (delay-forced-yet? obj) t)
                      val)))))
        (t obj)))

(define-pseudo (input-port? obj)
  (and (streamp obj) (input-stream-p obj)))

(defvar the-empty-string "")

(define-pseudo (list->string l)
  (if (null l)
      the-empty-string
      (coerce l 'string)))

(defvar the-empty-vector '#())

(define-pseudo (list->vector l)
  (if (null l)
      the-empty-vector
      (coerce l 'vector)))

(define-pseudo[subst] (list-ref list n)
  (nth n list))

(define-pseudo[subst] (list-tail list n)
  (nthcdr n list))

(define-pseudo[subst] (make-polar r th)
  (* r (cis th)))

(define-pseudo (make-string size
                            &optional fill)
  (cond ((= size 0) the-empty-string)
        (fill (lisp:make-string size :initial-element fill))
        (t (lisp:make-string size))))

(define-pseudo (make-vector size
                            &optional
                            (fill '*uninitialized-vector-element*))
  (if (= size 0)
      the-empty-vector
      (make-sequence 'vector size :initial-element fill)))

(define-pseudo[subst] (memq obj list)
  (lisp:member obj list :test #'eq))

(define-pseudo[subst] (member obj list)
  (lisp:member obj list :test #'equal))

(define-pseudo (number->string num format)
  (if (not (and (consp format) (eq? (car format) 'heur)))
      (cerror "act as if the format was (HEUR)"
              "unimplemented format: (NUMBER->STRING '~s '~s)"
              num format))
  (write-to-string num))

(define-pseudo[subst] (open-input-file string)
  (open string :direction :input))

(define-pseudo[subst] (open-output-file string)
  (open string :direction :output))

(define-pseudo (output-port? obj)
  (and (streamp obj) (output-stream-p obj)))

(define-pseudo (pp obj &optional (port *standard-input*))
  (lisp:let ((*print-pretty* t))
    (format port "~&")
    (lisp:print obj port)
    (values)))

(defparameter closures-might-be-conses?
	      (or (consp (lisp:eval '#'(lisp:lambda (x) x)))
		  (consp (lisp:compile nil '(lisp:lambda (x) x)))
		  (consp (let ((g (gensym)))
			   (lisp:eval `(progn (defun ,g () 0) #',g))))))

(define-pseudo (procedure? obj)
  (and (functionp obj)
       (not (symbolp obj))
       (or (not (consp obj))
	   closures-might-be-conses?)))

(define-pseudo[subst] (quotient n1 n2)
  (values (truncate n1 n2)))

(define-pseudo (read &optional (port *standard-input*))
  (lisp:read port nil 'eof-object))

(define-pseudo (read-char &optional (port *standard-input*))
  (lisp:read-char port nil 'eof-object))

(define-pseudo (real? obj)
  (and (numberp obj) (not (complexp obj))))

(define-pseudo[subst] (set-car! pair obj)
  (setf (car pair) obj))

(define-pseudo[subst] (set-cdr! pair obj)
  (setf (cdr pair) obj))

(define-pseudo[subst] (string->list string)
  (coerce string 'list))

(define-pseudo (string->number string exactness radix)
  exactness
  (let ((*read-base* (case radix
		       ((b) 2)
		       ((o) 8)
		       ((d) 10)
		       ((x) 16)
		       (else (error "bad radix arg to STRING->NUMBER"
				    radix)))))
    (with-input-from-string (s string)
      (read s))))  ;very crude approximation

(define-pseudo[subst] (string->symbol string)
  (values (intern string)))

(define-pseudo (string-append . strings)
  (apply #'concatenate 'string strings))

(define-pseudo[subst] (string-set! s k obj)
  (setf (aref (the simple-string s) k) obj))

(define-pseudo (symbol? x)
  (and x (symbolp x) (not (eq x t))))

(define-pseudo[subst] (tail stream)
  (force (cdr stream)))

(define-pseudo the-empty-stream '())

(define-pseudo user-initial-environment nil)

(define-pseudo (vector? obj)
  (and (simple-vector-p obj)
       ;; Strings are simple vectors in CLISP (this is a bug)
       #+tops-20
       (not (stringp obj))
       ;; Structures are vectors in CLISP, bolixlisp, etc.
       #+(or tops-20 Lispm vax)
       (not (typep obj 'structure))))

(define-pseudo[subst] (vector->list vec)
  (coerce vec 'list))

(define-pseudo[subst] (vector-set! vec k obj)
  (setf (svref vec k) obj))

(define-pseudo[subst] (with-input-from-file string thunk)
  (with-open-file (*standard-input* string :direction :input)
    (.call thunk)))

(define-pseudo[subst] (with-output-to-file string thunk)
  (with-open-file (*standard-output* string :direction :output
					    :if-exists :new-version)
    (.call thunk)))

(define-pseudo (write obj &optional (port *standard-output*))
  (lisp:let ((*standard-output* port))
    (really-write obj)))

;;; Temporary kludge...

(define (really-write obj)
  (cond ((null? obj) (princ "()"))
	((eq? obj t) (princ "#T"))
	(else (prin1 obj))))


;;;; The preprocessor

;;; Pass 1: Scheme -> readable-CL         [preprocessor]
;;; Pass 2: readable-CL -> executable-CL  [macros]

;;; Preprocessor to allow for combinations whose cars aren't symbols.

;;; There's the potential for doing a lot of translations and
;;; optimizations in a Scheme->CL preprocessor, but the idea is that we
;;; want to do as little translation as possible in order that
;;; pretty-printing and debugging can work as smoothly as possible.  Too
;;; much translation was the downfall of the previous Scheme-in-CL
;;; implementation: programs were impossible to debug.

;;; Be careful -- the output of the preprocessor is Common Lisp, not
;;; Scheme.  Don't call the preprocessor twice on a given piece of
;;; code.

;;;  - Global SET!'s must clobber the function cell, but local ones don't.
;;;  - Annotate binding constructs to say when MACROLET's are needed for
;;;    variables occurring in function position.
;;;  - Detect absence of SET! and do some rudimentary tail recursion
;;;    analysis for optimization of LETREC.

;;; The rudiments of global tail recursion processing are here (.CALL vs.
;;; .CALL[TAIL]) but don't imagine that tail recursion can be easily added.
;;; Two reasons:
;;;  (1) .CALL[TAIL] isn't inserted in all the places it needs to be, and
;;;  (2) it might be inserted in some places where it shouldn't be.
;;;	 I think there might be a bad interaction with the analysis for
;;;	 LETREC.

;;; Bindings
;;; For each bound variable, we want to know:
;;;  Does it occur in value position?
;;;  Does it occur in function position?
;;;    If so, does it appear with a consistent continuation?
;;;  Does it occur in a SET! ?
;;; ENV is a list of "binding" structures, one per variable, which keep
;;; track of this information.

(defstruct (binding
	      (:conc-name lisp:nil)
	      (:predicate lisp:nil)
	      (:copier lisp:nil)
	      (:constructor make-binding (old-name type new-name)))
  old-name
  type
  new-name
  (value-references? nil)
  (function-references? nil)
  (assigned? nil)
  (continuation-when-called nil))

(define[subst] (bind-var formal new-formal env)
  (cons (make-binding (formal-name formal)
		      (formal-type formal)
		      (formal-name new-formal))
	env))

(define (lookup name env)
  (locally (declare (optimize (speed 3) (safety 0)))
    (find name env :key #'old-name)))

(define[subst] (set-value-references?! binding)
  (setf (value-references? binding) t))

(define[subst] (set-function-references?! binding)
  (setf (function-references? binding) t))

(define[subst] (set-assigned?! binding)
  (setf (assigned? binding) t))

(define (set-continuation-when-called! binding want-k)
  (let ((have-k (continuation-when-called binding)))
    (cond ((null? have-k)
	   (setf (continuation-when-called binding) want-k))
	  ((not (eq? have-k want-k))
	   (setf (continuation-when-called binding) 'inconsistent)))))

;;; Syntax tables

(defstruct (syntax-table
#-TI-REL2    (:constructor make-syntax-table (parent))	;Arglist bug
	     (:predicate syntax-table?)
	     (:copier lisp:nil))
  a-list
  parent)

(define (get-preprocessor table sym)
  (and table
       (let ((probe (assq sym (syntax-table-a-list table))))
	 (if probe
	     (cdr probe)
	     (get-preprocessor (syntax-table-parent table) sym)))))

(define (add-preprocessor! table sym proc)
  (let ((a (syntax-table-a-list table)))
    (let ((probe (assq sym a)))
      (if probe
	  (set-cdr! probe proc)
	  (setf (syntax-table-a-list table) (cons (cons sym proc) a))))))

(defmacro define-preprocessor ((noise1 name) (noise2 formals . body))
  (declare (ignore noise1 noise2)) #+Symbolics (progn noise1 noise2)
  (lisp:let ((aux-name (concatenate-symbol 'preprocess/ name)))
    `(progn 'compile
	    (defun ,aux-name ,formals ,@body)
	    (add-preprocessor! scheme-syntax-table ',name #',aux-name)
	    ',name)))

(defparameter scheme-syntax-table	;Clobberred by DEFINE-PREPROCESSOR
  (make-syntax-table #-TI-REL2 nil))    ;Arglist bug


(defparameter *syntax-table*			;Something for the user to clobber
  #-TI-REL2 (make-syntax-table scheme-syntax-table)
  #+TI-REL2 (make-syntax-table :parent scheme-syntax-table)
  )

;;; The preprocessor

(define (preprocess-top e)
  (preprocess-internal e '() '(top)))

(define (preprocess-internal e env k)
  (cond ((symbol? e)
	 (let ((probe (lookup e env)))
	   (cond ((not probe)
		  (record-variable-usage e :variable)
		  e)    ;Free (i.e. global) variable
		 (t
		  (set-value-references?! probe)
		  (new-name probe)))))
	((not (pair? e)) e)
	((not (symbol? (car e)))
	 ;; Ought to pass continuation through to lambda-combinations...
	 `(,(if (eq? (car k) 'return) '.call[tail] '.call)
	   ,@(preprocess-list e env)))
	(else
	 (let ((probe (get-preprocessor *syntax-table* (car e))))
	   (if probe
	       (funcall probe e env k)
	       (preprocess-combination e env k))))))

(define (preprocess-combination e env k)
  (let* ((var (car e))
	 (probe (lookup var env)))
    (cond (probe
	   (set-function-references?! probe)
	   (set-continuation-when-called! probe k)
	   `(,(new-name probe) ,@(preprocess-list (cdr e) env)))
	  (t
	   (let ((meth (get-method var)))
	     (cond ((and (not meth)
			 (macro-function var))
		    ;; Quietly allow user to use Common Lisp macros.
		    (preprocess-internal (macroexpand-1 e) env k))
		   (t
		    (let ((args (preprocess-list (cdr e) env)))
		      (cond ((or (memq meth '(:variable :constant))
				 (memq :setq (get var 'history)))
			     (record-variable-usage var :variable)
			     `(,(if (eq? (car k) 'return) '.call[tail] '.call)
			       ,var ,@args))
			    (t
			     (cond ((eq? meth :macro)
				    (record-variable-usage var :macro))
				   ((and (not meth)
					 (special-form-p var))
				    (format *error-output*
					    "~&Warning!  A Lisp special form, ~S,~
~%  was encountered inside of a Scheme expression.  The Scheme preprocessor will~
~%  treat it as if it were a combination.~%"
					    e))
				   (t
				    (record-variable-usage var :function)))
			     `(,var ,@args)))))))))))

;;; Kludges to implement JAR's new macro proposal.

(define (preprocessed? e)
  (and (pair? e) (eq? (car e) '.preprocessed)))

(define (preprocess e st)
  (if (preprocessed? e)
      e
      `(.preprocessed ,e ,st)))

(define-preprocessor '.preprocessed
  (lambda (e env k)
    (let ((*syntax-table* (caddr e)))
      (preprocess-internal (cadr e) env k))))

(defmacro .preprocessed (e st)	   ;In case one somehow escapes to Lisp's EVAL
  (lisp:let ((*syntax-table* st))  ; (shouldn't happen, but...)
    (preprocess-internal e '() '(top))))

(define (add-keyword st symbol proc)
  (let ((new-st (make-syntax-table st)))
    (add-preprocessor! new-st symbol
		       (lambda (e env k)
			 (preprocess-internal (funcall proc e *syntax-table*) env k)))
    new-st))

(define (remove-keyword st symbol)
  (let ((new-st (make-syntax-table st)))
    (add-preprocessor! new-st symbol nil)
    new-st))  

;;; Special forms are in alphabetical order

(define-preprocessor 'and
  (lambda (e env k)
    `(.and ,@(if (null? (cdr e))
		 '()
		 (preprocess-sequence (cdr e) env k)))))

(define-preprocessor 'begin
  (lambda (e env k)
    (if (eq (car k) 'top)
	`(.begin ,@(map (lambda (e) (preprocess-internal e env '(top)))
			(cdr e)))
	`(.begin ,@(preprocess-sequence (cdr e) env k)))))

(define-preprocessor 'case
  (lambda (e env k)
    `(.case ,(preprocess-internal (cadr e) env '(value))
       ,@(map (lambda (clause)
		`(,(car clause) ,@(preprocess-sequence (cdr clause) env k)))
	      (cddr e)))))

(define-preprocessor 'cond
  (lambda (e env k)
    `(.cond ,@(map (lambda (clause)
		     (cons (preprocess-internal (car clause) env '(value))
			   (if (null? (cdr clause))
			       '()
			       (preprocess-sequence (cdr clause) env k))))
		   (cdr e)))))

;;; Nonstandard
(define-preprocessor 'cons-stream
  (lambda (e env k) k
    `(.cons-stream ,@(preprocess-list (cdr e) env))))

(define-preprocessor 'define
  (lambda (e env k)
    (if (not (eq (car k) 'top))
	(cerror "proceed as if it were at top level"
		"definition occurs in illegal context -- ~s"
		e))
    (let ((pat (cadr e))
	  (body (cddr e)))
      (if (pair? pat)
	  (preprocess-internal `(define ,(car pat)
				  (lambda ,(cdr pat) ,@body))
			       env k)
	  `(.define ,pat ,(preprocess-internal (car body) env '(value)))))))

;;; Nonstandard
(define-preprocessor 'define-macro
  (lambda (e env k)
    (if (not (eq (car k) 'top))
	(cerror "proceed as if it were at top level"
		"macro definition occurs in illegal context -- ~s"
		e))
    (let ((pat (cadr e))
	  (body (cddr e)))
      (let ((kludge (preprocess-internal `(lambda ,(cdr pat) ,@body) env '(value))))
	`(.define-macro (,(car pat) ,@(cadr kludge)) ,@(cddr kludge))))))

(define-preprocessor 'delay
  (lambda (e env k) k
    `(.delay ,@(preprocess-list (cdr e) env))))

(define-preprocessor 'do
  (lambda (e env k)
    (preprocess-internal (macroexpand-1 e) env k)))

(define-preprocessor 'if
  (lambda (e env k)
    `(.if ,(preprocess-internal (cadr e) env '(value))
	  ,@(map (lambda (e) (preprocess-internal e env k))
		 (cddr e)))))

(define-preprocessor 'lambda
  (lambda (e env k)
    (let* ((formals (flatten-formals (cadr e)))
	   (new-formals (renamed-formals formals))
	   (env (bind-vars formals new-formals env))
	   (body (preprocess-body (cddr e) formals env
				  (if (eq? (car k) 'letrec)
				      (cadr k)
				      '(return)))))
      `(.lambda ,(renamed-formals (cadr e))
	 ,@body))))

(define-preprocessor 'let
  (lambda (e env k)
    (if (symbol? (cadr e))
	(let ((tag (cadr e))
	      (bindings (caddr e)))
	  (preprocess-internal `(letrec ((,tag (lambda ,(map car bindings)
						 ,@(cdddr e))))
				  (,tag ,@(map cadr bindings)))
		      env k))
	(let* ((formals (map car (cadr e)))
	       (new-formals (renamed-formals formals))
	       (bindings (preprocess-bindings (cadr e) new-formals env))
	       (env (bind-vars formals new-formals env))
	       (body (preprocess-body (cddr e) formals env k)))
	  `(.let ,bindings ,@body)))))

(define (preprocess-bindings bindings new-vars env)
  (map (lambda (spec new-var)
	 `(,new-var ,(preprocess-internal (cadr spec) env '(value))))
       bindings
       new-vars))

(define-preprocessor 'let*
  (lambda (e env k)
    (let loop ((old (cadr e))
	       (new '())
	       (formals '())
	       (env env))
      (if (null? old)
	  `(.let* ,(reverse new)
	     ,@(preprocess-body (cddr e) formals env k))
	  (let* ((formal (caar old))
		 (new-formal (renamed-formal formal)))
	    (loop (cdr old)
		  (cons (cons new-formal
			      (preprocess-region (cdar old) formals env '(value)))
			new)
		  (cons formal formals)
		  (bind-var formal new-formal env)))))))

;;; For an input expression (letrec ((f (arg) f-body)) body) we can generate
;;; any of three possible outputs.  In order of decreasing generality and
;;; increasing efficiency:
;;; 
;;; 	 Pass-1 (preproc.)	  Pass-2 output (macroexpansion)
;;;         output
;;; 
;;;  1.  (.letrec ...)	          (let ((f :undefined))
;;; 				    (.functional (f)
;;; 				      (setq f #'(lambda ...))
;;; 				      ...))
;;; 
;;;  2.  (.letrec[labels] ...)    (labels ((f ...)) ...)
;;; 
;;;  3.  (.letrec[tagbody] ...)   (let ((reg))
;;;				    (tagbody ... f ... (setq reg ...) (go f) ...))

(define-preprocessor 'letrec
  (lambda (e env k)
    (let* ((formals (map car (cadr e)))
	   (new-formals (renamed-formals formals))
	   (env (bind-vars formals new-formals env))
	   (zs (map (lambda (formal) (lookup (formal-name formal) env)) formals))
	   (unique (list 'letrec-return))
	   (binding-cont  (list 'letrec unique))
	   (bindings (map (lambda (spec new-formal)
			    `(,new-formal
			      ,(preprocess-internal (cadr spec) env binding-cont)))
			  (cadr e)
			  new-formals))
	   (body (preprocess-sequence (undefinify (cddr e) 1) env unique))
	   (type
	     (cond ((null? bindings) '.let)
		   ((not (and (every (lambda (binding)
				       (and (lambda-expression? (cadr binding))
					    (proper-list? (cadr (cadr binding)))))
				     bindings)
			      (every (lambda (z)
				       ;; All refs are function refs.
				       (and (not (value-references? z))
					    (not (assigned? z))))
				     zs)))
		    '.letrec)
		   ((every (lambda (z)
			     (eq? (continuation-when-called z) unique))
			   zs)
		    (for-each (lambda (z)
				;; Permit nested loops
				;; (This doesn't work very well, actually)
				(set-continuation-when-called! z k))
			      zs)
		    '.letrec[tagbody])
		   (else
		    `.letrec[labels]))))
      `(,type ,bindings ,@body))))

(define (proper-list? thing)
  (or (null? thing)
      (and (pair? thing)
	   (null? (cdr (last-pair thing))))))

(define-preprocessor 'or
  (lambda (e env k)
    `(.or ,@(if (null? (cdr e))
		'()
		(preprocess-sequence (cdr e) env k)))))

(define-preprocessor 'quasiquote
  (lambda (e env k)
    (preprocess-internal (expand-quasiquote (cadr e) 0) env k)))

(define-preprocessor 'unquote
  (lambda (e env k)
    (cerror "act as if the comma wasn't there at all"
	    "comma not inside backquote form - ,~S" (cadr e))
    (preprocess-internal (cadr e) env k)))

(define-preprocessor 'unquote-splicing
  (lambda (e env k)
    (cerror "act as if the ,@ wasn't there at all"
	    "\",@\" not inside backquote form - ,@~S" (cadr e))
    (preprocess-internal (cadr e) env k)))

(define-preprocessor 'quote
  (lambda (e env k) env k
    e))

(define-preprocessor 'sequence
  (lambda (e env k)
    (preprocess-internal `(begin ,@(cdr e)) env k)))

(define-preprocessor 'set!
  (lambda (e env k) k
    (let* ((var (cadr e))
	   (val (preprocess-internal (caddr e) env e))
	   (probe (lookup var env)))
      (cond (probe
	     (set-assigned?! probe)		;assigned
	     `(setq ,(new-name probe) ,val))
	    (else
	     `(.set! ,var ,val))))))

;;; Auxiliary routines

(define (preprocess-sequence e-list env k)
  (cond ((null? (cdr e-list))
	 (list (preprocess-internal (car e-list) env k)))
	(else
	 ;; Continuation can't be IGNORE -- this routine is also used
	 ;; by AND, OR, and DO
	 (cons (preprocess-internal (car e-list) env '(value))
	       (preprocess-sequence (cdr e-list) env k)))))

(define (preprocess-list exp-list env)
  (map (lambda (e) (preprocess-internal e env '(value)))
       exp-list))

(define (preprocess-body body formals env k)
  (preprocess-region (undefinify body 1) formals env k))

;;; Allow function references to FORMALS within the sequence BODY.
;;; The output is never a longer list than the input.

(define (preprocess-region body formals env k)
  (let* ((body (preprocess-sequence body env k))  ;side-affects env
	 (formals (function-vars formals env)))
    (if formals
	`((.functional ,(convert-formals formals 'no-rest)
	    ,@body))
	body)))

(define (function-vars formals env)
  (do ((v formals (cdr v))
       (f '() ;; A VAX LISP bug prevents the use of LET here.
	      (lisp:let ((z (lookup (formal-name (car v)) env)))
		(if (function-references? z)
		    (cons (new-name z) f)
		    f))))
      ((null? v) f)))

(define (renamed-formals formals)		;deal with dotted
  (cond ((pair? formals)
	 (cons (renamed-formal (car formals))
	       (renamed-formals (cdr formals))))
	((rest? formals) (renamed-formal formals))
	(else formals)))

(defun renamed-formal (formal)
  (let ((name (formal-name formal)))
    (make-formal (or (get name 'lexical-name)		;Speed hacque
		     ;; (lisp:let ((*package* (symbol-package name))) ...)
		     ;;    - bad idea
		     (let ((lex (concatenate-symbol "." name)))
		       (setf (get name 'lexical-name) lex)
		       lex))
		 (formal-type formal))))

(setf (get '&optional 'lexical-name) '&optional)

(define (bind-vars formals new-formals env)
  (do ((formals formals (cdr formals))
       (new-formals new-formals (cdr new-formals))
       (env env (bind-var (car formals) (car new-formals) env)))
      ((null? formals) env)))

;;;; Reader syntax

;;; Quasiquote, #T, #F, #D, and ##.
;;; #I, #E, #S, #L, etc. are eschewed to avoid user deception.

(defvar scheme-readtable
  ;; This may not be needed for Rel2, but I don't have a system to
  ;; test against
  #+TI-REL2 *readtable*
  #-TI-REL2 (copy-readtable nil)
  )

#+Symbolics
(pushnew scheme-readtable si:*valid-readtables*)

(defun init-scheme-readtable ()
  (lisp:let ((*readtable* scheme-readtable))
    (set-macro-character #\` #'quasiquote-read-macro)
    (set-macro-character #\, #'unquote-read-macro)
    (set-dispatch-macro-character #\# #\# #'sharp-sharp-read-macro)
    (set-dispatch-macro-character #\# #\T #'sharp-T-read-macro)
    (set-dispatch-macro-character #\# #\F #'sharp-F-read-macro)
    (set-dispatch-macro-character #\# #\D #'sharp-D-read-macro)))

(defun sharp-T-read-macro (stream subchar arg)
  #-Symbolics (declare (ignore stream subchar arg))
  #+Symbolics (progn stream subchar arg)
  t)

(defun sharp-F-read-macro (stream subchar arg)
  #-Symbolics (declare (ignore stream subchar arg))
  #+Symbolics (progn stream subchar arg)
  nil)

(defun sharp-D-read-macro (stream subchar arg)
  #-Symbolics (declare (ignore subchar arg))
  #+Symbolics (progn subchar arg)
  (lisp:let ((*read-base* 10.))
    (read stream)))

(defvar cl-sharp-sharp
	#+Symbolics #'(lisp:lambda (&rest ignore) (error "Symbolics loses"))
	#-Symbolics (get-dispatch-macro-character #\# #\#))

(defun sharp-sharp-read-macro (stream subchar arg)
  (cond (arg (funcall cl-sharp-sharp stream subchar arg))
	(t 'lisp:*)))

(defun quasiquote-read-macro (stream c)
  #+Symbolics c #-Symbolics (declare (ignore c))
  (list 'quasiquote (lisp:read stream t nil t)))

(defun unquote-read-macro (stream c)
  #+Symbolics c #-Symbolics (declare (ignore c))
  (list (lisp:let ((following-char
		     (lisp:peek-char nil stream nil stream t)))
	  (cond ((or (char= following-char #\@)
		     (char= following-char #\.))
		 (read-char stream)
		 'unquote-splicing)
		(t 'unquote)))
	(lisp:read stream t nil t)))

#-TI-REL2
(init-scheme-readtable)

#+quasiquote-has-bugs
(defun quasiquote-revert ()
  (lisp:let ((rt (copy-readtable nil)))
    (set-syntax-from-char #\` #\` *readtable* rt)
    (set-syntax-from-char #\, #\, *readtable* rt)))


;;;; System-specific hacks

(defvar *scheme-source-file-type* "SCM")

;;; Meta-. stuff adapted from OZ:<DAM.PROVER>METAP.LISP.2.  Thanks to
;;; DAM for figuring this out.

#+Symbolics
(eval-when (lisp:eval lisp:load lisp:compile)
  (when (>= (si:get-release-version) 7)
    (pushnew :symbolics-rel-7-or-after *features*)))

#+Symbolics
(progn 'lisp:compile
(setf (get 'define              'zwei:definition-function-spec-type) 'defun)
(setf (get 'define-macro        'zwei:definition-function-spec-type) 'defun)
(setf (get 'define-pseudo       'zwei:definition-function-spec-type) 'defun)
(setf (get 'define-pseudo-macro 'zwei:definition-function-spec-type) 'defun)

;;; The value of the property ZWEI:DEFINITION-FUNCTION-SPEC-FINDER
;;; should be a function which takes the ZWEI point after the definition
;;; symbol (e.g. DEFINE) and returns the point at the begining of the fspec.
;;; See the function ZWEI:GET-DEFINITION-FUNCTION-SPEC
;;;
;;; DEFINE-FSPEC-FINDER goes forward to the begining of the next atom.
;;; This means skipping white space and left parenthesis.

(defun define-fspec-finder (bp)
  (zwei:forward-over (cons (if (numberp (zl:character 65))
			       40.
			       #\()
			   zwei:*whitespace-chars*)
		     bp))

(mapc #'(lisp:lambda (definer)
	  (setf (get definer 'zwei:definition-function-spec-finder)
		#'define-fspec-finder))
      '(define
        define[subst]
	define-macro
	define-pseudo
	define-pseudo-macro))

;;; This stuff still doesn't make PP work.

(zl:defprop quasiquote grind-quasiquote si:grind-macro)
(defun grind-quasiquote (e loc) loc
  (si:gtyo #.(zl:character (char-code #\`)))
  (si:grind-form (cadr e) (zl:locf (cadr e))))
(zl:defprop unquote grind-unquote si:grind-macro)
(defun grind-unquote (e loc) loc
  (si:gtyo #.(zl:character (char-code #\,)))
  (si:grind-form (cadr e) (zl:locf (cadr e))))
(zl:defprop unquote-splicing grind-unquote-splicing si:grind-macro)
(defun grind-unquote-splicing (e loc) loc
  (si:gtyo #.(zl:character (char-code #\,)))
  (si:gtyo #.(zl:character (char-code #\@)))
  (si:grind-form (cadr e) (zl:locf (cadr e))))

(lisp:let ((type (zl:string *scheme-source-file-type*)))
  (cond ((not (lisp:member type fs:*its-uninteresting-types* :test #'equal))
	 (push type fs:*its-uninteresting-types*))))

(fs:define-canonical-type :scheme #,*scheme-source-file-type*) ;Scheme source

;;; Default mode for scheme source is lisp.

(unless (lisp:assoc :scheme fs:*file-type-mode-alist*)
  (setq fs:*file-type-mode-alist*
	(append fs:*file-type-mode-alist* (list (cons :scheme :lisp)))))

;;; Allow one to write  -*- Syntax: Scheme; -*-
;;; (Thanks to Alan Bawden and Thomas A. Russ.)

(defun (:scheme fs:syntax-attribute-handler) ()
  (values (list 'zl:readtable) (list scheme-readtable)))

#+symbolics-rel-7-or-after
(si:define-lisp-syntax :scheme (:readtable-place *scheme-readtable*
				:external-name "Scheme"
				:packages-must-use (("Scheme")))
  (zl:ferror "Cannot Set Lisp Context to Scheme.  Call SCHEME:SCHEME instead."))

#-symbolics-rel-7-or-after
(setf (get :scheme 'si:lisp-syntax)
      (get :common-lisp 'si:lisp-syntax))

;;; The following allows one to write -*- Mode: Scheme; -*-

(zl:defflavor zwei:scheme-mode () (zwei:lisp-syntax-mode-forms-mixin
				   zwei:lisp-language-mixin
				   zwei:major-mode))

(zl:defmethod (zwei:scheme-mode :mode-line-name) ()
  '#.(zl:string "Scheme"))

(zl:defmethod (zwei:scheme-mode :case :get-default-attribute :base) ()
  10)

(zl:defmethod (zwei:scheme-mode :case :get-default-attribute :syntax) ()
  :scheme)

(zl:defmethod (zwei:scheme-mode :mode-forms) ()
  '((zwei:set-syntax-table-indirection zwei:*mode-list-syntax-table*
				       zwei:*cl-list-syntax-table*)
    (zwei:set-comtab zwei:*mode-comtab*
		     (if (numberp (zl:character 65))
			 '(602 zwei:com-evaluate-and-exit
			   858 zwei:com-evaluate-and-exit
			   593 zwei:com-fill-long-comment)
			 '(#\Meta-Z         zwei:com-evaluate-and-exit
			   #\Control-Meta-Z zwei:com-evaluate-and-exit
			   #\Meta-Q	    zwei:com-fill-long-comment)))))

(zwei:defmode zwei:com-scheme-mode zwei:scheme-mode
  #.(zl:string "Sets things up for editing Scheme.
Like Lisp Mode -- if you've only got 8 fingers...")
  :scheme)
  
(zwei:set-comtab zwei:*standard-comtab*
		 '()
		 (zwei:make-command-alist '(zwei:com-scheme-mode)))

(zl:defmethod (zwei:scheme-mode :default-source-file-type) ()
  :scheme)

;;; Allow control-shift-E in ZWEI

(zl:defmethod (zwei:scheme-mode :eval-print-function) ()
  #'scheme-evaluate-and-print)

(defun scheme-evaluate-and-print (object)
  (declare (special zwei:*use-typeout*))
  (let ((val (eval object user-initial-environment)))
    (if zwei:*use-typeout*
	(write val)
	(zwei:typein-line "~A" (with-output-to-string (stream)
				 (write val stream))))
    (values val object)))

)  ;(... ngorp) scilobmyS+#

;;;Start Explorer specifics
#+TI-REL3
(progn

;;Define Scheme major mode for Zmacs  
zwei:
(defmajor com-scheme-mode scheme-mode "Scheme"
	  "Sets things up for editing Scheme code." ()
  (setq *space-indent-flag* t)
  (setq *paragraph-delimiter-list* '(#\. #\space #\tab #\"))
  (setq *comment-start* 'lisp-find-comment-start-and-end)
  ;;The following three are non-Zmacs vars that are made settable by the setf's below
  (setq *print-array* t)			;print arrays readably
  (setq ucl:*default-prompt* 'pseudoscheme:scheme-prompt-when-appropriate)
  (setq ucl:*default-read-function* 'pseudoscheme:scheme-read-when-apropriate)
  (set-char-syntax list-slash *mode-list-syntax-table* #\\)
  (set-char-syntax list-alphabetic *mode-list-syntax-table* #\/)
  (set-comtab *mode-comtab*
	      '(#\tab com-indent-for-lisp
		#\rubout com-tab-hacking-rubout
		#\c-rubout com-rubout
		#\m-z com-compile-and-exit
		#\c-m-z com-evaluate-and-exit))
  )

;;This doesn't clobber any of the other Lisp modes (eg Common Lisp), since they
;;set their readtable when the major mode switch occurs.
(defvar zwei:scheme-mode-hook #'(lisp:lambda ()
				  (setq *readtable* pseudoscheme:scheme-readtable))
  "Simple function which specifies the use of the Scheme readtable within Scheme.")

;;Make these variables settable (and thus undoable) in Zmacs modes.
(dolist (symbol '(*print-array* ucl:*default-prompt* ucl:*default-read-function*))
  (setf (get symbol 'zwei:mode-settable-p) lisp:t))

;;; The following must be :common-lisp rather than the more intuitive
;;; value of :scheme.  The purpose of this system variable is to tell the difference
;;; between kludgy Zetalisp and modern Common Lisp.  We are doing Scheme on top
;;; of Common Lisp.
(defvar *scheme-value-for-sys-lisp-mode* :common-lisp "Appropriate Scheme value for the variable sys:*lisp-mode*")

;;Use the correct readtable when compiling Scheme forms/files
(ticl:advise (:property :mode fs:file-attribute-bindings) :around get-scheme-bindings-if-appropriate lisp:nil
  (lisp:let ((mode-keyword (third arglist)))
    (if (eq :scheme mode-keyword)
	(values '(sys:*lisp-mode* sys:*readtable* sys:*reader-symbol-substitutions* zwei::*default-major-mode*)
		`(,*scheme-value-for-sys-lisp-mode* ,pseudoscheme:scheme-readtable lisp:nil :scheme))
	:do-it)))

;;Compile the above advice
(eval-when (lisp:eval load lisp:compile)
  (ticl:compile-encapsulations '(:property :mode fs:file-attribute-bindings)))

(ticl:defprop zwei:scheme-mode lisp:t zwei:all-uppercase) ;;case is insignificant
(ticl:defprop zwei:scheme-mode :lisp zwei:editing-type)   ;;Scheme is Lisp

;;Handle DEFINE... top-level forms for sectionizing (yes, a hack, but this is better than including the complete function)
(ticl:advise zwei:symbol-from-string :around check-for-define lisp:nil
  (lisp:let ((str (first arglist))
	     (line (second arglist))
	     (sym (fourth arglist)))
    (if (and (consp sym) ;;eg (foo a)
	     (not (null line))
	     (> (length line) 10.)
	     (string-equal "(define" line :end2 7))
	(values (first sym) str)
	:do-it)))

;;Compile advise
(eval-when (lisp:eval load lisp:compile)
  (ticl:compile-encapsulations 'zwei:symbol-from-string))

;;Treat "SCM" just like "LISP"
(lisp:let ((type (string *scheme-source-file-type*)))
  (cond ((not (lisp:member type fs:*its-uninteresting-types* :test #'equal))
	 (push type fs:*its-uninteresting-types*))))

;;"One should always use canonical types"
(fs:define-canonical-type :scheme #,*scheme-source-file-type*) ;Scheme source

;;; Default Zmacs Major Mode for Scheme source is Scheme
(unless (lisp:assoc :scheme fs:*file-type-mode-alist*)
  (setq fs:*file-type-mode-alist*
	(append fs:*file-type-mode-alist* (list (cons :scheme :scheme)))))

;;Make Scheme Mode accessible
(zwei:set-comtab zwei:*standard-comtab*
		 '()
		 (zwei:make-command-alist '(zwei:com-scheme-mode)))

(defun in-scheme? ()
  "Boolean, which is true when we are in Scheme, meaning that the current readtable
is the Scheme readtable."
  (eq? *readtable* scheme-readtable))

(defvar *saved-prompt*  ucl:*default-prompt* "Saved old prompt; used when we temporarily want to set it")     

(defun scheme-prompt-when-appropriate ()
  "When in Scheme (using the Scheme readtable), present the user with the Scheme prompt (==>),
else use the standard default prompt."
  (let* ((scheme-prompt "==> ")
	 (default-non-scheme-prompt "> ")
	 (prompt (if (eq *saved-prompt* 'scheme-prompt-when-appropriate)
		     default-non-scheme-prompt *saved-prompt*))) ;avoid infinite loop
    (if (eq *readtable* scheme-readtable)
      scheme-prompt				; ** ==>  is specified in Revised^3
      (if (stringp prompt)			; can either be a string or a function
	  prompt
	  (funcall prompt)))))

(defvar *saved-read-function* ucl:*default-read-function*)

(defun scheme-read-when-apropriate ()
  "When in Scheme (using the Scheme readtable), do Scheme preprocessing."
  (let* ((read-function (if (eq *saved-read-function*  ucl:*default-read-function*)
			    'ucl:read-for-ucl	;avoid endless recursion
			    *saved-read-function*))
	 (expression (funcall read-function)))
    (if (eq *readtable* scheme-readtable)
	(preprocess-top expression)
	expression)))

) ;;;ngorp rerolpxE+#
;;;end Explorer specific things


#+(and VAX VMS)
(progn
(system::define-list-print-function quasiquote (list stream)
  (declare (list list))
  (if (= (length list) 2)
      (format stream "`~W" (second list))
      (format stream "~1!~@{~W~^ ~:_~}~." list)))

(system::define-list-print-function unquote (list stream)
  (declare (list list))
  (if (= (length list) 2)
      (format stream ",~W" (second list))
      (format stream "~1!~@{~W~^ ~:_~}~." list)))

(system::define-list-print-function unquote-splicing (list stream)
  (declare (list list))
  (if (= (length list) 2)
      (format stream ",@~W" (second list))
      (format stream "~1!~@{~W~^ ~:_~}~." list)))
);ngorp

;;;; Initialization

(defmacro setq-standard-value (var val)
  #+Symbolics
  `(zl:setq-standard-value
      ,(or (cadr (lisp:assoc var '((*package* zl:package)
				   (*print-array* si:*prinarray*)
				   (*readtable* zl:readtable))))
	   var)
      ,val)
  #-Symbolics
  `(setq ,var ,val))

(defvar *saved-package*     *package*)
(defvar *saved-readtable*   *readtable*)
(defvar *saved-print-array* *print-array*)
#+Symbolics
(defvar *saved-print*	    si:*command-loop-print-function*)
#+Symbolics
(defvar *saved-eval*	    si:*command-loop-eval-function*)

;;; (scheme:scheme) does the following:
;;;   - goes into the Scheme package
;;;   - makes vectors print correctly (by setting *print-array*)
;;;   - clobbers the read table
;;;   
;;; Calling this function is not necessary on Explorers when in Zmacs Break
;;; window of a Scheme file (presumably the most common case).


(defun scheme ()
  "Initialize for execution of Scheme programs."

  (when (or (not (eq *package* (find-package 'scheme)))
	    (not (eq *readtable* scheme-readtable)))
    (setq *saved-package* *package*)
    (setq *saved-readtable* *readtable*)
    (setq *saved-print-array* *print-array*)
    #+TI-REL3
    (progn
      (setq *saved-prompt* ucl:*default-prompt*)
      (setq *saved-read-function* ucl:*default-read-function*))
    #+Symbolics
    (progn
      (setq *saved-eval* si:*command-loop-eval-function*)
      (setq *saved-print* si:*command-loop-print-function*)))

  (setq-standard-value *package* (find-package 'scheme))
  #+TI-REL2
  (init-scheme-readtable)
  (setq-standard-value *readtable* scheme-readtable)
  (setq-standard-value *print-array* t)

  #+TI-REL3
  (progn
    (setq ucl:*default-prompt* 'scheme-prompt-when-appropriate)
    (setq ucl:*default-read-function* 'scheme-read-when-apropriate))

  #+Symbolics
  (progn
    (setq si:*command-loop-eval-function*
	  #'(lisp:lambda (obj)
	      (lisp:eval (preprocess-top obj))))

    (setq si:*command-loop-print-function*
	  (lambda (values)
	    (mapc (lambda (value)
		    (zl:send zl:standard-output :fresh-line)
		    (write value))
		  values))))

  (format t "~&Scheme.~&")
  (values))

(defun quit ()
  (setq-standard-value *package* *saved-package*)
  (setq-standard-value *readtable* *saved-readtable*)
  (setq-standard-value *print-array* *saved-print-array*)
  #+TI-REL3
  (progn 
    (setq ucl:*default-prompt* *saved-prompt*)
    (setq ucl:*default-read-function* *saved-read-function*))
  #+Symbolics
  (progn
    (setq si:*command-loop-eval-function* *saved-eval*)
    (setq si:*command-loop-print-function* *saved-print*))
  (format t "~&Not Scheme.~&")
  (values))

(defun declare-pseudoscheme-package (pkg)
  #-Symbolics (declare (ignore pkg))
  #+Symbolics (pushnew (if (packagep pkg) pkg (find-package pkg))
                       si:*reasonable-packages*)
  t)

(setq *preprocess?* t)

(provide "pseudoscheme")


;;; Set up a scheme package for casual use.  Other packages can be set
;;; up in the same manner as the scheme package, if desired.

(in-package 'scheme :use '(pseudoscheme))
(declare-pseudoscheme-package 'scheme)
(lisp:export '(scheme) (lisp:find-package 'scheme))
