;;; -*-Scheme-*-
;;;
;;;	Copyright (c) 1985 Massachusetts Institute of Technology
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3.  All materials developed as a consequence of the use of
;;;	this software shall duly acknowledge such use, in accordance
;;;	with the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5.  In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

;;;; State Space Implementation

(declare (usual-integrations))

;;;
;;; A STATE-SPACE is a tree of STATE-POINTs, except that the pointers
;;; in the tree point towards the root of the tree rather than its
;;; leaves.  These pointers are the NEARER-POINT of each point.
;;;
;;; Each point in the space has two procedures, a FORWARD-THUNK and a
;;; BACKWARD-THUNK.  To move the root of the space to an adjacent
;;; point, one executes the BACKWARD-THUNK of that point, then makes
;;; the FORWARD-THUNK and BACKWARD-THUNK of the old root be the
;;; BACKWARD-THUNK and the FORWARD-THUNK of the new root,
;;; respectively.  Thus after having done this, one moves back to the
;;; old root again by executing what was formerly the FORWARD-THUNK of
;;; the new root.
;;;

(define make-state-space)
(define execute-at-new-state-point)
(define translate-to-state-point)
(let ()

;;; State space abstraction

(set! make-state-space
(named-lambda (make-state-space)
  (let ((space (vector state-space-tag '())))
    ;; The state-space is kept in the nearer-point of the root point,
    ;; for convenience since it is naturally needed there.
    (vector-set! space 1 (make-state-point '() '() space))
    space)))

(define state-space-tag
  "State Space")

(define (state-space? object)
  (and (vector? object)
       (not (zero? (vector-length object)))
       (eq? (vector-ref object 0) state-space-tag)))

(define (nearest-point space)
  (vector-ref space 1))

(define (clobber-space! space nearest-point)
  (vector-set! space 1 nearest-point))

;;; State point abstraction

(define (make-state-point forward-thunk backward-thunk nearer-point)
  (vector state-point-tag forward-thunk backward-thunk nearer-point))

(define state-point-tag
  "State Point")

(define (state-point? object)
  (and (vector? object)
       (not (zero? (vector-length object)))
       (eq? (vector-ref object 0) state-point-tag)))

(define (forward-thunk point)
  (vector-ref point 1))

(define (backward-thunk point)
  (vector-ref point 2))

(define (nearer-point point)
  (vector-ref point 3))

(define (clobber-point! point forward-thunk backward-thunk nearer-point)
  (vector-set! point 1 forward-thunk)
  (vector-set! point 2 backward-thunk)
  (vector-set! point 3 nearer-point))

(set! execute-at-new-state-point
(named-lambda (execute-at-new-state-point space change restore body)
  (let ((old-point (nearest-point space)))
    (translate-to-adjacent-point (make-state-point restore change old-point))
    (let ((value (body)))
      (translate-to-state-point old-point)))))

(set! translate-to-state-point
(named-lambda (translate-to-state-point point)
  ;; Follow the nearer-point chain to its end, then move the root back
  ;; to the desired point one step at a time.
  (let ((point* (nearer-point point)))
    (if (state-point? point*)
	(begin (translate-to-state-point point*)
	       (translate-to-adjacent-point point))))))

(define (translate-to-adjacent-point new-root)
  ;; This next line turns off all interrupts except the GC interrupt,
  ;; and returns the old interrupt mask as its value.
  (let ((interrupt-enables (set-interrupt-enables! 1)))
    (let ((old-root (nearer-point new-root))
	  (old->new (backward-thunk new-root)))
      (let ((space (nearer-point old-root)))
	(clobber-space! space new-root)
	(clobber-point! old-root old->new (forward-thunk new-root) new-root)
	;; Set the forward and backward thunks of new-root so that
	;; they will be garbage collected if old-root is dropped.
	(clobber-point! new-root '() '() space)
	;; Pass the interrupt mask to the thunk so that it can decide
	;; when to re-enable interrupts.
	(old->new interrupt-enables)))))

;;; end LET.
)

;;; DYNAMIC-WIND implemented in terms of the state-space model.  The
;;; BEFORE and AFTER thunks are executed with interrupts disabled
;;; completely, as that is usually what is needed.  For finer control,
;;; one can use EXECUTE-AT-NEW-STATE-POINT directly.

(define system-state-space
  (make-state-space))

(define (current-dynamic-state)
  (nearest-state-point system-state-space))

(define (dynamic-wind before during after)
  (execute-at-new-state-point system-state-space
			      (lambda (interrupt-enables)
				(before)
				(set-interrupt-enables! interrupt-enables))
			      (lambda (interrupt-enables)
				(after)
				(set-interrupt-enables! interrupt-enables))
			      during))