;;; Scanner and Parser for Stacks example

;;; Assumes parse-utils.s is loaded (includes scanner)

;;; Also some common routines (eg primops)

;;; **********************************************************

;;; Top-level interface

(define scan&parse
  (lambda (string)
    (parse-top-level grammar-9
      (string->token-stream automaton-9 string))))

;;; ***********************************************************

;;; Lexical Specification

;;; same as automaton-4

(define automaton-9
  '((start-state
      (cond
        (#\space (drop (goto start-state)))
        (#\tab (drop (goto start-state)))
        (#\newline (drop (goto start-state)))
        (alphabetic (shift (goto identifier-state)))
        (numeric (shift (goto number-state)))
        (#\, (drop (emit comma)))
        (#\; (drop (emit semicolon)))
        (#\+ (shift (goto identifier-state)))
        (#\- (goto identifier-state))
        (#\* (goto identifier-state))
        (#\/ (shift (goto identifier-state)))
        (#\= (goto identifier-state))
        (#\( (drop (emit lparen)))
        (#\) (drop (emit rparen)))
        (#\nul (emit end-marker))
        (#\% (drop (goto comment-state)))))
    (identifier-state
      (cond
        (alphabetic (shift (goto identifier-state)))
        (numeric (shift (goto identifier-state)))
        (#\* (shift (goto identifier-state)))
        (#\+ (shift (goto identifier-state)))
        (#\- (shift (goto identifier-state)))
        (#\/ (shift (goto identifier-state)))
        (#\! (shift (goto identifier-state)))
        (#\= (shift (goto identifier-state)))
        (#\? (shift (goto identifier-state)))
        (else (emit cook-identifier))))
    (comment-state
      (cond
        (#\newline (drop (goto start-state)))
        (else (drop (goto comment-state)))))
    (number-state
      (cond
        (numeric (shift (goto number-state)))
        (else (emit cook-number))))
    ))

;;; Keywords for our language are defined as follows:

(define **keywords-list** '(if then else letrec in =))

;;; ****************************************************************

;;; Grammar

(define grammar-9
  '((start-state
      ((goto expression)))
    (expression
      (cond
        (number
          ((check/shift number)
           (reduce lit-exp)))
        (identifier
          ((check/shift identifier)
           (reduce var-exp)))
        (if
          ((check/drop if)
           (process-nt expression)
           (check/drop then)
           (process-nt expression)
           (check/drop else)
           (process-nt expression)
           (reduce if-exp)))
        (letrec
          ((check/drop letrec)
           (process-nt declaration-list)
           (check/drop in)
           (process-nt expression)
           (reduce letrec-exp)))
        (lparen
          ((check/drop lparen)
           (check/shift identifier)          ; operator must be an identifier
           (process-nt operand-list)
           (check/drop rparen)
           (reduce app-exp)))))
  (declaration
    (cond
      (lparen
        ((check/drop lparen)
         (check/shift identifier)
         (process-nt formal-parameter-list)
         (check/drop rparen)
         (check/drop =)
         (process-nt expression)
         (reduce rec-decl)))))
  (declaration-list                     ; bounded by "in"
    (cond
      (in
        ((emit-list)))
      (else
        ((process-nt declaration)
         (goto declaration-list)))))
  (formal-parameter-list		; bounded by rparen
    (cond
      (rparen
        ((emit-list)))
      (else
        ((check/shift identifier)	
         (goto formal-parameter-list)))))
  (operand-list                         ; bounded by rparen
    (cond
      (rparen
        ((emit-list)))
      (else
        ((process-nt expression)
         (goto operand-list)))))
    ))
       
;;; End of syntactic specification

;;; *****************************************************************
;;; *****************************************************************

;;; Some common things for all interpreters:

;;; *****************************************************************

;;; Data Structure Definitions

(define-record rec-decl (name formals body))

(define-record empty-env ())
(define-record extended-env (names vals old-env))
(define-record extended-rec-env (decls env))

;; unnecessary after interp1.s
(define-record closure (rec-decl env)) 


;;; *****************************************************************

;;; Primops

(define-record prim-1 (unary-op))
(define-record prim-2 (binary-op))

(define apply-unary-primop
  (lambda (primop arg)
    (case primop
      ((+1-op) (+ arg 1))
      ((-1-op) (- arg 1))
      ((zero-op) (if (zero? arg) 1 0))
      (else (error 'apply-unary-primop "Unknown Primop: ~s"
              primop)))))

(define apply-binary-primop
  (lambda (primop arg1 arg2)
    (case primop
      ((+-op)  (+ arg1 arg2))
      ((--op)  (- arg1 arg2))
      ((*-op)  (* arg1 arg2))
      (else (error 'apply-binary-primop "Unknown Primop: ~s"
              primop)))))

;; this gets grouped with the primops

(define true-value?
  (lambda (v)
    (not (zero? v))))

;;; ****************************************************************

;;; The Initial Environment

(define init-entries
  '((+ +-op 2)
    (- --op 2)
    (* *-op 2)
    (add1 +1-op 1)
    (sub1 -1-op 1)
    (zero? zero-op 1)))

(define make-init-env 
  (lambda () (build-init-env init-entries)))

;;; ****************************************************************

;;; Stacks

(define make-stack
  (lambda (size bp sp)
    (list '*stack sp bp (make-vector size))))

;; (stack-push v1 ... vn s) -- returns modified stack

(define stack-push
  (lambda l
    (let ((n (- (length l) 1))
          (s (car (last-pair l))))
      (let ((a (cadddr s))
            (sp (cadr s)))
        (let loop ((p (+ sp n))
                   (l l)
                   (count n))
          (if (zero? count)
            ;; nothing more to put on stack, so set sp and return the
            ;; stack object
            (begin
              (set-car! (cdr s) (+ sp n))
              s)
            ;; otherwise, put first elt of l on the stack, counting
            ;; down from eventual top
            (begin
              (vector-set! a p (car l))
              (loop (- p 1) (cdr l) (- count 1)))))))))

;; (stack-pop s n (lambda (v1 ... vn s) ...))  -- inverse of stack-push
;; maybe n should be number of lambda-items rather than number of v's.

(define stack-pop
  (lambda (count s rcvr)
    (let ((a (cadddr s))
          (sp (cadr s)))
      (let loop ((l (list s))           ; output list
                 (n count)              ; number of output values to
                                        ; be retrieved
                 (p (- sp count -1))); stack pointer, starting from bottom
        (if (zero? n)
          (begin
            (set-car! (cdr s) (- sp count))       ; set the sp
            (apply rcvr l))
          (loop 
            (cons (vector-ref a p) l)
            (- n 1)
            (+ p 1)))))))

(define stack-ref
  (lambda (s ptr)
    (vector-ref (cadddr s) ptr)))

(define stack->sp
  (lambda (s)
    (cadr s)))

(define stack->bp
  (lambda (s)
    (caddr s)))

(define stack-set-bp
  (lambda (bp s)
    (set-car! (cddr s) bp)
    s))

(define stack-set-sp
  (lambda (sp s)
    (set-car! (cdr s) sp)
    s))

(define debug-print
  (lambda (s . l)                       ; optional 2nd arg is starting point
    (let ((sp (if (null? l) (stack->sp s) (car l))))
      (letrec
        ((loop (lambda (kp n)
                 (if (or (zero? kp) (zero? n))
                   (printf "~%")
                   (begin
                     (printf "a[~s] = ~s~%" kp (stack-ref s kp))
                     (loop (- kp 1) (- n 1)))))))
        (printf "sp = ~s bp = ~s~%" 
          (stack->sp s)
          (stack->bp s))
        (loop sp 10)))))

(define debug-points '())

;;; Tests

(define pgm1 "1")

(define pgm2 "(+ 40 50)")

(define pgm3 
  "letrec (fact x) = if (zero? x) then 1 else (* x (fact (sub1 x)))
  in (fact 6)")

(define pgm4
  "letrec (diff x y) = (- x y) in (diff 6 4)") ; test argument order


(define test3
  (lambda (n)
    (run (format
           "letrec (fact x) = if (zero? x) then 1 else (* x (fact (sub1 x)))
              in (fact ~s)"
           n))))
