;;; -*- Scheme -*-
;;; This was hacked at Gustavus Adolphus College in 1992 to provide a
;;; graphical display of the machines in the Schematik programming environment.
;;; Other than those changes, which should all be marked with ; [max, ...]
;;; comments, it is essentially identical to the original MIT version
;;; which bore the following notice:
;;;
;;; Copyright (c) 1990 Massachusetts Institute of Technology
;;; 
;;; This material was developed by the Scheme project at the 
;;; Massachusetts Institute of Technology, Department of Electrical 
;;; Engineering and Computer Science.  Permission to copy this 
;;; material, to redistribute it, and to use it for any 
;;; non-commercial purpose is granted, subject to the following 
;;; restrictions and understandings.
;;;
;;; 1. Any copy made of this material must include this copyright 
;;; notice in full.
;;; 
;;; 2. Users of this material agree to make their best efforts (a) 
;;; to return to the MIT Scheme project any improvements or 
;;; extensions that they make, so that these may be included in 
;;; future releases; and (b) to inform MIT of noteworthy uses of 
;;; this material.
;;;
;;; 3. All materials developed as a consequence of the use of this
;;; material shall duly acknowledge such use, in accordance with 
;;; the usual standards of acknowledging credit in academic 
;;; research.
;;;
;;; 4. MIT has made no warrantee or representation that this 
;;; material (including the operation of software contained 
;;; therein) will be error-free, and MIT is under no obligation to 
;;; provide any services, by way of maintenance, update, or 
;;; otherwise.
;;;
;;; 5. In conjunction with products arising from the use of this 
;;; material, there shall be no use of the name of the 
;;; Massachusetts Institute of Technology nor of any adaptation 
;;; thereof in any advertising, promotional, or sales literature 
;;; without prior written consent from MIT in each case. 


;;;;  6.001 Register Machine Simulator

;;;;  Machine-checking stuff has been added -- see comments
;;;; with ***

;;; Syntactic sugar for DEFINE-MACHINE
;;;   Magic syntax hack... DO NOT expect to understand this. 
;;;   Hal doesn't and he wrote it!
(enable-language-features)
(define-macro (define-machine name registers controller)
  `(define ,name
     (sequence
       (with-errors-ignored (lambda () (close-windows ,name))) ; [max, 11/26/91]
       (build-model ',(cdr registers) ',(cdr controller) ',name))))
(syntax-table-define
    (access *student-syntax-table* student-package) ; [max 11/26/91]
    'define-machine 
  (syntax-table-ref system-global-syntax-table 'define-machine))
(define dynamic-wind dynamic-wind) ; [max, 11/26/91]
(define make-graphics-device make-graphics-device) ; [max, 11/26/91]
(define schematik-style-graphics-device-type ; [max, 11/26/91]
  schematik-style-graphics-device-type) ; [max, 11/26/91]
(define graphics-set-coordinate-limits
  graphics-set-coordinate-limits)  ; [max, 11/26/91]
(define graphics-device-coordinate-limits
  graphics-device-coordinate-limits)  ; [max, 11/26/91]
(define graphics-coordinate-limits graphics-coordinate-limits)  ; [max, 11/26/91]
(define graphics-draw-text graphics-draw-text) ; [max, 11/26/91]
(define graphics-operation graphics-operation) ; [max, 11/26/91]
(define graphics-clear graphics-clear) ; [max, 11/26/91]
(define graphics-bind-drawing-mode graphics-bind-drawing-mode) ; [max, 11/26/91]
(define graphics-close graphics-close) ; [max, 11/26/91]
(define graphics-flush graphics-flush) ; [max, 11/26/91]
(define graphics-enable-buffering graphics-enable-buffering) ; [max, 11/26/91]
(define condition/internal? condition/internal?) ; [max, 11/26/91]
(define condition/error? condition/error?) ; [max, 11/26/91]
(define call-with-current-continuation
  call-with-current-continuation) ; [max, 11/26/91]
(define bind-condition-handler bind-condition-handler) ; [max, 11/26/91]
(define object->string object->string)  ; [max, 11/26/91]
(define with-values with-values) ; [max, 11/26/91]
(define string-append string-append) ; [max, 11/26/91]
(define symbol->string symbol->string) ; [max, 11/26/91]
(define string->symbol string->symbol) ; [max, 5/6/92]
(define string-upcase string-upcase) ; [max, 5/14/92]
(define real-time-clock real-time-clock) ; [max, 12/1/91]
(disable-language-features)
 
(define (with-errors-ignored thunk) ; [max, 11/26/91]
  (call-with-current-continuation
   (lambda (return)
     (bind-condition-handler '()
                             (lambda (condition)
                               (if (and (condition/error? condition)
                                        (not (condition/internal? condition)))
                                   (return #f)
                                   #f))
                             thunk))))
;;;model building

(define (build-model registers controller name)
  (let ((machine (make-new-machine)))
    (set-up-registers machine registers)
    (set-up-controller machine controller)
    (remote-define machine '*name* name)  ; [max, 11/26/91]
    (set-up-windows machine) ; [max, 11/26/91]
    machine))

(define (set-up-registers machine registers)
  (mapc (lambda (register-name)
          (make-machine-register machine register-name))
        registers))

(define (mapc proc l)
  (if (null? l)
      'done
      (sequence (proc (car l))
                (mapc proc (cdr l)))))

(define (set-up-controller machine controller)
  (let ((instructions                                  ; ***
         (build-instruction-list machine (cons '*start* controller))))
    (remote-set machine '*pc-positions*    ; [max, 11/26/91]
                (reverse (remote-get machine '*pc-positions*)))
    (remote-set machine '*controller* controller) ; [max, 11/26/91]
    (mapc (lambda (inst) (check-instruction machine (cdr inst)))  ; ***
         instructions)))

(define (build-instruction-list machine op-list)
  (define (iter op-list counter) ; [max, 11/26/91] added counter stuff
    (if (null? op-list)
        '()
        (let ((rest-of-instructions
               (iter (cdr op-list) (+ 1 counter))))
          (if (label? (car op-list))
              (sequence
                (declare-label machine
                               (car op-list)
                               rest-of-instructions)
                rest-of-instructions)
              (sequence
                (remote-set machine '*pc-positions*
                            (cons counter
                                  (remote-get machine '*pc-positions*)))
                (cons (make-machine-instruction machine
                                                (car op-list))
                      rest-of-instructions))))))
  (iter op-list 0))

(define (label? expression)
  (symbol? expression))

(define (make-machine-register machine name)
  (remote-set machine                                  ; ***
              '*registers*
              (cons name (remote-get machine '*registers*)))
  (remote-define machine name (make-register name machine))) ; [max, 11/26/91] machine added to make-register

;;;register model

(define (make-register name machine) ; [max, 11/26/91] machine added
  (define contents nil)
  (define (get) contents)
  (define (set value)
    (update-registers-window machine name contents value) ; [max, 11/26/91]
    (set! contents value))
  (define (dispatch message)
    (cond ((eq? message 'get) (get))
          ((eq? message 'set) set)
          (else (error "Unknown request -- REGISTER"
                       name
                       message))))
  dispatch)

(define (get-contents register)
  (register 'get))

(define (set-contents register value)
  ((register 'set) value))

(define (declare-label machine label labeled-entry)
  (let ((defined-labels (remote-get machine '*labels*)))
    (if (memq label defined-labels)
        (error "Multiply-defined label" label)
        (sequence
         (remote-define machine label labeled-entry)
         (remote-set machine
                      '*labels*
                      (cons label defined-labels))))))


;;; stack model -- monitored stack

(define (make-stack)
  (define s '())
  (define depth 0) ; [max, 11/26/91]
  (define max-depth 0) ; [max, 12/1/91]
  (define machine #f) ; [max, 11/26/91]
  (define (reset-statistics)            ; [max, 12/1/91]
    (set! max-depth 0))
  (define (statistics)                  ; [max, 12/1/91]
    (list 'max 'stack 'depth: max-depth))
  (define (push x)
    (set! s (cons x s))
    (update-stack-window-push machine depth x) ; [max, 11/26/91]
    (set! depth (1+ depth)) ; [max, 11/26/91]
    (set! max-depth (max max-depth depth))) ; [max, 12/1/91]
  (define (pop)
    (if (null? s)
        (error "Empty stack -- POP")
        (let ((top (car s)))
          (set! depth (-1+ depth)) ; [max, 11/26/91]
          (update-stack-window-pop machine depth top) ; [max, 11/26/91]
          (set! s (cdr s))
          top)))
  (define (initialize)
    (update-stack-window-initialize machine) ; [max, 11/26/91]
    (set! s '())
    (set! depth 0)) ; [max, 11/26/91]
  (define (set-machine new-machine) ; [max, 11/26/91]
    (set! machine new-machine))
  (define (dispatch message)
    (cond ((eq? message 'push) push)
          ((eq? message 'pop) (pop))
          ((eq? message 'initialize) (initialize))
          ((eq? message 'statistics) (statistics))
          ((eq? message 'reset-statistics) (reset-statistics))
          ((eq? message 'set-machine) set-machine) ; [max, 11/26/91]
          (else (error "Unknown request -- STACK" message))))
  dispatch)


(define (pop stack)
  (stack 'pop))

(define (push stack value)
  ((stack 'push) value))

;;;name-value association

(define (remote-get machine variable)
  (eval variable machine))

(define (remote-set machine variable value)
  (eval (list 'set! variable (list 'quote value))
        machine))

(define (remote-define machine variable value)
  (eval (list 'define variable (list 'quote value))
        machine))

;Want to check instructions after they've been processed, because some
;checks cannot be done on the first pass.  But instructions get
;turned into procedures, and we want the original expression for
;checking.  Don't know how to extract body text from a procedure, so
;instead I'll change instructions to be a pair: (procedure . expression)

(define (make-machine-instruction machine op)
  (cons (eval (list 'lambda '() op) machine)
        op))



;;; Monitored stack machine maker.

(define (make-new-machine)
  (make-environment
   (define *step* #f)                   ; [max, 11/26/91]
   (define *stack-window* #f)           ; [max, 11/26/91]
   (define *registers-window* #f)       ; [max, 11/26/91]
   (define *controller-window* #f)      ; [max, 11/26/91]
   (define *pc-positions* '())          ; [max, 11/26/91]
   (define *controller* #f)             ; [max, 11/26/91]
   (define *machine* #f)                ; [max, 11/26/91]
   (define *registers* '())                            ; *** ADDED
   (define *labels* '())
   (define *the-stack* (make-stack))
   (define *instruction-counter* 0)     ; [max, 12/1/91]
   (define *start-time* 0)              ; [max, 12/1/91]
   (define (initialize-stack)
     (*the-stack* 'initialize))
   (define fetch get-contents)

   (define *program-counter* '())
   (define (execute ops)
     (set! *instruction-counter* (1+ *instruction-counter*)) ; [max, 12/1/91]
     (update-controller-window *machine* *program-counter* ops)
     (set! *program-counter* ops)
     (if (null? *program-counter*)
         (statistics *machine*)         ; [max, 12/1/91]
         (if *step*                     ; [max, 11/26/91]
             'stepped                   ; [max, 11/26/91]
             ((car (car *program-counter*))))))             ; *** see below
   (define (normal-next-instruction)
     (execute (cdr *program-counter*)))

   (define (assign register value)
     (set-contents register value)
     (normal-next-instruction))

   (define (save reg)
     (push *the-stack* (get-contents reg))
     (normal-next-instruction))

   (define (restore reg)
     (set-contents reg (pop *the-stack*))
     (normal-next-instruction))

   (define (goto new-sequence)
     (execute new-sequence))

   (define (branch predicate alternate-next)
     (if predicate
         (goto alternate-next)
         (normal-next-instruction)))

   (define (perform operation)
     (normal-next-instruction))

   ;; end of make-new-machine
   ))


;;;rest of simulator interface

(define (reset-statistics machine)      ; [max, 12/1/91]
  ((remote-get machine '*the-stack*) 'reset-statistics)
  (remote-set machine '*instruction-counter* 0)
  (remote-set machine '*start-time* (real-time-clock)))

(define (statistics machine)            ; [max, 12/1/91]
  (let ((stop-time (real-time-clock)))
    (let ((elapsed-time (/ (- stop-time (remote-get machine '*start-time*))
                           1000)))
      (list (list (remote-get machine '*instruction-counter*) 'ops)
            ((remote-get machine '*the-stack*) 'statistics)
            (list 'elapsed 'time: elapsed-time 'secs)))))

(define (remote-fetch machine register-name)
  (get-contents (remote-get machine register-name)))

(define (remote-assign machine register-name value)
  (set-contents (remote-get machine register-name) value)
  'done)

(define (start machine)
  (reset-statistics machine)
  (eval '(goto *start*) machine))

(define (proceed machine)               ; [max, 11/26/91]
  (if (null? (remote-get machine '*program-counter*))
      'no-next-instruction
      (eval '((car (car *program-counter*))) machine)))

(define (step machine)                  ; [max, 11/26/91]
  (dynamic-wind
   (lambda () (remote-set machine '*step* #t))
   (lambda () (proceed machine))
   (lambda () (remote-set machine '*step* #f))))

(define (set-pc machine label)          ; [max, 11/26/91]
  (if (eq? (remote-get machine label)   ; [max, 12/1/91] (kludge)
           (remote-get machine '*start*))
      (reset-statistics machine))
  (dynamic-wind
   (lambda () (remote-set machine '*step* #t))
   (lambda () (eval (list 'goto label) machine))
   (lambda () (remote-set machine '*step* #f)))
  'done)

(define (initialize-stack machine)      ; [max, 11/26/91]
  (eval '(initialize-stack) machine))

(define (close-windows machine)  ; [max, 11/26/91]
  (mapc (lambda (sym)
          (let ((win (remote-get machine sym)))
            (if win
                (sequence
                  (graphics-close win)
                  (remote-set machine sym #f)))))
        '(*stack-window* *registers-window* *controller-window*)))

;;; [max, 11/26/91] The below has absolutely no business being in this file,
;;; but heads off such a common frustration trying out the factorial machine
;;; at such a low cost that I couldn't resist.
(define (=1? n)
  (= n 1))

;;; [max, 11/26/91] graphics support
(define (set-up-windows machine)
  (remote-set machine '*machine* machine)
  (((remote-get machine '*the-stack*) 'set-machine) machine)
  (remote-set machine '*registers-window* (make-registers-window machine))
  (remote-set machine '*controller-window* (make-controller-window machine))
  (if (uses-stack? machine)
      (remote-set machine '*stack-window* (make-stack-window machine)))
  'done)

(define (uses-stack? machine)
  (define (loop instrs)
    (cond ((null? instrs) #f)
          ((label? (car instrs)) (loop (cdr instrs)))
          ((eq? (instruction-type (car instrs)) 'save) #t)
          ((eq? (instruction-type (car instrs)) 'restore) #t)
          (else (loop (cdr instrs)))))
  (loop (remote-get machine '*controller*)))

(define baseline-skip 12)
(define text-depth 6)
(define registers-width 150)
(define max-stack-depth 100)
(define stack-width 150)
(define controller-width 200)
(define controller-indent 10)
(define normal-font "Helvetica")
(define title-font "Helvetica-BoldOblique")
(define highlight-font "Helvetica-Bold")
(define highlight-filter (lambda (x) x))
(define font-size 12)
(define fg-gray 1)
(define bg-gray 0)
(define large-type-factor 2.5)
(define large-type-normal-font "Courier-Bold")
(define large-type-title-font "Courier-Oblique")
(define large-type-highlight-font "Courier-Bold")
(define large-type-highlight-filter string-upcase)
(define large-type-fg-gray 0)
(define large-type-bg-gray 1)

(define (large-type)
  (set! baseline-skip (* baseline-skip large-type-factor))
  (set! font-size (* font-size large-type-factor))
  (set! text-depth (* text-depth large-type-factor))
  (set! registers-width (* registers-width large-type-factor))
  (set! stack-width (* stack-width large-type-factor))
  (set! controller-width (* controller-width large-type-factor))
  (set! controller-indent (* controller-indent large-type-factor))
  (set! normal-font large-type-normal-font)
  (set! title-font large-type-title-font)
  (set! highlight-font large-type-highlight-font)
  (set! highlight-filter large-type-highlight-filter)
  (set! fg-gray large-type-fg-gray)
  (set! bg-gray large-type-bg-gray))

(define (draw-text-in-font window x y string font)
  (in-font window font (lambda () (graphics-draw-text window x y string))))

(define (in-font window font thunk)
  (graphics-operation window 'draw-postscript "gsave")
  (with-values (lambda () (graphics-coordinate-limits window))
    (lambda (x0 y0 x1 y1)
      (with-values (lambda () (graphics-device-coordinate-limits window))
        (lambda (dx0 dy0 dx1 dy1)
          (graphics-set-coordinate-limits window dx0 dy0 dx1 dy1)))
      (graphics-operation window 'draw-postscript
                          (string-append "/" font 
                                         " findfont "
                                         (object->string font-size)
                                         " scalefont setfont"))
      (graphics-set-coordinate-limits window x0 y0 x1 y1)))
  (thunk)
  (graphics-operation window 'draw-postscript "grestore"))

(define (position element lst)
  (define (iter l count)
    (cond ((null? l)
           (error "Position: element not in list" element lst))
          ((eq? element (car l))
           count)
          (else
           (iter (cdr l) (1+ count)))))
  (iter lst 0))

(define (make-registers-window machine)
  (let ((names (remote-get machine '*registers*)))
    (let ((height (+ (* baseline-skip (1+ (length names)))
                     text-depth)))
      (let ((win (make-graphics-device schematik-style-graphics-device-type
                                       'points registers-width height)))
        (graphics-enable-buffering win)
        (graphics-operation win 'set-background-gray fg-gray)
        (graphics-operation win 'set-foreground-gray bg-gray)
        (with-values (lambda () (graphics-coordinate-limits win))
          (lambda (x0 y0 x1 y1)
            (with-values (lambda () (graphics-device-coordinate-limits win))
              (lambda (dx0 dy0 dx1 dy1)
                (graphics-set-coordinate-limits win dx0 dy0 dx1 dy1)))
            (graphics-operation win 'draw-postscript
                                (string-append "/" normal-font 
                                               " findfont "
                                               (object->string font-size)
                                               " scalefont setfont"))
            (graphics-set-coordinate-limits win x0 y0 x1 y1)))
        (graphics-clear win)
        (graphics-set-coordinate-limits win
                                        0 (- (/ (- text-depth) baseline-skip)
                                             1)
                                        registers-width (length names))
        (mapc (lambda (r)
                (draw-register win (position r names) r
                               (ready-for-printing (remote-fetch machine r)
                                                   machine)))
              names)
        (draw-text-in-font win 0 -1 (string-append
                                     (symbol->string (remote-get machine
                                                                 '*name*))
                                     " registers")
                           title-font)
        (graphics-flush win)
        win))))

(define (draw-register window position name value)
  (graphics-draw-text window 0 position
                      (string-append (symbol->string name)
                                     ": "
                                     (object->string value))))

(define (update-registers-window machine name old new)
  (let ((window (remote-get machine '*registers-window*))
        (position (position name (remote-get machine '*registers*))))
    (graphics-bind-drawing-mode window 0 
                                (lambda () (draw-register window position
                                                          name 
                                                          (ready-for-printing
                                                           old
                                                           machine))))
    (draw-register window position name (ready-for-printing new machine))
    (graphics-flush window)))

(define (make-stack-window machine)
  (let ((height (+ (* baseline-skip (1+ max-stack-depth))
                   text-depth)))
    (let ((win (make-graphics-device schematik-style-graphics-device-type
                                     'points stack-width height)))
      (graphics-enable-buffering win)
      (graphics-operation win 'set-background-gray fg-gray)
      (graphics-operation win 'set-foreground-gray bg-gray)
      (with-values (lambda () (graphics-coordinate-limits win))
        (lambda (x0 y0 x1 y1)
          (with-values (lambda () (graphics-device-coordinate-limits win))
            (lambda (dx0 dy0 dx1 dy1)
              (graphics-set-coordinate-limits win dx0 dy0 dx1 dy1)))
          (graphics-operation win 'draw-postscript
                              (string-append "/" normal-font 
                                             " findfont "
                                             (object->string font-size)
                                             " scalefont setfont"))
          (graphics-set-coordinate-limits win x0 y0 x1 y1)))
      (graphics-clear win)
      (graphics-set-coordinate-limits win
                                      0 (- (/ (- text-depth) baseline-skip) 1)
                                      stack-width max-stack-depth)
      (draw-text-in-font win 0 -1 (string-append
                                   (symbol->string (remote-get machine
                                                               '*name*))
                                   " stack")
                         title-font)
      (graphics-flush win)
      win)))

(define (update-stack-window-initialize machine)
  (let ((win (remote-get machine '*stack-window*)))
    (graphics-clear win)
    (draw-text-in-font win 0 -1 (string-append
                                 (symbol->string (remote-get machine
                                                             '*name*))
                                 " stack")
                       title-font)
    (graphics-flush win)))

(define (update-stack-window-push machine position item)
  (let ((window (remote-get machine '*stack-window*)))
    (graphics-draw-text window
                        0 position
                        (object->string (ready-for-printing item machine)))
    (graphics-flush window)))

(define (update-stack-window-pop machine position item)
  (graphics-bind-drawing-mode (remote-get machine '*stack-window*) 0
                              (lambda ()
                                (update-stack-window-push machine
                                                          position item))))

(define (make-controller-window machine)
  (let ((controller (remote-get machine '*controller*)))
    (let ((lines (1+ (length controller))))
      (let ((height (+ (* baseline-skip lines)
                       text-depth)))
        (let ((win (make-graphics-device schematik-style-graphics-device-type
                                         'points controller-width height)))
          (graphics-enable-buffering win)
          (graphics-operation win 'set-background-gray fg-gray)
          (graphics-operation win 'set-foreground-gray bg-gray)
          (with-values (lambda () (graphics-coordinate-limits win))
            (lambda (x0 y0 x1 y1)
              (with-values (lambda () (graphics-device-coordinate-limits win))
                (lambda (dx0 dy0 dx1 dy1)
                  (graphics-set-coordinate-limits win dx0 dy0 dx1 dy1)))
              (graphics-operation win 'draw-postscript
                                  (string-append "/" normal-font 
                                                 " findfont "
                                                 (object->string font-size)
                                                 " scalefont setfont"))
              (graphics-set-coordinate-limits win x0 y0 x1 y1)))
          (graphics-clear win)
          (graphics-set-coordinate-limits win
                                          0 (+ lines
                                               (/ text-depth baseline-skip))
                                          registers-width 0)
          (mapc (lambda (i)
                  (draw-instruction win (1+ (position i controller)) i))
                controller)
          (draw-text-in-font win 0 lines (string-append
                                          (symbol->string (remote-get machine
                                                                      '*name*))
                                          " controller")
                             title-font)
          (graphics-flush win)
          win)))))

(define (update-controller-window machine old-pc new-pc)
  (let ((win (remote-get machine '*controller-window*))
        (pc-positions (remote-get machine '*pc-positions*))
        (controller (remote-get machine '*controller*)))
    (if old-pc
        (let ((pos (nth (-1+ (length old-pc)) pc-positions)))
          (in-font win highlight-font
                   (lambda ()
                     (graphics-bind-drawing-mode 
                      win 0
                      (lambda ()
                        (draw-instruction win pos
                                          (nth (-1+ pos) controller)
                                          highlight-filter)))))
          (draw-instruction win pos
                            (nth (-1+ pos) controller))))
    (if new-pc
        (let ((pos (nth (-1+ (length new-pc)) pc-positions)))
          (graphics-bind-drawing-mode win 0
                                      (lambda ()
                                        (draw-instruction win pos
                                                          (nth (-1+ pos)
                                                               controller))))
          (in-font win highlight-font
                   (lambda ()
                     (draw-instruction win pos
                                       (nth (-1+ pos) controller)
                                       highlight-filter)))))
    (graphics-flush win)))

(define (draw-instruction win pos op . hf)
  (let ((s (object->string op)))
    (let ((s (if (null? hf)
                 s
                 ((car hf) s))))
      (graphics-draw-text win (if (label? op) 0 controller-indent) pos 
                          s))))

(define (ready-for-printing x machine)
  (if (null? x)
      x
      (let ((entry (assq x
                         (mapcar (lambda (l) (cons (remote-get machine l) l))
                                 (remote-get machine '*labels*)))))
        (if (null? entry)
            x
            (string->symbol (string-append "["
                                           (symbol->string (cdr entry)) "]"))))))

;;;*** From here on is stuff for machine-checking (not in the book)

;;INSTRUCTION SYNTAX
;get type of any instruction
(define (instruction-type inst) (car inst))

;get register of assign, save, restore, fetch
(define (expr-reg expr)
  (cadr expr))

(define (fetch? expr)
  (if (atom? expr) nil (eq? (car expr) 'fetch)))

(define (constant? expr)
  (if (atom? expr) t (eq? (car expr) 'quote)))

(define (goto-target goto)
  (cadr goto))

(define (branch-target branch)
  (caddr branch))

(define (test branch)
  (cadr branch))

(define (action perform)
  (cadr perform))

(define (source assign)
  (caddr assign))

(define (operation-args op)
  (cdr op))

(define (check-instruction machine inst)
  (let ((type (instruction-type inst)))
    (cond ((or (eq? type 'save) (eq? type 'restore))   ; ok
           (check-declared-register machine inst))
          ((eq? type 'assign)                          ; ok
           (check-declared-register machine inst)
           (let ((source (source inst)))
             (cond ((constant? source) (check-constant machine source))
                   ((fetch? source) (check-declared-register machine source))
                   (else (check-operation machine source)))))
          ((eq? type 'perform) (check-operation machine (action inst)))  ; ok
          ((eq? type 'goto)                            ; ok
           (let ((target (goto-target inst)))
             (if (fetch? target)
                 (check-declared-register machine target)
                 (check-declared-label machine target))))
          ((eq? type 'branch)                          ; ok
           (check-operation machine (test inst))
           (check-declared-label machine (branch-target inst)))
          (else (error "Invalid instruction type" inst)))))  ; ok

(define (check-declared-register machine expr)         ; ok
  (let ((reg (expr-reg expr)))
    (if (not (declared-register? reg machine))
        (error "Undeclared register" reg expr))))

(define (declared-register? reg machine)               ; ok
  (memq reg (remote-get machine '*registers*)))

(define (check-declared-label machine lab)             ; ok
  (if (not (memq lab (remote-get machine '*labels*)))
      (error "Undeclared label" lab)))

;Check an operation (test, action, or function)
(define (check-operation machine op)                   ; ok
  (mapc (lambda (arg)
          (cond ((fetch? arg) (check-declared-register machine arg))
                ((constant? arg) (check-constant machine arg))
                (else (error "Bad argument in test, action, or function" op))))
        (operation-args op)))

(define (check-constant machine c)                     ; ok
  (if (atom? c)
      (if (declared-register? c machine)
          (error "Register not inside a FETCH" c))))
