;;; This file is excerpted from mceval.scm, which was adapted from the
;;; MIT file PS8-EVAL.SCM, which bears the following copyright notice:
;;; Copyright (c) 1990 Massachusetts Institute of Technology
;;; 
;;; This material was developed by the Scheme project at the 
;;; Massachusetts Institute of Technology, Department of Electrical 
;;; Engineering and Computer Science.  Permission to copy this 
;;; material, to redistribute it, and to use it for any 
;;; non-commercial purpose is granted, subject to the following 
;;; restrictions and understandings.
;;;
;;; 1. Any copy made of this material must include this copyright 
;;; notice in full.
;;; 
;;; 2. Users of this material agree to make their best efforts (a) 
;;; to return to the MIT Scheme project any improvements or 
;;; extensions that they make, so that these may be included in 
;;; future releases; and (b) to inform MIT of noteworthy uses of 
;;; this material.
;;;
;;; 3. All materials developed as a consequence of the use of this
;;; material shall duly acknowledge such use, in accordance with 
;;; the usual standards of acknowledging credit in academic 
;;; research.
;;;
;;; 4. MIT has made no warrantee or representation that this 
;;; material (including the operation of software contained 
;;; therein) will be error-free, and MIT is under no obligation to 
;;; provide any services, by way of maintenance, update, or 
;;; otherwise.
;;;
;;; 5. In conjunction with products arising from the use of this 
;;; material, there shall be no use of the name of the 
;;; Massachusetts Institute of Technology nor of any adaptation 
;;; thereof in any advertising, promotional, or sales literature 
;;; without prior written consent from MIT in each case. 

;;; These are the portions of mceval.scm you may have to change.
;;; In addition to changing some or all of these procedures to
;;; produce thunks and to "force" them, you will need to add a
;;; constructor and selectors for thunks and a procedure for
;;; "forcing" a value if it is a thunk (and then continuing to
;;; force that result if it is again a thunk).

;;; We define a special way of printing out the values of 
;;; computations here, which avoids printing the environment
;;; part of a compound procedure, since the latter is a very
;;; long (or even circular) list.  This is analogous to the
;;; user-print procedure in the book's version, the difference
;;; being that we are not doing the actual printing, just
;;; transforming the value into what should be printed.  In the
;;; normal scheme the special #[compound-procedure ....] notation
;;; is used to solve the same problem.

;;; This is the routine you would change to handle other kinds of
;;; values specially; for example, if you introduce thunks, you
;;; would "force" them here to cause the actual result of rather
;;; than evaluation to be printed out just the "promise".

(define (ready-value-for-printing value)
  (if (compound-procedure? value)
      (list 'compound-procedure
            (parameters value)
            (procedure-body value)
            'procedure-env)
      value))

;;; THE CORE OF THE EVALUATOR -- from section 4.1.1
;;; You will need to modify mini-eval or mini-apply (which are
;;; what we are calling eval and apply, to distinguish them from
;;; the eval and apply built into the "real" scheme) to make sure
;;; that the operator of a procedure application is forced if it
;;; is a thunk.

(define (mini-eval exp env)
  (cond ((self-evaluating? exp) exp)
        ((quoted? exp) (text-of-quotation exp))
        ((variable? exp) (lookup-variable-value exp env))
        ((definition? exp) (eval-definition exp env))
        ((assignment? exp) (eval-assignment exp env))
        ((lambda? exp) (make-procedure exp env))
        ((conditional? exp) (eval-cond (clauses exp) env))
        ((application? exp)
         (mini-apply (mini-eval (operator exp) env)
                     (list-of-values (operands exp) env)))
        (else (error "Unknown expression type -- MINI-EVAL" exp))))

(define (mini-apply procedure arguments)
  (cond ((primitive-procedure? procedure)
         (apply-primitive-procedure procedure arguments))
        ((compound-procedure? procedure)
         (eval-sequence (procedure-body procedure)
                        (extend-environment
                         (parameters procedure)
                         arguments
                         (procedure-environment procedure))))
        (else
         (error "Unknown procedure type -- MINI-APPLY" procedure))))


;;; This procedure is the one that evaluates the arguments of a
;;; procedure application.  If you wanted to do something other
;;; than evaluate them, this might be a good place to do it
;;; (though perhaps the name should change).

(define (list-of-values exps env)
  (cond ((no-operands? exps) '())
        (else (cons (mini-eval (first-operand exps) env)
                    (list-of-values (rest-operands exps)
                                    env)))))

;;; eval-cond searches for a clause with a test that evaluates
;;; true; why would you need to change this if you introduce
;;; thunks?

(define (eval-cond clist env)
  (cond ((no-clauses? clist) false)
        ((else-clause? (first-clause clist))
         (eval-sequence (actions (first-clause clist))
                        env))
        ((true? (mini-eval (predicate (first-clause clist)) env))
         (eval-sequence (actions (first-clause clist))
                        env))
        (else (eval-cond (rest-clauses clist) env))))

;;; APPLYING PRIMITIVE PROCEDURES

;;; The mechanism for applying primitive procedures is somewhat
;;; different from the one given in section 4.1.4 of the text.
;;; The modification is as suggested in exercise 4.8 of the text.
;;; Instead of representing a primitive as a list
;;;  (primitive <name-of-primitive>) we represent it as
;;;  (primitive <Scheme procedure to apply>)

;;; To apply a primitive procedure, we ask the underlying Scheme
;;; system to perform the application.  (Of course, an
;;; implementation on a low-level machine would perform the
;;; application in some other way.)

;;; If you want to treat some primitives specially (for example,
;;; if you make most primitives force any thunk arguments, but you
;;; don't want cons to do that) the simplest (though not cleanest)
;;; thing to do would be to just check whether (primitive-id p) is
;;; the procedure you want to treat specially (the actual
;;; procedure in the parent scheme, not its name nor the
;;; "primitive procedure" object of the metacircular scheme).

(define (apply-primitive-procedure p args)
  (apply (primitive-id p) args))
