;;; -*- Scheme -*-

;;; It contains the metacircular evaluator, as described in section 4.1 of the
;;; text, with a few minor modifications.  In particular, the dispatch on 
;;; expression type in EVAL is done by a dispatch table.

;;; We initialize the global environment by snarfing a few primitives from the
;;; underlying scheme system.  This is different from the treatment of
;;; primitives in the book.  (See the comments below under "applying primitive
;;; procedures".)  If you want to add more primitives to your evaluator, just
;;; modify the list PRIMITIVE-PROCEDURE-NAMES to include more Scheme
;;; primitives.

;;; The actual structure of the environment is determined by the constructor
;;; EXTEND-ENVIRONMENT which is listed below together with the code that
;;; manipulates environments.

(define primitive-procedure-names
  '(+ - * / = < > 1+ -1+ cons car cdr pair? eq? null? procedure? not
    user-write-line user-display user-load))

(define (setup-environment)
  (let ((initial-env
	 (extend-environment primitive-procedure-names
			     (map (lambda (pname)
				    (list 'primitive
					  (eval pname
						user-initial-environment)))
				  primitive-procedure-names)
			     '())))
    (define-variable! 'false false initial-env)
    (define-variable! 'true true initial-env)
    initial-env))

(define the-global-environment '())

;;; INITIALIZATION AND DRIVER LOOP

;;; To start the metacircular evaluator, call INITIALIZE-EVALUATOR.  This
;;; initializes the global environment, and starts the DRIVER-LOOP.  Use
;;; INITIALIZE-EVALUATOR instead of DRIVER-LOOP if you want to erase any
;;; definitions you have accumulated and start fresh with a clean global
;;; environment


(define (initialize-evaluator)
  (set! the-global-environment (setup-environment))
  (driver-loop))

;;; The driver loop reads an expression, evaluates it in the global
;;; environment, and prints the result.  Note that the driver uses a prompt of
;;; "MC-EVAL==> " to help you avoid confusing typing to the metacircular
;;; evaluator with typing to the underlying SCHEME interpreter.  (Also, we have
;;; called our evaluator procedures MINI-EVAL and MINI-APPLY to help you avoid
;;; confusing them with the underlying Scheme's EVAL and APPLY.)

;;; When/If your interaction with the evaluator bombs out in an error, you
;;; should restart it by calling DRIVER-LOOP again.

(define (driver-loop)
  (newline)
  (display "MC-EVAL==> ")
  (user-write-line (mini-eval (read) the-global-environment))
  (driver-loop))


;;; We use a special WRITE-LINE here, which avoids printing the environment part of
;;; a compound procedure, since the latter is a very long (or even circular)
;;; list.  These will be mapped to WRITE-LINE and DISPLAY. 

(define (user-write-line object)
  (cond ((compound-procedure? object)
         (write-line (list 'compound-procedure
			   (parameters object)
			   (procedure-body object)
			   'procedure-env)))
        (else (write-line object))))

(define (user-display object)
  (cond ((compound-procedure? object)
         (display (list 'compound-procedure
			(parameters object)
			(procedure-body object)
			'procedure-env)))
        (else (display object))))



;;; We also need a loader for files of user code.

(define (user-load filename)
  (let ((channel (open-input-file filename)))
    (define (loop)
  
    (let ((exp (read channel)))
	(if (eof-object? exp)
	    (begin
	      (close-input-port channel)
	      'done)
	    (search-eval exp
			 the-global-environment
			 (lambda (val fail)
			   (user-write-line val)
			   (loop))
			 (lambda ()
			   (user-write-line 'failed)
			   (loop))))))
    (loop)))

;;;This is to keep the Scheme printer from going into an infinite loop
;;;if you try to print a circular data structure, such as an environment
(set! *unparser-list-depth-limit* 10)
(set! *unparser-list-breadth-limit* 10)

;;; THE CORE OF THE EVALUATOR

(define (mini-eval exp env)
  (cond ((self-evaluating? exp) exp)
	((variable? exp) (lookup-variable-value exp env))
	(else
	 (let ((evaluation-procedure
		(lookup-in-syntax-table (car exp))))
	   (if evaluation-procedure
	       (evaluation-procedure exp env)
	       (mini-apply (mini-eval (operator exp) env)
			   (list-of-values (operands exp) env)))))))


(define (mini-apply procedure arguments)
  (cond ((primitive-procedure? procedure)
         (apply-primitive-procedure procedure arguments))
        ((compound-procedure? procedure)
         (eval-sequence (procedure-body procedure)
                        (extend-environment
                         (parameters procedure)
                         arguments
                         (procedure-environment procedure))))
        (else
         (error "Unknown procedure type -- APPLY" procedure))))


(define (list-of-values exps env)
  (cond ((no-operands? exps) '())
        (else (cons (mini-eval (first-operand exps) env)
                    (list-of-values (rest-operands exps)
                                    env)))))

(define (eval-cond clist env)
  (cond ((no-clauses? clist) false)
        ((else-clause? (first-clause clist))
         (eval-sequence (actions (first-clause clist))
                        env))
        ((mini-eval (predicate (first-clause clist)) env)
         (eval-sequence (actions (first-clause clist))
                        env))
        (else (eval-cond (rest-clauses clist) env))))

(define (eval-sequence exps env)
  (cond ((last-exp? exps) (mini-eval (first-exp exps) env))
        (else (mini-eval (first-exp exps) env)
              (eval-sequence (rest-exps exps) env))))


(define (eval-assignment exp env)
  (let ((new-value (mini-eval (assignment-value exp) env)))
    (set-variable-value! (assignment-variable exp)
                         new-value
                         env)
    new-value))

(define (eval-definition exp env)
  (define-variable! (definition-variable exp)
                    (mini-eval (definition-value exp) env)
                    env)
  (definition-variable exp))

;;; Syntax of the language -- from section 4.1.2

(define (self-evaluating? exp)
  (or (number? exp) (string? exp)))

(define (quoted? exp)
  (and (pair? exp)
       (eq? (car exp) 'quote)))

(define (text-of-quotation exp) (cadr exp))

(define (variable? exp) (symbol? exp))

(define (assignment? exp)
  (and (pair? exp)
       (eq? (car exp) 'set!)))

(define (assignment-variable exp) (cadr exp))

(define (assignment-value exp) (caddr exp))

(define (definition? exp)
  (and (pair? exp)
       (eq? (car exp) 'define)))

(define (definition-variable exp)
  (if (variable? (cadr exp))
      (cadr exp)
      (caadr exp)))

(define (definition-value exp) 
  (if (variable? (cadr exp))
      (caddr exp)
      (cons 'lambda
	    (cons (cdadr exp)    
		  (cddr exp)))))


(define (lambda? exp)
  (and (pair? exp)
       (eq? (car exp) 'lambda)))


(define (conditional? exp)
  (and (pair? exp)
       (eq? (car exp) 'cond)))


(define (clauses exp) (cdr exp))

(define (no-clauses? clauses) (null? clauses))

(define (first-clause clauses) (car clauses))

(define (rest-clauses clauses) (cdr clauses))

(define (predicate clause) (car clause))

(define (actions clause) (cdr clause))

(define (else-clause? clause)
  (eq? (predicate clause) 'else))

(define (last-exp? seq) (null? (cdr seq)))

(define (first-exp seq) (car seq))

(define (rest-exps seq) (cdr seq))

(define (application? exp) true)

(define (operator app) (car app))

(define (operands app) (cdr app))

(define (no-operands? args) (null? args))

(define (first-operand args) (car args))

(define (rest-operands args) (cdr args))

(define (make-procedure lambda-exp env)
  (list 'procedure lambda-exp env))

(define (compound-procedure? proc)
  (and (pair? proc)
       (eq? (car proc) 'procedure)))


(define (parameters proc) (cadr (cadr proc)))

(define (procedure-body proc) (cddr (cadr proc)))

(define (procedure-environment proc) (caddr proc))


;;; APPLYING PRIMITIVE PROCEDURES

;;; The mechanism for applying primitive procedures is somewhat different from
;;; the one given in section 4.1.4 of the text.  The modification is as
;;; suggested in exercise 4.8 of the text.  Instead of representing a primitive
;;; as a list
;;;  (primitive <name-of-primitive>)
;;; we represent it as
;;;  (primitive <Scheme procedure to apply>)

(define (primitive-procedure? proc)
  (and (pair? proc)
       (eq? (car proc) 'primitive)))

(define (primitive-id proc) (cadr proc))

;;; To apply a primitive procedure, we ask the underlying Scheme system to
;;; perform the application.  (Of course, an implementation on a low-level
;;; machine would perform the application in some other way.)

(define (apply-primitive-procedure p args)
  (apply (primitive-id p) args))


(define (procedure? p)
  (or (primitive-procedure? p) (compound-procedure? p)))

(define *syntax-table* '())

(define (add-syntax! keyword procedure)
  (let ((vcell (assq keyword *syntax-table*)))
    (if vcell
	(set-cdr! vcell procedure)
	(set! *syntax-table*
	      (cons (cons keyword procedure)
		    *syntax-table*)))
    'done))

(define (lookup-in-syntax-table keyword)
  (let ((vcell (assq keyword *syntax-table*)))
    (if vcell
	(cdr vcell)
	false)))


(add-syntax! 'quote
	     (lambda (exp env)
	       (text-of-quotation exp)))
(add-syntax! 'define eval-definition)
(add-syntax! 'set! eval-assignment)
(add-syntax! 'lambda make-procedure)
(add-syntax! 'cond
	     (lambda (exp env)
	       (eval-cond (clauses exp) env)))

;;; ENVIRONMENTS -- implemented as backbone and ribs, 
;;;  with 2-rail-lookup.

(define (lookup-variable-value var env)
  (let ((b (binding-in-env var env)))
    (if (found-binding? b)
        (binding-value b)
        (error "Unbound variable" var))))

(define (binding-in-env var env)
  (if (no-more-frames? env)
      no-binding
      (let ((b (binding-in-frame var (first-frame env))))
        (if (found-binding? b)
            b
            (binding-in-env var (rest-frames env))))))

(define (extend-environment variables values base-env)
  (adjoin-frame (make-frame variables values) base-env))

(define (set-variable-value! var val env)
  (let ((b (binding-in-env var env)))
    (if (found-binding? b)
        (set-binding-value! b val)
        (error "Unbound variable" var))))

(define (define-variable! var val env)
  (let ((b (binding-in-frame var (first-frame env))))
    (if (found-binding? b)
        (set-binding-value! b val)
	(set-first-frame!
	 env
	 (adjoin-binding var val (first-frame env))))))



(define (first-frame env) (car env))

(define (rest-frames env) (cdr env))

(define (no-more-frames? env) (null? env))

(define (adjoin-frame frame env) (cons frame env))

(define (set-first-frame! env new-frame)
  (set-car! env new-frame))


(define (make-frame variables values)
  (cond ((not (= (length variables) (length values)))
	 (error "Wrong number of arguments supplied" variables values))
        (else (cons variables values))))

(define (adjoin-binding var val frame)
  (cons (cons var (car frame))
	(cons val (cdr frame))))

(define (binding-in-frame var frame)
  (2-rail-lookup var (car frame) (cdr frame)))


(define (2-rail-lookup key keys values)
  (cond ((null? keys) false)
	((eq? key (car keys)) values)
	(else (2-rail-lookup key (cdr keys) (cdr values)))))

(define (found-binding? b)
  (not (eq? b no-binding)))

(define no-binding false)


(define (binding-value binding)
  (car binding))

(define (set-binding-value! binding value)
  (set-car! binding value))

