;;;;               This is the file SEARCH.SCM

;;; It contains the metacircular evaluator, as described in section 4.1 of the
;;; text, with a few minor modifications, and arranged to allow the use of the 
;;; non-deterministic choice operator, AMB.  We call Scheme augmented by AMB 
;;; AMBScheme.

;;; You should just load this file into Scheme without editing it.  The new
;;; procedures that you will need to modify in order to do the problem set have
;;; been copied into in a separate file for your convenience.


;;; INITIALIZATION AND DRIVER LOOP

;;; To start the metacircular evaluator, call INITIALIZE-EVALUATOR.  This
;;; initializes the global environment, and starts the DRIVER-LOOP.  Use
;;; INITIALIZE-EVALUATOR instead of DRIVER-LOOP if you want to erase any
;;; definitions you have accumulated and start fresh with a clean global
;;; environment

(define the-global-environment '())

(define (initialize-evaluator)
  (set! the-global-environment (setup-environment))
  (driver-loop))

;;; The driver loop reads an expression, evaluates it in the global
;;; environment, and prints the result.  Note that the driver uses a prompt of
;;; "SEARCH-EVAL==> " to help you avoid confusing typing to the metacircular
;;; evaluator with typing to the underlying SCHEME interpreter.  (Also, we have
;;; called our evaluator procedures SEARCH-EVAL and SEARCH-APPLY to help you avoid
;;; confusing them with Scheme's EVAL and APPLY.)

;;; When/If your interaction with the evaluator bombs out in an error, you
;;; should restart it by calling DRIVER-LOOP again.

(define (driver-loop)
  (read-search-print (exhausted-next-generator '(driver-loop))))

(define (exhausted-next-generator expression)
  (define exhausted-generator
    (lambda ()
      (newline)
      (display "There are no more values of ")
      (display expression)
      (read-search-print exhausted-generator)))
  exhausted-generator)

(define (read-search-print next-generator)
  (newline)
  (display "SEARCH-EVAL==>> ")
  (let ((exp (read-from-keyboard)))
    (if (equal? exp '(next))
	(next-generator)
	(search-eval exp
		     the-global-environment
		     (lambda (value new-next-generator)
		       (user-write-line value)
		       (read-search-print new-next-generator))
		     (exhausted-next-generator exp)))))


;;; The search evaluator will need a few special I/O routines.

;;; We use a special WRITE-LINE here, which avoids printing the environment part of
;;; a compound procedure, since the latter is a very long (or even circular)
;;; list.  These will be mapped to WRITE-LINE and DISPLAY in AMBScheme. 

(define (user-write-line object)
  (cond ((compound-procedure? object)
         (write-line (list 'compound-procedure
			   (parameters object)
			   (procedure-body object)
			   'procedure-env)))
        (else (write-line object))))

(define (user-display object)
  (cond ((compound-procedure? object)
         (display (list 'compound-procedure
			(parameters object)
			(procedure-body object)
			'procedure-env)))
        (else (display object))))



;;; We also need a loader for files of user code.

(define (user-load filename)
  (let ((channel (open-input-file filename)))
    (define (loop)
      (let ((exp (read channel)))
	(if (eof-object? exp)
	    (begin
	      (close-input-port channel)
	      'done)
	    (search-eval exp
			 the-global-environment
			 (lambda (val fail)
			   (user-write-line val)
			   (loop))
			 (lambda ()
			   (user-write-line 'failed)
			   (loop))))))
    (loop)))


;;;This is to keep the Scheme printer from going into an infinite loop
;;;if you try to print a circular data structure, such as an environment
(set! *unparser-list-depth-limit* 10)
(set! *unparser-list-breadth-limit* 10)

;;; SETTING UP THE ENVIRONMENT

;;; We initialize the global environment by snarfing a few primitives
;;; from the underlying scheme system.  This is different from the
;;; treatment of primitives in the book.  (See the comments below
;;; under "applying primitive procedures".)  If you want to add more
;;; primitives to your evaluator, just modify the list
;;; PRIMITIVE-PROCEDURE-NAMES to include more Scheme primitives.  You
;;; can also use IMPORT-PRIMITIVES to gobble primitive procedures from
;;; the underlying Scheme system.

;;; The actual structure of the environment is determined by the constructor
;;; EXTEND-ENVIRONMENT which is listed below together with the code that
;;; manipulates environments.

(define primitive-procedure-names
  '(+ - * / = < > 1+ -1+ quotient sqrt expt round abs gcd max min
    list cons car cdr 
    caar cadr cdar cddr caaar caadr cadar caddr cdaar cdadr cddar cdddr
    eq? equal? null? number? symbol? not
    assoc assq memq member length append
    (write-line user-write-line)
    (display user-display)
    newline
    (load user-load)))

(define (setup-environment)  
  (let ((initial-env
         (extend-environment '() '() the-empty-environment)))
    (define (import-primitives procedure-names)
      (map (lambda (pname)
	     (define-variable!
	       (if (pair? pname) (car pname) pname)
	       (list 'primitive
		     (eval-in-initial-env
		      (if (pair? pname) (cadr pname) pname)))
	       initial-env)
	     pname)
	   procedure-names))
    (import-primitives primitive-procedure-names)
    (define-variable! 'false false initial-env)
    (define-variable! 'true true initial-env)
    (define-variable! 'import-primitives
      (list 'primitive import-primitives)
      initial-env)
    initial-env))

;;; THE CORE OF THE EVALUATOR -- modified from section 4.1.1

;;; FAIL is a procedure of zero arguments.
;;; VALUE-EATER is a procedure of two arguments, 
;;;   the value and the failure continuation

(define (search-eval exp env value-eater next-generator)
  (cond ((self-evaluating? exp)
         (value-eater exp next-generator))
        ((quoted? exp)
         (value-eater (text-of-quotation exp) next-generator))
        ((variable? exp)
         (value-eater (lookup-variable-value exp env) next-generator))
        ((definition? exp)
         (eval-definition exp env value-eater next-generator))
        ((assignment? exp)
         (eval-assignment exp env value-eater next-generator))
	((set-car? exp)
         (eval-set-car exp env value-eater next-generator))
	((set-cdr? exp)
         (eval-set-cdr exp env value-eater next-generator))
        ((lambda? exp)
         (value-eater (make-procedure exp env) next-generator))		  
        ((conditional? exp)
         (eval-cond (clauses exp) env value-eater next-generator))
        ((let? exp)
         (eval-let exp env value-eater next-generator))
        ((amb? exp)
         (eval-amb (operands exp) env value-eater next-generator))
        ((all? exp)
         (eval-all (operands exp) env value-eater next-generator))
        ((application? exp)
         (search-eval (operator exp) 
		      env 
		      (lambda (proc next-generator1)
			(list-of-values (operands exp) 
					env
					(lambda (args next-generator2)
					  (search-apply 
					    proc args value-eater next-generator2))
					next-generator1)) 
		      next-generator))
        (else (error "Unknown expression type -- EVAL" exp))))

(define (search-apply procedure arguments value-eater next-generator)
  (cond ((primitive-procedure? procedure)
         (value-eater (apply-primitive-procedure procedure arguments)
		      next-generator))
        ((compound-procedure? procedure)
         (eval-sequence (procedure-body procedure)
                        (extend-environment (parameters procedure)
                                            arguments
                                            (procedure-environment procedure))
                        value-eater
                        next-generator))
        (else
         (error "Unknown procedure type -- APPLY" procedure))))

(define (list-of-values exps env value-eater next-generator)
  (cond ((no-operands? exps) (value-eater '() next-generator))
        (else
         (search-eval (first-operand exps) 
		      env
		      (lambda (arg next-generator1)
			(list-of-values (rest-operands exps)
					env 
					(lambda (args next-generator2)
					  (value-eater (cons arg args) next-generator2))
					next-generator1))
		      next-generator))))

(define (eval-cond clist env value-eater next-generator)
  (cond ((no-clauses? clist)
         (value-eater false next-generator))
        ((else-clause? (first-clause clist))
         (eval-sequence (actions (first-clause clist))
                        env
                        value-eater
                        next-generator))
        (else
         (search-eval (predicate (first-clause clist)) 
		      env
		      (lambda (pred next-generator1)
			(if (true? pred)
			    (eval-sequence (actions (first-clause clist))
					   env
					   value-eater
					   next-generator1)
			    (eval-cond (rest-clauses clist) env
				       value-eater
				       next-generator1)))
		      next-generator))))


(define (eval-let exp env value-eater next-generator)
  (search-eval (cons (cons 'lambda
			   (cons (map car (let-bindings exp))
				 (let-body exp)))
		     (map cadr (let-bindings exp)))
	       env
	       value-eater
	       next-generator))

(define (eval-amb exps env value-eater next-generator)
  (if (no-operands? exps)
      (next-generator)
      (search-eval (first-operand exps) 
		   env
		   value-eater
		   (lambda ()
		     (eval-amb (cdr exps) env value-eater next-generator)))))

(define (eval-sequence exps env value-eater next-generator)
  (cond ((last-exp? exps)
         (search-eval (first-exp exps) env value-eater next-generator))
        (else
         (search-eval (first-exp exps) 
		      env
		      (lambda (value next-generator1)
			(eval-sequence (rest-exps exps) 
				       env
				       value-eater
				       next-generator1))
		      next-generator))))

(define (eval-assignment exp env value-eater next-generator)
  (search-eval (assignment-value exp) 
	       env
	       (lambda (value next-generator1)
		 (let ((oldvalue (lookup-variable-value 
				   (assignment-variable exp) env)))
		   (set-variable-value! (assignment-variable exp)
					value
					env)
		   (value-eater value 
				(lambda ()
				  (set-variable-value! (assignment-variable exp)
						       oldvalue
						       env)
				  (next-generator1)))))
	       next-generator))

(define (eval-set-car exp env value-eater next-generator)
    (search-eval (cell-part exp)
		 env
		 (lambda (cell next-generator1)
		   (search-eval (value-part exp)
				env
				(lambda (new-value next-generator3)
				  (let ((old-value (car cell)))
				    (set-car! cell new-value)
				    (value-eater new-value
						 (lambda ()
						   (set-car! cell old-value)
						   (next-generator3)))))
				next-generator1))
		 next-generator))

(define (eval-set-cdr exp env value-eater next-generator)
    (search-eval (cell-part exp)
		 env
		 (lambda (cell next-generator1)
		   (search-eval (value-part exp)
				env
				(lambda (new-value next-generator3)
				  (let ((old-value (cdr cell)))
				    (set-cdr! cell new-value)
				    (value-eater new-value
						 (lambda ()
						   (set-cdr! cell old-value)
						   (next-generator3)))))
				next-generator1))
		 next-generator))

;; Instead of the above and all-values, we should define some more general machinery
;; to let the user-level code get hold of what to do when all of some set of amb's
;; is exhausted.

(define (eval-definition exp env value-eater next-generator)
  (search-eval (definition-value exp) 
	       env
	       (lambda (value next-generator1)
		 (define-variable! (definition-variable exp)
				   value
		   env)
		 (value-eater (definition-variable exp) next-generator1))
	       next-generator))

(define (eval-all seq env value-eater next-generator)
  (if (last-exp? seq)
      (let ((result '()))
        (search-eval (first-exp seq)
		     env
		     (lambda (value next-generator1)
		       (set! result (cons value result))
		       (next-generator1))
		     (lambda ()
		       (value-eater (reverse result) next-generator))))
      (error "Too many forms in ALL-VALUES" seq)))

;;; Syntax of the language -- from section 4.1.2

(define (self-evaluating? exp)
  (if (number? exp)
      true
      (string? exp)))

(define (quoted? exp)
  (and (pair? exp)
       (eq? (car exp) 'quote)))

(define (text-of-quotation exp) (cadr exp))

(define (variable? exp) (symbol? exp))

(define (assignment? exp)
  (and (pair? exp)
       (eq? (car exp) 'set!)))

(define (assignment-variable exp) (cadr exp))

(define (assignment-value exp) (caddr exp))

(define (set-car? exp)
  (and (pair? exp)
       (eq? (car exp) 'set-car!)))

(define (set-cdr? exp)
  (and (pair? exp)
       (eq? (car exp) 'set-cdr!)))

(define (cell-part exp) (cadr exp))

(define (value-part exp) (caddr exp))

(define (definition? exp)
  (and (pair? exp)
       (eq? (car exp) 'define)))

(define (definition-variable exp)
  (if (variable? (cadr exp))
      (cadr exp)
      (caadr exp)))

(define (definition-value exp) 
  (if (variable? (cadr exp))
      (caddr exp)
      (cons 'lambda
            (cons (cdadr exp)    
                  (cddr exp)))))

(define (lambda? exp)
  (and (pair? exp)
       (eq? (car exp) 'lambda)))


(define (conditional? exp)
  (and (pair? exp)
       (eq? (car exp) 'cond)))

(define (clauses exp) (cdr exp))

(define (no-clauses? clauses) (null? clauses))

(define (first-clause clauses) (car clauses))

(define (rest-clauses clauses) (cdr clauses))

(define (predicate clause) (car clause))

(define (actions clause) (cdr clause))

(define (true? x) x)

(define (else-clause? clause)
  (eq? (predicate clause) 'else))




(define (let? exp)
  (and (pair? exp)
       (eq? (car exp) 'let)))

(define (let-bindings exp)
  (cadr exp))

(define (let-body exp)
  (cddr exp))


(define (amb? exp)
  (and (pair? exp)
       (eq? (car exp) 'amb)))

(define (all? exp)
  (and (pair? exp)
       (eq? (car exp) 'all-values)))

(define (last-exp? seq) (null? (cdr seq)))

(define (first-exp seq) (car seq))

(define (rest-exps seq) (cdr seq))

(define (application? exp) (pair? exp))

(define (operator app) (car app))

(define (operands app) (cdr app))

(define (no-operands? args) (null? args))

(define (first-operand args) (car args))

(define (rest-operands args) (cdr args))

(define (make-procedure lambda-exp env)
  (list 'procedure lambda-exp env))

(define (compound-procedure? proc)
  (and (pair? proc)
       (eq? (car proc) 'procedure)))


(define (parameters proc) (cadr (cadr proc)))

(define (procedure-body proc) (cddr (cadr proc)))

(define (procedure-environment proc) (caddr proc))


;;; APPLYING PRIMITIVE PROCEDURES

;;; The mechanism for applying primitive procedures is somewhat different from
;;; the one given in section 4.1.4 of the text.  The modification is as
;;; suggested in exercise 4.8 of the text.  Instead of representing a primitive
;;; as a list
;;;  (primitive <name-of-primitive>)
;;; we represent it as
;;;  (primitive <Scheme procedure to apply>)

(define (primitive-procedure? proc)
  (and (pair? proc)
       (eq? (car proc) 'primitive)))

(define (primitive-id proc) (cadr proc))

;;; To apply a primitive procedure, we ask the underlying Scheme system to
;;; perform the application.  (Of course, an implementation on a low-level
;;; machine would perform the application in some other way.)

(define (apply-primitive-procedure p args)
  (apply (primitive-id p) args))

;;; ENVIRONMENTS -- from section 4.1.3

(define (lookup-variable-value var env)
  (let ((b (binding-in-env var env)))
    (if (found-binding? b)
        (binding-value b)
        (error "Unbound variable" var))))

(define (binding-in-env var env)
  (if (no-more-frames? env)
      no-binding
      (let ((b (binding-in-frame var (first-frame env))))
        (if (found-binding? b)
            b
            (binding-in-env var (rest-frames env))))))

(define (extend-environment variables values base-env)
  (adjoin-frame (make-frame variables values) base-env))

(define (set-variable-value! var val env)
  (let ((b (binding-in-env var env)))
    (if (found-binding? b)
        (set-binding-value! b val)
        (error "Unbound variable" var))))

(define (define-variable! var val env)
  (let ((b (binding-in-frame var (first-frame env))))
    (if (found-binding? b)
        (set-binding-value! b val)
        (set-first-frame!
         env
         (adjoin-binding (make-binding var val)
                         (first-frame env))))))



(define (first-frame env) (car env))

(define (rest-frames env) (cdr env))

(define (no-more-frames? env) (null? env))

(define (adjoin-frame frame env) (cons frame env))

(define (set-first-frame! env new-frame)
  (set-car! env new-frame))

(define the-empty-environment '())

(define (make-frame variables values)
  (cond ((and (null? variables) (null? values)) '())
        ((null? variables)
         (error "Too many values supplied" values))
	((variable? variables)
	 (list (make-binding variables values)))
        ((null? values)
         (error "Too few values supplied" variables))
        (else
         (cons (make-binding (car variables)
                             (car values))
               (make-frame (cdr variables)
                           (cdr values))))))

(define (adjoin-binding binding frame)
  (cons binding frame))

;;;  A reminder:
;;;    (define (assq key bindings)
;;;      (cond ((null? bindings) no-binding)
;;;            ((eq? key (binding-variable (car bindings))) (car bindings))
;;;            (else (assq key (cdr bindings)))))


(define (binding-in-frame var frame)
  (assq var frame))

(define (found-binding? b)
  (not (eq? b no-binding)))

(define no-binding false)

(define (make-binding variable value)
  (cons variable value))

(define (binding-variable binding)
  (car binding))

(define (binding-value binding)
  (cdr binding))

(define (set-binding-value! binding value)
  (set-cdr! binding value))



;;; Definitions for MIT Scheme

(define (eval-in-initial-env form)
  (eval form user-initial-environment))

;;; For use with EDWIN
(define read-from-keyboard read)
