;;; -*- Scheme -*-

(define (lock-thread-mutex mutex)
  (guarantee-thread-mutex mutex lock-thread-mutex)
  (without-interrupts
   (lambda ()
     (let ((thread (current-thread))
	   (owner (thread-mutex/owner mutex)))
       (cond ((not owner)
	      (set-thread-mutex/owner! mutex thread))
	     ((eq? owner thread)
	      (signal-thread-deadlock thread "lock thread mutex"
				      lock-thread-mutex mutex))
	     ((thread-dead? owner)
	      (%forcibly-unlock-mutex mutex thread))
	     (else
	      (%lock-thread-mutex mutex thread)))))))

(define-integrable (%lock-thread-mutex mutex thread)
  (ring/enqueue (thread-mutex/waiting-threads mutex) thread)
  (%await-thread-mutex mutex thread))

(define-integrable (%await-thread-mutex mutex thread)
  (do () ((eq? thread (thread-mutex/owner mutex)))
    (suspend-current-thread)))

(define (%forcibly-unlock-mutex mutex thread)
  (warn "lock-thread-mutex: Dead owner" mutex (thread-mutex/owner mutex))
  (ring/enqueue (thread-mutex/waiting-threads mutex) thread)
  (let ((thread* (%unlock-thread-mutex mutex)))
    (if (not (eq? thread thread*))
	(%await-thread-mutex mutex thread))))

(define (unlock-thread-mutex mutex)
  (guarantee-thread-mutex mutex unlock-thread-mutex)
  (without-interrupts
   (lambda ()
     (if (not (eq? (thread-mutex/owner mutex) (current-thread)))
	 (error "Don't own mutex:" mutex))
     (%unlock-thread-mutex mutex))))

(define (%unlock-thread-mutex mutex)
  (let loop ()
    (let ((thread (ring/dequeue (thread-mutex/waiting-threads mutex) false)))
      (cond ((not thread)
	     (set-thread-mutex/owner! mutex thread)
	     thread)
	    ((thread-dead? thread)
	     (loop))
	    (else
	     (set-thread-mutex/owner! mutex thread)
	     (signal-thread-event thread false)
	     thread)))))

(define (with-thread-mutex-locked mutex thunk)
  (guarantee-thread-mutex mutex lock-thread-mutex)
  (let ((thread (current-thread))
	(grabbed-lock?))
    (dynamic-wind
     (lambda ()
       (let ((owner (thread-mutex/owner mutex)))
	 (cond ((eq? owner thread)
		(set! grabbed-lock? false)
		unspecific)
	       (else
		(set! grabbed-lock? true)
		(cond ((not owner)
		       (set-thread-mutex/owner! mutex thread))
		      ((thread-dead? owner)
		       (%forcibly-unlock-mutex mutex thread))
		      (else
		       (%lock-thread-mutex mutex thread)))))))
     thunk
     (lambda ()
       (if (and grabbed-lock? (eq? (thread-mutex/owner mutex) thread))
	   (%unlock-thread-mutex mutex))))))