;;; desp.scm
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;      Here is the alternative approach to auditing the
;;; Silverado Bank, based on the notion of logging records of
;;; deposits and withdrawals rather than interlocking the auditor
;;; and depositors.
;;;
;;; This code was prepared by Peter Szolovits based on an approach
;;; suggested by Jerry Saltzer (who has no responsibility for the
;;; implementation!).  The missing sections are indicated by lines
;;; containing "***********************************************".

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; First, some preliminaries.  We define the structure of a log entry,
;;; then of a log as a list of entries, and finally a set of logs as a
;;; list of logs.
;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Define log entries to hold a record of a single transaction:
;;; depositor making it, amount of deposit (negative if withdrawal),
;;; the new balance after the transaction, and its time.
;;;

(define (make-log-entry person amount new-balance time)
  (list person amount new-balance time))
(define log-person car)
(define log-amount cadr)
(define (zero-log-amount! entry) (set-car! (cdr entry) 0))
(define log-new-balance caddr)
(define log-time cadddr)

(define (earlier? log-entry1 log-entry2)
  (< (log-time log-entry1) (log-time log-entry2)))

(define (same-depositor? log-entry1 log-entry2)
  (eq? (log-person log-entry1) (log-person log-entry2)))

(define (withdrawal? log-entry)
  (< (log-amount log-entry) 0))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; A log is represented as a list of log entries, sorted in reverse
;;; chronological order.  Logs actually stored within accounts may have
;;; new log entries added only at the front, because time is monotonic.
;;; Other logs (e.g., depositor logs) may be constructed from transactions
;;; that are not originally given in reverse chronological order; therefore,
;;; new entries must be inserted in their proper place.  We choose to use
;;; destructive list operations to avoid repeatedly copying lists.  Note
;;; that these would be risky if subject to interference from other processes,
;;; but here we use them only within a single auditor process on data
;;; structures private to that auditor.
;;;

(define (add-to-log log-entry log)
  ;; Walks down log until either the end or when the current log entry is
  ;; earlier than the one to be inserted.  Then splice in the new entry and
  ;; return the whole log, unless the newly-added entry is first, in which
  ;; case we must return the new pair that is the new head of the log. 
  (define (aux l last)
    (if (or (null? l)
            (earlier? (car l) log-entry))
      (cond ((null? last) (cons log-entry l))
            (else (set-cdr! last (cons log-entry l))
                  log))
      (aux (cdr l) l)))
  (aux log '()))

(define (log-before log time)
  ;; Yield the tail of the log whose entries all precede time
  (define (aux l)
    (if (null? l)
      '()
      (if (< (log-time (car l)) time)
        l
        (aux (cdr l)))))
  (aux log))

(define (belongs-to-log? log-entry log test)
  (and (not (null? log)) (test log-entry (car log))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; A set of logs is represented by an unordered list of logs (each of which
;;; is, in turn, a list of log entries).  As for logs, we use destructive
;;; operations to maintain log sets.
;;;

(define (add-to-log-set log-entry log-set belongs-test)
  ;; Walks down the log set either until it runs out, in which case we return
  ;; a new log set extended by a log for this entry; or until we find a log
  ;; to which this entry belongs, in which case we add it to that log.
  (define (aux ls)
    (if (null? ls)
      (cons (add-to-log log-entry '())
            log-set)
      (cond ((belongs-to-log? log-entry (car ls) belongs-test)
             (set-car! ls (add-to-log log-entry (car ls)))
             log-set)
            (else (aux (cdr ls))))))
  (aux log-set))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Create a log-set (organized by a belongs-test) from another set of logs
;;;

(define (do-for-each the-list operation starting-val)
  ;; This is a generalized control procedure somewhat like for-each,
  ;; except that for each element of the-list, it is combined with the
  ;; previous partial result by operation.  The initial partial result is
  ;; starting-val.  For example, (do-for-each '(a b c) cons '()) would 
  ;; simply reverse the list (a b c).
  (define (aux l ans)
    (if (null? l)
      ans
      (aux (cdr l) (operation (car l) ans))))
  (aux the-list starting-val))

(define (create-log-set logs belongs-test)
  ;; Build a new set of logs, one log per depositor.  Include all transactions
  ;; that are in any of the transactions of all accounts under audit.
  (do-for-each logs
               (lambda (log log-set)
                 (do-for-each log 
                              (lambda (log-entry log-set)
                                (add-to-log-set
                                 log-entry log-set belongs-test))
                              log-set))
               '()))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; We need to define a clock with finer resolution than the system clock
;;; defined by runtime.  Because (runtime) often cannot assign different times
;;; to successive transactions, the ordering of logs by time can get confused.
;;; Here we add a very small gain to the runtime that is incremented each
;;; time (runtime1) is called.  Note that we update clock-gain within a
;;; serializer specially made for defining runtime1, to assure that no
;;; two calls to runtime1 can interfere with each other and accidentally
;;; yield the same time.
;;;

(define clock-gain 0)

(define runtime1
  (let ((clock-serialized (make-serializer)))
    (clock-serialized
     (lambda ()
       (set! clock-gain (+ clock-gain 1))
       (+ (runtime) (* .00001 clock-gain))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Having defined the structure of logs, we are ready to give the revised
;;; definition of make-account.  This version corrects the problem identified
;;; in Problem 3 and adds a log, plus methods to manipulate it.  Note that
;;; we create an initial log entry when we create the account, and always leave
;;; at least one log entry when clearing the log.  This assures that the
;;; balance can be updated while the auditor is at work, yet the auditor
;;; can maintain a consistent picture of the account at some particular time.
;;;

(define (make-account account-name owners balance)
  (let ((serialized (make-serializer))
        (log (list (make-log-entry '() 0 balance (runtime1)))))
    (define (account mess)
      (cond ((account-queried account mess))
            ((eq? mess 'account-name)
             account-name)
            ((eq? mess 'owners)
             owners)
            ((eq? mess 'balance)
             balance)
            ((eq? mess 'log)
             log)
            ((eq? mess 'withdraw)
             (lambda (person amount)
               (cond ((withdrawal-requested account person amount))
                     ((not (memq person owners)) 0)
                     ((<= amount 0) 0)
                     (else
                      ((serialized
                        (lambda ()
                          (cond 
                           ((< balance amount) 0)
                           (else (set! balance (- balance amount))
                                 (set! log
                                       (cons (make-log-entry person (- amount)
                                                             balance (runtime1))
                                             log))
                                 amount)))))))))
            ((eq? mess 'deposit)
             (lambda (person amount)           
               (cond ((deposit-requested account person amount))
                     ((<= amount 0) 0)
                     (else
                      ((serialized
                        (lambda ()
                          (set! balance (+ balance amount))
                          (set! log
                                (cons (make-log-entry person amount balance
                                                      (runtime1))
                                      log))
                          amount)))))))
            ((eq? mess 'clear-log-to)
             ;; Note that this is NOT serialized.
             (lambda (clear-time actives)
               (clear-unnecessary-log-entries! log clear-time actives)))
            (else (error "Unknown message -- ACCOUNT" mess))))
    account))

(define (clear-unnecessary-log-entries! log clear-time active-log-entries)
  ;; Clearing unnecessary log entries after a clean audit must leave us with
  ;; 1.  at least one log entry, so that we may determine the balance at the
  ;;     time of the next audit even if no transactions take place between now
  ;;     and then, but do take place while that audit is in progress, and
  ;; 2.  log entries for all withdrawals that do not have corresponding deposits
  ;;     in the depositor-logs, so that we can keep track of depositors with
  ;;     money in hand; note that because deposits in general do not go into
  ;;     the same accounts from which the money was withdrawn, there is no
  ;;     way to tell which withdrawals have not yet been deposited somewhere
  ;;     solely from the log of a single account, so we need to refer to the
  ;;     depositor logs.  We keep any log entry in active-log-entries.

  ;; Note: There is a subtle issue here, which led me into trouble when I first wrote
  ;;     this code:  If the first entry kept in the log (to satisfy 1.) is a withdrawal,
  ;;     but it is not among active-log-entries, then its corresponding deposit
  ;;     transaction log (in a different account) may get deleted.  Next time an
  ;;     audit is done, then, the auditor will complain that there is a withdrawal
  ;;     without a corresponding deposit!  To avoid this, we change the log-amount
  ;;     of such an entry to 0, so it winds up being used only to keep track of the
  ;;     balance, as required by 1, but is ignored by the next audit.
  ;;
  ;; We used to think that this code would need to run serialized with depositors
  ;; who are adding log entries to the log, but it turns out not to be necessary.
  ;; The key is that depositors only extend the log by set!-ing log
  ;; to a new list whose cdr is the old entry.  This clearing procedure
  ;; never sets log at all, but only set-cdr!-s what it believes to be the newest
  ;; entry in the log.  Even if a depositor adds a new log element in front of
  ;; the one about to be changed here, however, it is still appropriate to change
  ;; this one, and the same result is achieved as if they were serialized.
  
  (define (aux l last)
    (cond ((null? l)			;lop off the rest of the log
	   (set-cdr! last '()))
	  ((and (< (log-time (car l)) clear-time)
		(not (memq (car l) active-log-entries)))
	   (set-cdr! last (cdr l))	; delete this log entry, scan rest
	   (aux (cdr l) last))
	  (else (aux (cdr l) l))))
  (aux (cdr log) log)
  ;; As per note above, destroy the fact that the first entry is a withdrawal if
  ;; it would have been deleted except that it's used to keep the balance.
  (if (and (< (log-time (car log)) clear-time)
	   (not (memq (car log) active-log-entries)))
      (zero-log-amount! (car log)))
  #t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; With the preliminaries done, we can now turn to our implementation of
;;; the audit procedure itself.
;;;
;;; We must choose some time at which to reconcile the snapshots of all
;;; the accounts.  The simplest idea is for the auditor to check the time when
;;; the audit begins (before it looks up any of the account balances), and then
;;; arrange to ignore all transactions that took place after (or at) that time.
;;; (The reason we ignore transactions at the same time is in case the
;;; resolution of the timer can allow the same time to be assigned to a later
;;; transaction.  This CAN happen in our Scheme implementation.)
;;;
;;; The auditor will first create a set of logs for the depositors, so it can
;;; examine their individual behaviors.	 It then calculates the total amount
;;; of money in the system, by summing the current balances in each account
;;; and the amounts being held in hand by each depositor.
;;;

(define total-audit-balance #f)	 ;; indicates unknown previous balance

(define (do-audit accounts)
  (let* ((audit-time (runtime1))
	 (a-logs (map (lambda (acct)
			(log-before (acct 'log) audit-time))
		      accounts))
	 (d-logs (create-log-set a-logs same-depositor?))
	 (actives (do-for-each d-logs
			       (lambda (log ans)
				 (let ((entry? (log-shows-incomplete? log)))
				   (if entry? (cons entry? ans) ans)))
			       '()))
	 (total-$ (compute-total a-logs actives)))
    (write-line (list 'audit-total total-$ 'at audit-time))
    (cond ((or (not total-audit-balance)
	       (= total-audit-balance total-$))
	   ;; The audit is clean, because the total amount of money is
	   ;; correct, or was previously unknown.  The first time, we don't
	   ;; know what the balance should have been!
	   (set! total-audit-balance total-$))
	  (else
	   ;; Something is not kosher.
	   (write-line 
	    (list 'current 'balance total-$ 'does 'not 'equal 'previous
		  'balance 'of total-audit-balance))))
    ;; check each depositor's log to see if they've pulled any fast ones
    (for-each (lambda (depositor-log)
		(check-depositor-honesty depositor-log))
	      d-logs)
    ;; clear out old logs after we've reported any cheats
    (for-each (lambda (acct)
		((acct 'clear-log-to) audit-time actives))
	      accounts)
    total-$))

(define (log-shows-incomplete? log)
  ;; Determines whether any money is in the hands of the depositor
  ;; whose log this is.	 This relies heavily on knowing that depositors always
  ;; follow the withdraw/deposit cycle.	 Thus, if the last transaction was a
  ;; withdrawal, its amount should be in hand and the depositor's transaction
  ;; is incomplete.  This could NOT be the basis for auditing real banks,
  ;; of course!
  ;; (No log will be empty, or else it would not have been created!)
  ;; This "predicate" returns the actual log entry as its true value.
  (if (withdrawal? (car log))
    (car log)
    #f))


(define (compute-total a-logs actives)

  0

  ;;**********************************************************
  ;;**************** Missing text here ***********************
  ;;**********************************************************
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Here we try to detect cheaters:
;;;

(define kite-tolerance 0.011)

(define (check-depositor-honesty log)
  ;; We check if a depositor is honest.	 To be honest, s/he must have
  ;; transactions that alternate between deposits and withdrawals, the
  ;; amount deposited must equal the previous amount withdrawn, and the
  ;; time elapsed between withdrawal and deposit must be within kite-tolerance.

  0
  ;;**********************************************************
  ;;**************** Missing text here ***********************
  ;;**********************************************************
  )

;;; (start-bank)
;;; (stop-bank)

