#|
The procedures in these files are used to create census and geography
data strutures from information on the TIGER CD ROMs.  This file does
not need to be loaded, once the data has been dumped in binary form.

Before loading this, make sure that the DATA-STRUCTURES file is loaded.

|#
(define (make-data-subset exclude-pred file-name)
  (let ((new-streets 
	 (list-transform-negative
	     (town/streets *geographic-data*)
	   exclude-pred)))
    (let ((new-tracts
	   (extract-tracts new-streets)))
      (let ((new-census
	     (list-transform-positive
		 *census-data*
	       (lambda (x)
		 (member (cons (pop-data/tract x)
			       (pop-data/blkgrp x))
			 new-tracts)))))
	(initialize-and-dump-data (streets->town new-streets)
				  new-census
				  (string-append ps4-data-directory
						 "/"
						 file-name 
						 ".bin"))))))

(define (initialize-and-dump-data geographic census file-name)
  (initialize-pop-data geographic census)
  (fasdump (make-dataset geographic census)
	   file-name))


(define state-data-file-1)
(define state-data-file-2)
(define state-data-file-3)
(define state-data-file-4)
(define state-data-file-7)
(define state-data-file-8)

;;;
;;; Header structure:
;;; sumlev (3 chars -- 0 3)
;;; statefp (2 chars -- 3 5)
;;; cnty (3 chars  -- 5 8)
;;; cousubfp (5 chars -- 8 13)
;;; placefp (5 chars -- 13 18)
;;; tractbna (6 chars -- 18 24)
;;; blkgr (1 char -- 24 25)
;;; logrecnu (6 chars -- 25 31)

;;;
;;; Magic Constants
;;;

(define pop-header-length 3266)

(define record-header-length 31)

(define record-data-length 837)

(define record-length (+ record-data-length record-header-length 1))

(define pop-header-length-7 4002)

(define record-header-length-7 31)

(define record-data-length-7 1044)

(define record-length-7 (+ record-data-length-7 record-header-length-7 1))

(define pop-header-length-8 3938)

(define record-header-length-8 31)

(define record-data-length-8 1170)

(define record-length-8 (+ record-data-length-8 record-header-length-8 1))

(define (init state)
  (set! state-data-file-1
	(file-open-input-channel
	 (string-append "STF1A1" state ".DBF;1")))
  (set! state-data-file-2
	(file-open-input-channel
	 (string-append "STF1A2" state ".DBF;1")))
  (set! state-data-file-3
	(file-open-input-channel
	 (string-append "STF1A3" state ".DBF;1")))
  (set! state-data-file-4
	(file-open-input-channel
	 (string-append "STF1A4" state ".DBF;1")))
  (set! state-data-file-7
	(file-open-input-channel
	 (string-append "STF1A7" state ".DBF;1")))
  (set! state-data-file-8
	(file-open-input-channel
	 (string-append "STF1A8" state ".DBF;1")))
)

(define (read-record file-num #!optional number parse?)
  (let ((n (if (default-object? number)
	       1
	       number))
	(record-string (make-string record-length)))
    (let ((state-data-file
	   (cond 
	    ((= file-num 1) state-data-file-1)
	    ((= file-num 2) state-data-file-2)
	    ((= file-num 3) state-data-file-3)
	    ((= file-num 4) state-data-file-4)
	    (else
	     (error "Illegal file number")))))
      (file-set-position state-data-file
			 (+ pop-header-length (* (-1+ n) record-length)))
      (channel-read state-data-file record-string 0 record-length)
      record-string)))

(define (read-record-7 #!optional number parse?)
  (let ((n (if (default-object? number)
	       1
	       number))
	(record-string (make-string record-length-7)))
    (file-set-position state-data-file-7
		       (+ pop-header-length-7 (* (-1+ n) record-length-7)))
    (channel-read state-data-file-7 record-string 0 record-length-7)
    record-string))

(define (read-record-8 #!optional number parse?)
  (let ((n (if (default-object? number)
	       1
	       number))
	(record-string (make-string record-length-8)))
    (file-set-position state-data-file-8
		       (+ pop-header-length-8 (* (-1+ n) record-length-8)))
    (channel-read state-data-file-8 record-string 0 record-length-8)
    record-string))

(define (extract-header record-string)
  (substring record-string 0 record-header-length))

(define (extract-data record-string)
  (substring record-string record-header-length (-1+ record-length)))

(define (extract-data-7 record-string)
  (substring record-string record-header-length-8 (-1+ record-length-7)))

(define (extract-data-8 record-string)
  (substring record-string record-header-length-8 (-1+ record-length-8)))


(define (extract-age-data data-string num)
  (substring data-string (* 31 9 num) (* 31 9 (1+ num))))

(define (age-string->age age-string)
  (let loop ((index 0)
	     (string-index 0)
	     (age-list '()))
    (if (< string-index (string-length age-string))
	(loop (1+ index) (+ string-index 9)
	      (cons
	       (string->number (substring age-string string-index (+ string-index 9)))
	       age-list))
	(apply make-age-data (reverse age-list)))))

(define (extract-ppu-data data-string)
  (substring data-string 801 864))

(define (ppu-string->persons-per-unit ppu-string)
  (let loop ((index 0)
	     (string-index 0)
	     (ppu-list '()))
    (if (< string-index (string-length ppu-string))
	(loop (1+ index) (+ string-index 9) 
	      (cons 			 
	       (string->number (substring ppu-string string-index (+ string-index 9)))
	       ppu-list))
	(apply make-persons-per-unit (reverse ppu-list)))))

(define (extract-house-data data-string)
  (substring data-string 135 315))

(define (house-string->house house-string)
  (let loop ((index 0)
	     (string-index 0)
	     (house-list '()))
    (if (< string-index (string-length house-string))
	(loop (1+ index) (+ string-index 9)
	      (cons
	       (string->number (substring house-string string-index (+ string-index 9)))
	       house-list))
	(apply make-house-data (reverse house-list)))))


(define (extract-rent-data data-string)
  (substring data-string 657 810))

(define (rent-string->rent rent-string)
  (let loop ((index 0)
	     (string-index 0)
	     (rent-list '()))
    (if (< string-index (string-length rent-string))
	(loop (1+ index) (+ string-index 9)
	      (cons
	       (string->number (substring rent-string string-index (+ string-index 9)))
	       rent-list))
	(apply make-rent-data (reverse rent-list)))))



(define (record-number->pop-data record-number)
  (define (convert-to-tract tract-string)
    (let ((tract-num (string->number (substring tract-string 0 4)))
	  (tract-suffix (string->number (substring tract-string 4 6))))
      (if tract-num
	  (if tract-suffix
	      (+ tract-suffix (* 100 tract-num))
	      (* 100 tract-num))
	  (if tract-suffix
	      tract-suffix
	      0))))
  (write-line (string-append "Converting " (number->string record-number)))
  (let ((s1 (read-record 1 record-number)))
    (let ((record-head (extract-header s1))
	  (string1 (extract-data s1))
	  (string2 (extract-data (read-record 2 record-number)))
	  (string3 (extract-data (read-record 3 record-number)))
	  (string4 (extract-data (read-record 4 record-number)))
	  (string7 (extract-data-7 (read-record-7 record-number)))
	  (string8 (extract-data-8 (read-record-8 record-number))))
      (let ((data-string (string-append string1 string2 string3 string4)))
	(let ((wh-m-age (age-string->age (extract-age-data data-string 1)))
	      (wh-f-age (age-string->age (extract-age-data data-string 2)))
	      (bl-m-age (age-string->age (extract-age-data data-string 3)))
	      (bl-f-age (age-string->age (extract-age-data data-string 4)))
	      (in-m-age (age-string->age (extract-age-data data-string 5)))
	      (in-f-age (age-string->age (extract-age-data data-string 6)))
	      (as-m-age (age-string->age (extract-age-data data-string 7)))
	      (as-f-age (age-string->age (extract-age-data data-string 8)))
	      (ot-m-age (age-string->age (extract-age-data data-string 9)))
	      (ot-f-age (age-string->age (extract-age-data data-string 10)))
	      (persons-per-unit (ppu-string->persons-per-unit
				 (extract-ppu-data string7)))
	      (house (house-string->house (extract-house-data string8)))
	      (rent (rent-string->rent (extract-rent-data string8))))
	  (make-pop-data
	   (string->number (substring record-head 3 5))
	   (string->number (substring record-head 5 8))
	   (convert-to-tract (substring record-head 18 24))
	   (string->number (substring record-head 24 25))
	   (make-race-sex
	    wh-m-age wh-f-age
	    bl-m-age bl-f-age
	    in-m-age in-f-age
	    as-m-age as-f-age
	    ot-m-age ot-f-age)
	   persons-per-unit
	   house
	   rent
	   false))))))

(define (extract-matching-blkgrps county-num extract-list #!optional index)
  (define (convert-to-tract tract-string)
    (let ((tract-num (string->number (substring tract-string 0 4)))
	  (tract-suffix (string->number (substring tract-string 4 6))))
      (if tract-num
	  (if tract-suffix
	      (+ tract-suffix (* 100 tract-num))
	      (* 100 tract-num))
	  (if tract-suffix
	      tract-suffix
	      0))))
  (let loop ((index 
	      (if (default-object? index)
		  1
		  index))
	     (extract-list extract-list)
	     (pop-data-list '()))
    (if (null? extract-list)
	pop-data-list
	(let ((s1 (extract-header (read-record 1 index))))
	  (let ((sumlev (string->number (substring s1 0 3)))
		(cnty (string->number (substring s1 5 8))))
	    (if (and 
		     (= sumlev 150)
		     (= cnty county-num))
		(let* ((tract (convert-to-tract (substring s1 18 24)))
		       (blkgrp (string->number (substring s1 24 25)))
		       (the-tract (cons tract blkgrp)))
		  (if (member the-tract extract-list)
		      (loop
		       (1+ index)
		       (delete the-tract extract-list)
		       (cons
			(record-number->pop-data index)
			pop-data-list))
		      (loop (1+ index) extract-list pop-data-list)))
		(loop (1+ index) extract-list pop-data-list)))))))

(define (convert-to-tract tract-string)
  (let ((tract-num (string->number (substring tract-string 0 4)))
	(tract-suffix (string->number (substring tract-string 4 6))))
    (if tract-num
	(if tract-suffix
	    (+ tract-suffix (* 100 tract-num))
	    (* 100 tract-num))
	(if tract-suffix
	    tract-suffix
	    0))))

(define (get-street-record-by-division file-string division-number)
  (let ((f1 (open-input-file file-string))
	(c1 (chars->char-set '(#\linefeed))))
    (let loop ((l1 '()))
      (let ((next-record (read-string c1 f1)))
	(if (not (eof-object? next-record))
	    (let ((left-place-code (string->number (substring next-record 140 145)))
		  (right-place-code (string->number (substring next-record 145 150))))
	      (read-char f1)
	      (if (or (and left-place-code
			   (= left-place-code division-number))
		      (and right-place-code
			   (= right-place-code division-number)))
		  (let ((from-long (string->number (string-trim-left (substring next-record 190 200))))
			(from-lat (string->number (substring next-record 200 209)))
			(to-long (string->number (string-trim-left (substring next-record 209 219))))
			(to-lat (string->number (substring next-record 219 228)))
			(cfcc (substring next-record 55 58))
			(name (string-trim (substring next-record 19 49)))
			(suffix (string-trim (substring next-record 49 53)))
			(from-addr-l (string->number (substring next-record 58 69)))
			(to-addr-l (string->number (substring next-record 69 80)))
			(from-addr-r (string->number (substring next-record 80 91)))
			(to-addr-r (string->number (substring next-record 91 102)))
			(zip-l (string->number (substring next-record 106 111)))
			(zip-r (string->number (substring next-record 111 116)))
			(tract-l (convert-to-tract (substring next-record 170 176)))
			(tract-r (convert-to-tract (substring next-record 176 182)))
			(blkgrp-l (string->number (substring next-record 182 183)))
			(blkgrp-r (string->number (substring next-record 186 187))))
		    (loop (cons 
			   (make-street from-long from-lat to-long to-lat cfcc name
					suffix from-addr-l to-addr-l from-addr-r to-addr-r
					zip-l zip-r tract-l tract-r blkgrp-l blkgrp-r)
			   l1)))
		  (loop l1)))
	    (begin
	      (close-output-port f1)
	      l1))))))

(define (get-street-record-by-place-code file-string place-number)
  (let ((f1 (open-input-file file-string))
	(c1 (chars->char-set '(#\linefeed))))
    (let loop ((l1 '()))
      (let ((next-record (read-string c1 f1)))
	(if (not (eof-object? next-record))
	    (let ((left-place-code (string->number (substring next-record 160 165)))
		  (right-place-code (string->number (substring next-record 165 170))))
	      (read-char f1)
	      (if (or (and left-place-code
			   (= left-place-code place-number))
		      (and right-place-code
			   (= right-place-code place-number)))
		  (let ((from-long (string->number (string-trim-left (substring next-record 190 200))))
			(from-lat (string->number (substring next-record 200 209)))
			(to-long (string->number (string-trim-left (substring next-record 209 219))))
			(to-lat (string->number (substring next-record 219 228)))
			(cfcc (substring next-record 55 58))
			(name (string-trim (substring next-record 19 49)))
			(suffix (string-trim (substring next-record 49 53)))
			(from-addr-l (string->number (substring next-record 58 69)))
			(to-addr-l (string->number (substring next-record 69 80)))
			(from-addr-r (string->number (substring next-record 80 91)))
			(to-addr-r (string->number (substring next-record 91 102)))
			(zip-l (string->number (substring next-record 106 111)))
			(zip-r (string->number (substring next-record 111 116)))
			(tract-l (convert-to-tract (substring next-record 170 176)))
			(tract-r (convert-to-tract (substring next-record 176 182)))
			(blkgrp-l (string->number (substring next-record 182 183)))
			(blkgrp-r (string->number (substring next-record 186 187))))
		    (loop (cons 
			   (make-street from-long from-lat to-long to-lat cfcc name
					suffix from-addr-l to-addr-l from-addr-r to-addr-r
					zip-l zip-r tract-l tract-r blkgrp-l blkgrp-r)
			   l1)))
		  (loop l1)))
	    (begin
	      (close-output-port f1)
	      l1))))))

(define (streets->town street-list)
  (let ((min-lat
	 (apply min (map (lambda (x) (min (street/from-lat x) (street/to-lat x))) street-list)))
	(max-lat
	 (apply max (map (lambda (x) (max (street/from-lat x) (street/to-lat x))) street-list)))
	(min-long
	 (apply min (map (lambda (x) (min (street/from-long x) (street/to-long x))) street-list)))
	(max-long
	 (apply max (map (lambda (x) (max (street/from-long x) (street/to-long x))) street-list))))
    (make-town
     min-long
     min-lat
     max-long
     max-lat
     street-list)))

(define (get-places file-string place-num)
  (let ((f1 (open-input-file file-string))
	(c1 (chars->char-set '(#\linefeed))))
    (let loop ((l1 '()))
      (let ((next-record (read-string c1 f1)))
	(if (not (eof-object? next-record))
	    (let ((left-place-code (string->number (substring next-record 160 165)))
		  (right-place-code (string->number (substring next-record 165 170))))
	      (read-char f1)
	      (if (or (and left-place-code
			   (= left-place-code place-num))
		      (and right-place-code
			   (= right-place-code place-num)))
		  (let ((from-long (string->number (string-trim-left (substring next-record 190 200))))
			(from-lat (string->number (substring next-record 200 209)))
			(to-long (string->number (string-trim-left (substring next-record 209 219))))
			(to-lat (string->number (substring next-record 219 228)))
			(cfcc (substring next-record 55 58))
			(name 
			 (string-append (string-trim (substring next-record 19 49))
					" "
					(string-trim (substring next-record 49 53)))))
		    (loop (cons (vector from-long from-lat to-long to-lat cfcc name) l1)))
		  (loop l1)))
	    (begin
	      (close-output-port f1)
	      l1))))))

(define (get-zip-code file-string place-num)
  (let ((f1 (open-input-file file-string))
	(c1 (chars->char-set '(#\linefeed))))
    (let loop ((l1 '()))
      (let ((next-record (read-string c1 f1)))
	(if (not (eof-object? next-record))
	    (let ((left-place-code (string->number (string-trim (substring next-record 106 111))))
		  (right-place-code (string->number (string-trim (substring next-record 111 116)))))
	      (read-char f1)
	      (if (or (and left-place-code
			   (= left-place-code place-num))
		      (and right-place-code
			   (= right-place-code place-num)))
		  (let ((from-long (string->number (string-trim-left (substring next-record 190 200))))
			(from-lat (string->number (substring next-record 200 209)))
			(to-long (string->number (string-trim-left (substring next-record 209 219))))
			(to-lat (string->number (substring next-record 219 228)))
			(cfcc (substring next-record 55 58))
			(name 
			 (string-append (string-trim (substring next-record 19 49))
					" "
					(string-trim (substring next-record 49 53)))))
		    (loop (cons (vector from-long from-lat to-long to-lat cfcc name) l1)))
		  (loop l1)))
	    (begin
	      (close-output-port f1)
	      l1))))))

(define (filter-county file-string)
  (let ((f1 (open-input-file file-string))
	(c1 (chars->char-set '(#\linefeed))))
    (let loop ((l1 '()))
      (let ((next-record (read-string c1 f1)))
	(if (not (eof-object? next-record))
	    (let ((cfcc (substring next-record 55 58)))
	      (let ((cfcc/class (string-ref cfcc 0))
		    (cfcc/subclass (string->number (substring cfcc 1 2))))
		(read-char f1)
		(let ((from-long (string->number (string-trim-left (substring next-record 190 200))))
		      (from-lat (string->number (substring next-record 200 209)))
		      (to-long (string->number (string-trim-left (substring next-record 209 219))))
		      (to-lat (string->number (substring next-record 219 228)))
		      (name 
		       (string-append (string-trim (substring next-record 19 49))
				      " "
				      (string-trim (substring next-record 49 53)))))
		  (let ((the-vec
			 (vector from-long from-lat to-long to-lat cfcc name)))
		    (loop (cons the-vec l1))))))
	    (begin
	      (close-output-port f1)
	      l1))))))


;;; This procedure initializes the block group boundary information

(define (initialize-pop-data town census-data)
  (let ((streets-list (list-transform-positive (town/streets town)
			blkgrp-boundary?)))
    (for-each (lambda (pop-data)
		(make-blkgrp-boundary pop-data streets-list))
	      census-data)))


(define (make-blkgrp-boundary pop-data streets-list)
  (if (not (pop-data/blkgrp-boundary pop-data))
      (let ((the-tract (pop-data/tract pop-data))
	    (the-blkgrp (pop-data/blkgrp pop-data)))
	(let ((boundary-streets
	       (list-transform-positive streets-list
		 (lambda (x)
		   (or (and (= (street/tract-l x) the-tract)
			    (= (street/blkgrp-l x) the-blkgrp))
		       (and (= (street/tract-r x) the-tract)
			    (= (street/blkgrp-r x) the-blkgrp)))))))
	  (set-pop-data/blkgrp-boundary! 
	   pop-data
	   (connect-segments
	    (map (lambda (x)
		   (cons
		    (cons (street/from-long x) (street/from-lat x))
		    (cons (street/to-long x) (street/to-lat x))))
		 boundary-streets))))))
  (pop-data/blkgrp-boundary pop-data))

(define (connect-segments segment-list)
  (let loop ((s-list (cdr segment-list))
	     (p-list (list (car (car segment-list)) (cdr (car segment-list))))
	     (reverse? #t))
    (if (null? s-list)
	p-list
	(let ((point-to-match (car p-list)))
	  (let ((matching-segs (list-transform-positive s-list
				 (lambda (seg) (or (equal? (car seg) point-to-match)
						   (equal? (cdr seg) point-to-match))))))
	    (if (not (null? matching-segs))
		(let ((added-seg (car matching-segs)))
		  (if (equal? (car added-seg) point-to-match)
		      (loop (delete added-seg s-list) (cons (cdr added-seg) p-list) reverse?)
		      (loop (delete added-seg s-list) (cons (car added-seg) p-list) reverse?)))
		(if reverse? 
		    (loop s-list (reverse p-list) #f)
		    (loop (cdr s-list) (cons (car (car s-list)) 
					     (cons (cdr (car s-list)) (reverse p-list))) #t)
		    )))))))


(define (connected? seg1 seg2)
  (or (equal? (car seg1) (car seg2))
      (equal? (car seg1) (cdr seg2))
      (equal? (cdr seg1) (car seg2))
      (equal? (cdr seg1) (cdr seg2))))


(define (extract-tracts data-set)
  (let loop ((tract-list '())
	     (data-set data-set))
    (if (null? data-set)
	(delete '(0) tract-list)
	(let ((head (car data-set))
	      (rest (cdr data-set)))
	  (let ((the-tract-l (cons (street/tract-l head)
				   (street/blkgrp-l head)))
		(the-tract-r (cons (street/tract-r head)
				   (street/blkgrp-r head))))
	    (cond ((equal? the-tract-l the-tract-r)
		   (if (member the-tract-l tract-list)
		       (loop tract-list rest)
		       (loop (cons the-tract-l tract-list) rest)))
		  ((member the-tract-l tract-list)
		   (if (member the-tract-r tract-list)
		       (loop tract-list rest)
		       (loop (cons the-tract-r tract-list) rest)))
		  (else
		   (if (member the-tract-r tract-list)
		       (loop (cons the-tract-l tract-list) rest)
		       (loop (cons the-tract-l 
				   (cons the-tract-r tract-list)) rest)))))))))




