;;; This is file PS8-CODE.SCM

;;; It contains the metacircular evaluator, as described in section 4.1 of the
;;; textbook, with a few minor modifications.

;;; You should just load this file into Scheme without editing it.  The few
;;; procedures that you will need to modify in order to do the problem set
;;; have been copied into a separate file for your convenience.

;;; SETTING UP THE ENVIRONMENT
;;; --------------------------
;;; We initialize the global environment by snarfing a few primitives from the
;;; underlying Scheme system, and binding them (to symbols of the same name).
;;; The actual structure of the environment is determined by the constructor
;;; EXTEND-ENVIRONMENT which is listed below together with the code that
;;; manipulates environments.  If you want to add more primitives to your
;;; evaluator, you can modify the list PRIMITIVE-NAMES to include more Scheme
;;; primitives.

(define primitive-names
  '(+ - * / = < > 1+ -1+ cons car cdr atom? eq? null? not user-print))

(define (setup-environment)
  (define initial-env
    (extend-environment
      primitive-names
      (mapcar (lambda (pname)
		(eval pname user-initial-environment))
	      primitive-names)
      (new-environment)))
  (define-variable! 'nil   '()   initial-env)
  (define-variable! 't     true  initial-env)
  (define-variable! 'true  true  initial-env)
  (define-variable! 'false false initial-env)
  initial-env)

;;; Note that the definition of SETUP-ENVIRONMENT differs from that in the
;;; textbook in that we use the MAPCAR above to extract the underlying Scheme
;;; versions of the primitive procedures.


;;; INITIALIZATION AND DRIVER LOOP
;;; ------------------------------
;;; The following code initializes the machine and starts the Lisp system.
;;; You should not call it very often because it will clobber the global
;;; environment, and you will lose any definitions you have accumulated.

(define (initialize-lisp)
  (set! the-global-environment (setup-environment))
  (force magic-recontinue-promise)
  (driver-loop))

;;; Here is the actual driver loop.  It reads in an expression, passes it to
;;; the machine to be evaluated in the global environment, and prints the
;;; result

;;; When/If your interaction with the evaluator bombs out in an error, you
;;; should restart it by calling DRIVER-LOOP.  Note that the driver uses a
;;; prompt of "MC-EVAL==> " to help you avoid confusing typing to the simulator
;;; with typing to the underlying Scheme interpreter (which has prompt "==> ").

(define driver-loop
  (lambda ()
    (define (real-driver-loop)
      (newline)
      (princ "MC-EVAL==> ")
      ;; Provide a nice exit mechanism    [EXIT-ON? defined on last page]
      (let ((input (read-from-keyboard)))
	(cond ((exit-on? input)	
	       (if (atom? input) input (car input)))
	      (else (user-print (mini-eval input the-global-environment))
		    (driver-loop)))))
    ;; Shameless hack to auto-initialize the first time you call driver-loop
    (newline)
    (princ "Initializing the mini-evaluator's global environment...")
    (set! driver-loop real-driver-loop)
    (initialize-lisp)))

;;; We use a special PRINT here, which avoids printing the environment part of
;;; a compound procedure, since the latter is a very long (or even circular)
;;; list.

(define (user-print object)
  (cond ((compound-procedure? object)
	 (print (list 'compound-procedure
		      (procedure-text object)
		      '<env>)))
	(else (print object))))


;;; THE GUTS OF THE EVALUATOR
;;; -------------------------

; EVAL

(define (mini-eval exp env)
  (cond ((self-evaluating? exp) exp)
        ((quoted?          exp) (text-of-quotation     exp     ))
        ((variable?        exp) (lookup-variable-value exp  env))
        ((definition?      exp) (eval-definition       exp  env))
        ((assignment?      exp) (eval-assignment       exp  env))
        ((lambda?          exp) (make-procedure        exp  env))
        ((conditional?     exp) (eval-cond    (clauses exp) env))
        ((application?     exp) ;; NB: This must be last form tested
         (mini-apply (mini-eval      (operator exp) env)
		     (list-of-values (operands exp) env)))
        (else (mini-error "Unknown expression type --MINI-EVAL" exp))))

; APPLY

(define (mini-apply procedure arguments)
  (cond ((primitive-procedure? procedure)
         (apply-primitive-procedure procedure arguments))
        ((compound-procedure? procedure)
         (eval-sequence (procedure-body procedure)
                        (extend-environment
			  (parameters procedure)
			  arguments
			  (procedure-environment procedure))))
        (else (mini-error "Unknown procedure type -- MINI-APPLY" procedure))))

; ARGUMENTS TO PROCEDURES

(define (list-of-values exps env)
  (cond ((no-operands? exps) '())
        (else (cons (mini-eval      (first-operand exps) env)
                    (list-of-values (rest-operands exps) env)))))

; CONDITIONALS

(define (eval-cond clist env)
  (cond (( no-clauses? clist) '())
        ((else-clause? (first-clause clist))
         (eval-sequence    (actions   (first-clause clist)) env))
        ((true? (mini-eval (predicate (first-clause clist)) env))
         (eval-sequence    (actions   (first-clause clist)) env))
        (else              (eval-cond (rest-clauses clist)  env))))

; SEQUENCES

(define (eval-sequence exps env)
  (cond ((last-exp? exps) (mini-eval (first-exp exps) env))
        (else (mini-eval     (first-exp exps) env)
              (eval-sequence (rest-exps exps) env))))

; ASSIGNMENTS

(define (eval-assignment assn env)
  (let ((new-value (mini-eval (assignment-value    assn) env)))
    (set-variable-value! (assignment-variable assn) new-value env)
    new-value))

; DEFINITIONS

(define (eval-definition defn env)
  (define-variable! (definition-variable defn)
                    (mini-eval (definition-value defn) env)
                    env)
  (definition-variable defn))


;;; SYNTAX OF THE LANGUAGE
;;; ----------------------

; SELF-EVALUATING ITEMS

(define (self-evaluating? exp) (number? exp))
  
; VARIABLES

(define (variable?            exp) (symbol? exp))
(define (same-variable? var1 var2) (eq? var1 var2))  ;; Nice abstraction

; SPECIAL FORMS SYNTAX

(define (form-with-tag? tag exp)  ;; Notice that Scheme special forms are in
  (cond ((not (atom? exp))	  ;;  fact type tagged by the special form
	 (eq? (car exp) tag))	  ;;  name. It's good abstraction to exploit
	(else false)))            ;;  this.

; QUOTATIONS

(define (quoted?            exp) (form-with-tag? 'quote exp))
(define (text-of-quotation quot) (cadr quot))

; ASSIGNMENTS

(define (assignment?          exp) (form-with-tag? 'set! exp))
(define (assignment-variable assn) (cadr  assn))
(define (assignment-value    assn) (caddr assn))

; DEFINITIONS

(define (definition? exp) (form-with-tag? 'define exp))

(define (definition-variable defn)
  (if (variable? (cadr defn))       ;;   (DEFINE  foo      ...)
      (cadr  defn)
      (caadr defn)))                ;;   (DEFINE (foo ...) ...)

(define (definition-value defn) 
  (if (variable? (cadr defn))       ;;   (DEFINE  foo      ...)
      (caddr defn)
      (cons 'lambda                 ;;   (DEFINE (foo p...) b...)
	    (cons (cdadr defn)      ;; = (DEFINE  foo (lambda (p...) b...))
		  (cddr  defn)))))

; LAMBDA EXPRESSIONS

(define (lambda? exp) (form-with-tag? 'lambda exp))

; CONDITIONALS

(define (conditional?  exp) (form-with-tag? 'cond exp))
(define (clauses      cndl) (cdr cndl))

(define (   no-clauses? clauses) (null? clauses))
(define (first-clause   clauses) (car   clauses))
(define ( rest-clauses  clauses) (cdr   clauses))
(define ( else-clause?  clause ) (eq? (predicate clause) 'else))

(define (predicate clause) (car clause))
(define (actions   clause) (cdr clause))

(define (true? x) (not (null? x)))

; SEQUENCES OF EXPRESSIONS

(define ( last-exp? seq) (null? (cdr seq)))
(define (first-exp  seq) (car seq))
(define (rest-exps  seq) (cdr seq))

; PROCEDURE APPLICATIONS

(define (application? exp) (not (atom? exp)))
(define (operator     app) (car app))
(define (operands     app) (cdr app))

(define (   no-operands? rands) (null? rands))
(define (first-operand   rands) (car   rands))
(define ( rest-operands  rands) (cdr   rands))
(define ( last-operand?  rands) (null? (cdr rands)))	;**

; COMPOUND PROCEDURES... look like (PROCEDURE (LAMBDA (param...) body...) env)

(define (make-procedure lambda-exp env)
  (list 'procedure lambda-exp env))

(define (compound-procedure?    exp) (form-with-tag? 'procedure exp))
(define (procedure-text        proc) (cadr proc))	;; For USER-PRINT
(define (parameters            proc) (cadr (cadr proc)))
(define (procedure-body        proc) (cddr (cadr proc)))
(define (procedure-environment proc) (caddr proc))


;;; APPLYING PRIMITIVE PROCEDURES
;;; -----------------------------
;;; The mechanism for applying primitive procedures is somewhat different from
;;; the one given in the textbook.  We can recognize primitive procedures
;;; (which are all inherited from Scheme) by asking Scheme if the object we
;;; have is a Scheme procedure.

(define (primitive-procedure? p) (applicable? p))

;;; To apply a primitive procedure, we ask the underlying Scheme system to
;;; perform the application.  (Of course, an implementation on a low-level
;;; machine would perform the application in some other way.)

(define (apply-primitive-procedure p args)
  (apply p args))


;;; ENVIRONMENTS
;;; ------------

; OPERATIONS ON ENVIRONMENTS

(define (lookup-variable-value var env)
  (let ((b (binding-in-env var env)))
    (cond ((found-binding? b)
	   (binding-value  b))
	  (else (mini-error "Unbound variable" var)))))

(define (binding-in-env var env)
  (if (no-more-frames? env)
      no-binding
      (let ((b (binding-in-frame var (first-frame env))))
	(if (found-binding? b)
	    b
	    (binding-in-env var (rest-frames env))))))

(define (extend-environment variables values  base-env)
  (adjoin-frame (make-frame variables values) base-env))

(define (set-variable-value! var val env)
  (let ((b (binding-in-env var env)))
    (cond ((found-binding? b)
	   (set-binding-value! b val))
	  (else (mini-error "Unbound variable" var)))))

(define (define-variable! var val env)
  (let ((b (binding-in-frame var (first-frame env))))
    (if (found-binding? b)
	(set-binding-value! b val)
	(set-first-frame! env
			  (adjoin-binding (make-binding var val)
					  (first-frame env))))))

; REPRESENTING ENVIRONMENTS... lists of FRAMEs

(define (new-environment) '())

(define the-global-environment (new-environment))

(define (first-frame env) (car env))
(define (rest-frames env) (cdr env))

(define (no-more-frames? env) (null? env))

(define (adjoin-frame frame env) (cons frame env))

(define (set-first-frame! env new-frame)
  (set-car! env new-frame))


; FRAMES... lists of BINDINGs

(define (make-frame variables values)
  (cond ((and (null? variables) (null? values)) '())
	((null? variables)
	 (mini-error "Too many values supplied" values))
	((null? values)
	 (mini-error "Too many variables supplied" variables))
	(else
	 (cons (make-binding (car variables) (car values))
	       (make-frame   (cdr variables) (cdr values))))))

(define (adjoin-binding binding frame) (cons binding frame))

(define (binding-in-frame var frame)
  (define (scan bindings)
    (cond ((null? bindings) no-binding)
	  ((same-variable? var (binding-variable (car bindings)))
	   (car bindings))
	  (else (scan (cdr bindings)))))
  (scan frame))

; BINDINGS... pairs of <var . value>

(define (found-binding? b)
  (not (eq? b no-binding)))

(define no-binding nil)

(define (make-binding variable value) (cons variable value))

(define (binding-variable binding) (car binding))
(define (binding-value    binding) (cdr binding))

(define (set-binding-value! binding value) (set-cdr! binding value))

;;; AUXILIARY UTILITIES
;;; ---------------------

; Some familiar useful procedures (from the Adventure game)

(define (user-display-message list-of-stuff)
  (newline)
  (for-each (lambda (s) (princ s) (princ " "))
	    list-of-stuff))

(define for-each mapc)  ; A rose by this name smells sweeter.

; EXIT-ON?

(define (exit-on? input) (member input the-many-names-of-exit))

(define the-many-names-of-exit
  '( quit   exit   done   punt   game-over   fini
    (quit) (exit) (done) (punt) (game-over) (fini)))


; USER-SAYS-YES?

(define (user-says-yes?) (member (read-from-keyboard) the-many-names-of-yes))

(define the-many-names-of-yes
  '(y yes yup ya uh-huh yep yepper ok sure why-not? ja affirmative
    t true not-nil non-negative 1 oui))


; MINI-ERROR... a magic way to give you the option of staying in DRIVER-LOOP

(define (mini-error string . args)
  (user-display-message (cons 'OOPS! (cons string args)))
  (newline)
  (princ "Wanna fall through to the real debugger? (y/n): ")
  (cond ((user-says-yes?)
	 (error "FELL THROUGH TO REAL DEBUGGER (type `y' then `u'): " string))
	(else (newline) (newline)
	      (princ "OK, continuing with the driver-loop...")
	      (magic-recontinue))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; SURGEON GENERAL'S WARNING: Don't look... this hack will fry your eyes!! ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define magic-recontinue-promise
  (delay
   (sequence (enable-language-features)
	     (let ((call-with-reentrant-exit call-with-current-continuation))
	       (disable-language-features)
	       (call-with-reentrant-exit
		(lambda (exit)
		  (set! magic-recontinue-promise
			(delay (lambda () (exit 'Zoinks!!))))
		  'Yow!))))))

(define (magic-recontinue)
  ((force magic-recontinue-promise)))

; fini
