;;;		     MASSACHUSETTS INSTITUTE OF TECHNOLOGY
;;;	   Department of Electrical Engineering and Computer Science
;;;	   6.001---Structure and Interpretation of Computer Programs
;;;			      Fall Semester, 1990
;;;				 Problem Set 6
;;;
;;;			    Code file PS6-ADV.SCM


;;; ----------------------------------------------------------------------------
;;; Simple object system with inheritance

(define (ask object message . args)  ;; See your Scheme manual to explain `.'
  (let ((method (get-method object message)))
    (if (method? method)
	(apply method (cons object args))
	(error "No method" message (cadr method)))))

(define (get-method object message)
  (object message))

(define (no-method name)
  (list 'no-method name))

(define (no-method? x)
  (if (pair? x)
      (eq? (car x) 'no-method)
      false))

(define (method? x)
  (not (no-method? x)))


;;; ----------------------------------------------------------------------------
;;; Persons, places, and possessions will all be kinds of named objects

(define (make-named-object name)
  (lambda (message) 
    (cond ((eq? message 'name)
	   (lambda (self) name))
	  (else (no-method name)))))

;;; A thing is something that can be owned

(define (make-thing name)
  (let ((named-obj (make-named-object name))
	(owner 'nobody))
    (lambda (message)
      (cond ((eq? message 'ownable?)
	     (lambda (self) true))
	    ((eq? message 'owned?)
	     (lambda (self)
	       (not (eq? owner 'nobody))))
	    ((eq? message 'change-owner)
	     (lambda (self new-owner)
	       (set! owner new-owner)
	       'done))
	    ((eq? message 'owner)
	     (lambda (self) owner))
	    (else
	     (get-method named-obj message))))))

;;; Implementation of places

(define (make-place name)
  (let ((named-obj (make-named-object name))
	(neighbor-map '())
	(things       '()))
    (lambda (message)
      (cond ((eq? message 'things)
	     (lambda (self) things))
	    ((eq? message 'neighbors)
	     (lambda (self) (mapcar cdr neighbor-map)))
	    ((eq? message 'exits)
	     (lambda (self) (mapcar car neighbor-map)))
	    ((eq? message 'neighbor-towards)
	     (lambda (self direction)
	       (let ((places (assq direction neighbor-map)))
		 (if (null? places) false (cdr places)))))
	    ((eq? message 'appear)
	     (lambda (self new-thing)
	       (cond ((memq new-thing things)
		      (append (list (ask new-thing 'name))
			      '(is already at)
			      (list name)))
		     (else (set! things (cons new-thing things))
			   'appeared))))
	    ((eq? message 'gone)
	     (lambda (self thing)
	       (cond ((not (memq thing things))
		      (append (list (ask thing 'name))
			      '(is not at)
			      (list name)))
		     (else (set! things (delq thing things)) ;; DELQ defined
			   'disappeared))))                  ;;  below
	    ((eq? message 'new-neighbor)
	     (lambda (self direction new-neighbor)
	       (cond ((assq direction neighbor-map)
		      (append '(direction already assigned)
			      (list direction name)))
		     (else
		      (set! neighbor-map
			    (cons (cons direction new-neighbor) neighbor-map))
		      'connected))))
	    (else (get-method named-obj message))))))

;;; Implementation of people

(define (make-person name place threshold)
  (let ((possessions '())
	(restlessness 0)
	(named-obj (make-named-object name)))
    (lambda (message)
      (cond ((eq? message 'person?)     (lambda (self) true))
	    ((eq? message 'place)       (lambda (self) place))
	    ((eq? message 'possessions) (lambda (self) possessions))
	    ((eq? message 'exits)       (lambda (self) (ask place 'exits)))
	    ((eq? message 'move)
	     (lambda (self)
	       (cond ((>= restlessness threshold)
		      (ask self 'act)
		      (set! restlessness 0))
		     (else (set! restlessness (1+ restlessness))))
	       'moved))
	    ((eq? message 'say)
	     (lambda (self list-of-stuff)
	       (display-message
		(append '("At") (list (ask place 'name)) '(":")
			(list name) '("says --")
			(if (null? list-of-stuff)
			    '("Oh, nevermind.")
			    list-of-stuff)))))
	    ((eq? message 'look-around)
	     (lambda (self)
	       (let ((other-things
		      (mapcar (lambda (thing) (ask thing 'name))
			      (delq self (ask place 'things)))))  ;; DELQ
		 (ask self 'say					  ;; defined
		      (append '("I see")			  ;; below
			      (if (null? other-things)
				  '("nothing")
				  other-things))))))
	    ((eq? message 'take)
	     (lambda (self thing)
	       (if (if (memq thing (ask place 'things))
		       (is-a thing 'ownable?)
		       false)
		   (let ((owner (ask thing 'owner)))
		     (cond ((ask thing 'owned?)
			    (ask owner 'lose thing)
			    (ask owner 'have-fit))
			   (else 'unowned))
		     (ask thing 'change-owner self)
		     (set! possessions (cons thing possessions))
		     (ask self 'say
			  (append '("I take") (list (ask thing 'name)))))
		   (append '(you can not take) (list (ask thing 'name))))))
	    ((eq? message 'lose)
	     (lambda (self thing)
	       (cond ((eq? self (ask thing 'owner))
		      (set! possessions (delq thing possessions)) ;; DELQ
		      (ask thing 'change-owner 'nobody)           ;; defined
		      (ask self 'say                              ;; below
			   (append '("I lose") (list (ask thing 'name)))))
		     (else
		      (append (list name) '(does not own) (ask thing 'name))))))
	    ((eq? message 'list-possessions)
	     (lambda (self)
	       (ask self 'say
		    (append '("I have")
			    (if (null? possessions)
				'("nothing")
				(mapcar (lambda (p) (ask p 'name))
					possessions))))))
	    ((eq? message 'go)
	     (lambda (self direction)
	       (let ((new-place (ask place 'neighbor-towards direction)))
		 (if (not (null? new-place))
		     (ask self 'move-to new-place)
		     (append '(you cannot go) (list direction) '(from)
			     (list (ask place 'name)))))))

	    ((eq? message 'act)
	     (lambda (self)
	       (let ((new-place (random-place place)))
		 (if (not (null? new-place))
		     (ask self 'move-to new-place)))))
	    ((eq? message 'have-fit)
	     (lambda (self)
	       (ask self 'say '("Yaaaah! I am upset!"))))
	    ((eq? message 'move-to)
	     (lambda (self new-place)
	       (announce-move name place new-place)
	       (ask place 'gone self)
	       (for-each (lambda (p)
			   (ask place 'gone p)
			   (ask new-place 'appear p))
			 possessions)
	       (ask new-place 'appear self)
	       (set! place new-place)
	       (greet-people self (other-people-at-place self new-place))
	       'moved))
	    (else (get-method named-obj message))))))

;;; A troll is a kind of person (but not a kind person!)

(define (make-troll name place threshold)
  (let ((me (make-person name place threshold)))
    (lambda (message)
      (cond ((eq? message 'act)
	     (lambda (self)
	       (let ((others (other-people-at-place
			      self
			      (ask self 'place))))
		 (if (not (null? others))
		     (ask self 'eat-person (pick-random others))
		     ((get-method me 'act) self)))))
	    ((eq? message 'eat-person)
	     (lambda (self person)
	       (ask self 'say
		    (append '("Growl.... I'm going to eat you,")
			    (list (ask person 'name))))
	       (go-to-heaven person)
	       (ask self 'say
		    (append '("Chomp chomp.") (list (ask person 'name))
			    '("tastes yummy!")))
	       'eaten))
	    (else (get-method me message))))))

(define (go-to-heaven person)
  (for-each (lambda (item) (ask person 'lose item))
	    (ask person 'possessions))
  (ask person 'say '("Better to have hacked and lost than never to have hacked at all!"))
  (ask person 'move-to heaven)
  (remove-from-clock-list person)
  'game-over-for-you-dude)

;;; --------------------------------------------------------------------------
;;; Clock routines

(define *clock-list* '())

(define (initialize-clock-list)
  (set! *clock-list* '())
  'initialized)

(define (add-to-clock-list person)
  (set! *clock-list* (cons person *clock-list*))
  'added)

(define (remove-from-clock-list person)
  (set! *clock-list* (delq person *clock-list*))  ;; DELQ defined below
  'removed)

(define (clock)
   (for-each (lambda (person) (ask person 'move))
	     *clock-list*))
	     

(define (run-clock n)
  (cond ((= n 0) 'done)
	(else (clock) (run-clock (-1+ n)))))
;;; --------------------------------------------------------------------------
;;; Miscellaneous procedures

(define (is-a object type)
  (let ((m (get-method object type)))
    (if (method? m)
	(ask object type)
	false)))

(define (install-person person)
    (ask (ask person 'place) 'appear person)
    (add-to-clock-list person)
    'installed)

(define (other-people-at-place person place)
  (filter (lambda (object)
	    (if (not (eq? object person))
		(is-a object 'person?)
		false))
	  (ask place 'things)))

(define (greet-people person people)
  (if (not (null? people))
      (ask person 'say
	   (append '("Hi")
		   (mapcar (lambda (p) (ask p 'name))
			   people)))))

(define (display-message list-of-stuff)
  (newline)
  (for-each (lambda (s) (princ s) (princ " "))
	    list-of-stuff))

(define (announce-move name old-place new-place)
  (display-message
   (list name "moves from" (ask old-place 'name)
	 "to" (ask new-place 'name))))

(define (random-place old-place)
  (pick-random (ask old-place 'neighbors)))

(define (filter predicate lst)
  (cond ((null? lst) '())
	((predicate (car lst))
	 (cons (car lst) (filter predicate (cdr lst))))
	(else (filter predicate (cdr lst)))))


(define (pick-random lst)
  (if (null? lst)
      '()
      (list-ref lst (random (length lst)))))  ;; See manual for LIST-REF

(define (delq item lst)
  (cond ((null? lst) '())
	((eq? item (car lst)) (delq item (cdr lst)))
	(else (cons (car lst) (delq item (cdr lst))))))

;; FOR-EACH is another (preferred) name for the primitive
;; procedure MAPC.

(define for-each mapc)

