;;; 6.001 Register machine simulator for multiprocessors.
;;; Derived from register machine simulator in textbook.
;;; Nikhil, November 24, 1988

;;; ****************************************************************
;;; Registers

(define make-register (lambda (name)
    (define contents nil)
    (define trace false)
    (define set (lambda (value)
        (if trace
	    (print (list 'reg-trace: name 'new-value: value)))
	(set! contents value)))
    (define dispatch (lambda (message)
        (cond ((eq? message 'get) contents)
	      ((eq? message 'set) set)
	      ((eq? message 'trace-on)  (set! trace t))
	      ((eq? message 'trace-off) (set! trace false))
	      ((eq? message 'display)   (list name contents))
	      (else (error "Unknown request -- REGISTER"
			   name
			   message)))))
    dispatch))

(define get-register-contents (lambda (register)
    (register 'get)))
(define fetch (lambda (register)
    (get-register-contents register)))

(define set-register-contents! (lambda (register value)
    ((register 'set) value)))

(define register-trace-on (lambda (register)
    (register 'trace-on)))
(define register-trace-off (lambda (register)
    (register 'trace-off)))

;;; ****************************************************************
;;; Processors:

(define get-processor-register (lambda (processor reg-name)
    (eval reg-name processor)))

(define advance-program-counter (lambda (processor)
  (let
        ((pc (get-processor-register processor 'program-counter)))
    (set-register-contents! pc (cdr (get-register-contents pc))))))

(define set-program-counter (lambda (processor new-pc-value)
  (let
        ((pc (get-processor-register processor' program-counter)))
    (set-register-contents! pc new-pc-value))))

(define abbrev-list (lambda (l n)
  (cond
    ((not (pair? l)) l)
    ((= n 0)  '(...) )
    (else (cons (abbrev-display (car l))
		(abbrev-list (cdr l) (- n 1)))))))
      
(define abbrev-display (lambda (l)
    (if (not (pair? l))
	l
	(abbrev-list l 3))))

(define display-processor (lambda (processor)
  (list
    'processor
    (eval 'processor-name processor)
    (abbrev-display ((get-processor-register processor 'program-counter) 'display))
    (abbrev-display ((get-processor-register processor 'exp) 'display))
    (abbrev-display ((get-processor-register processor 'env) 'display))
    (abbrev-display ((get-processor-register processor 'val) 'display))
    (abbrev-display ((get-processor-register processor 'fun) 'display))
    (abbrev-display ((get-processor-register processor 'unev) 'display))
    (abbrev-display ((get-processor-register processor 'argl) 'display))
    (abbrev-display ((get-processor-register processor 'continue) 'display))
    (abbrev-display ((get-processor-register processor 'stack) 'display))
)))

;;; ****************************************************************
;;; Instruction execution in processors.
;;; Executing an instruction returns a list of processors:

(define execute-instruction (lambda (processor processor-list)
  (let
        ((instruction-sequence (fetch (get-processor-register processor 'program-counter))))
    (if (null? instruction-sequence)
	processor-list
	(let
              ((instruction (car instruction-sequence)))
	  (let
	        ((opcode (car instruction))
		 (args   (mapcar (lambda (operand) (eval operand processor))
				 (cdr instruction))))
	    ((instruction-method opcode) processor args processor-list)))))))

;;; ****************************************************************
;;; Instruction table (``microcode'')

(define exec-assign (lambda (processor args processor-list)
  (let
        ((register (first args))
	 (value    (second args)))
    (set-register-contents! register value)
    (advance-program-counter processor)
    (cons processor processor-list))))

(define exec-save (lambda (processor args processor-list)
  (let
        ((register (first args))
	 (stack    (get-processor-register processor 'stack)))
    (set-register-contents! stack
			    (cons (get-register-contents register)
				  (get-register-contents stack)))
    (advance-program-counter processor)
    (cons processor processor-list))))

(define exec-restore (lambda (processor args processor-list)
  (let
        ((register (first args))
	 (stack    (get-processor-register processor 'stack)))
    (set-register-contents! register
			    (car (get-register-contents stack)))
    (set-register-contents! stack
			    (cdr (get-register-contents stack)))
    (advance-program-counter processor)
    (cons processor processor-list))))

(define exec-goto (lambda (processor args processor-list)
  (let
        ((new-sequence (first args))
	 (pc (get-processor-register processor 'program-counter)))
    (set-register-contents! pc new-sequence)
    (cons processor processor-list))))

(define exec-branch (lambda (processor args processor-list)
  (let
        ((predicate (first args)))
    (if predicate
        (set-program-counter processor (second args))
        (advance-program-counter processor))
    (cons processor processor-list))))

(define exec-perform (lambda (processor args processor-list)
  (advance-program-counter processor)
  (cons processor processor-list)))

(define exec-spawn (lambda (processor args processor-list)
  (let
        ((new-processor (eval '(make-processor) processor)))
    (let
          ((stack (get-processor-register new-processor 'stack))
           (pc    (get-processor-register new-processor 'program-counter)))
      (set-register-contents! stack (first args))
      (set-register-contents! pc (eval '*start* processor))
      (advance-program-counter processor)
      (cons processor
	    (cons new-processor
		  processor-list))))))

(define exec-set-I-cell (lambda (processor args processor-list)
  (let
        ((i-cell (first args))
	 (value  (second args)))
    (let
           ((waiting-processors (i-cell-waiting-list i-cell)))
      (set-i-cell-value! i-cell value)
      (set-i-cell-flag! i-cell)
      (advance-program-counter processor)
      (cons processor
	    (unordered-append waiting-processors
			      processor-list))))))

(define exec-get-i-cell (lambda (processor args processor-list)
  (let
        ((register (first args))
	 (i-cell   (second args)))
    (if (i-cell-flag-set? i-cell)
	(sequence
	      (set-register-contents! register (i-cell-value i-cell))
	      (advance-program-counter processor)
	      (cons processor processor-list))
	(sequence
	      (add-to-i-cell-waiting-list i-cell processor)
	      processor-list)))))

(define instruction-table
    (make-environment
      (define assign      exec-assign)
      (define save        exec-save)
      (define restore     exec-restore)
      (define goto        exec-goto)
      (define branch      exec-branch)
      (define perform     exec-perform)
      (define spawn       exec-spawn)
      (define set-i-cell! exec-set-i-cell)
      (define get-i-cell  exec-get-i-cell)
))

(define instruction-method (lambda (opcode)
    (eval opcode instruction-table)))

;;; ****************************************************************
;;; General help functions

;;; UNORDERED-APPEND is used in EXEC-SET-I-CELL method above.  Appends
;;; two lists, but since order does not matter, does it iteratively,
;;; ``pouring'' l1 onto l2.

(define unordered-append (lambda (l1 l2)
    (if (null? l1)
	l2
	(unordered-append (cdr l1)
			  (cons (car l1) l2)))))

;;; THREADED-MAP is used in SWEEP function inside the multiprocessor below.

(define threaded-map (lambda (proc L)    ;;; proc: (Item, List) -> List
    (define loop (lambda (Lin Lout)
        (if (null? Lin)
	    Lout
	    (loop (cdr Lin)
		  (proc (car Lin) Lout)))))
    (loop L nil)))

;;; ****************************************************************
;;; Multiprocessor model

(define make-raw-multiprocessor (lambda ()
    (make-environment

      (define *labels* nil)
      (define *main-processor* nil)
      (define processor-list nil)
      (define *final-value* 'unknown)

      (define *sweep-count* 0)
      (define *processor-count* 0)
      (define *max-parallelism* 0)
      (define *total-instruction-count* 0)
      (define *parallelism-profile* nil)

      (define init-stats (lambda ()
          (set! *sweep-count* 0)
	  (set! *processor-count* 0)
	  (set! *max-parallelism* 0)
	  (set! *total-instruction-count* 0)
	  (set! *parallelism-profile* nil)))

      (define *trace* false)
      (define start-trace (lambda () (set! *trace* true)))
      (define stop-trace  (lambda () (set! *trace* false)))

      (define make-processor (lambda ()
          (make-environment
            (define processor-name  *processor-count*)
            (set! *processor-count* (1+ *processor-count*))
	    (define program-counter (make-register 'program-counter))
	    (define exp             (make-register 'exp))
	    (define env             (make-register 'env))
	    (define val             (make-register 'val))
	    (define fun             (make-register 'fun))
	    (define unev            (make-register 'unev))
	    (define argl            (make-register 'argl))
	    (define continue        (make-register 'continue))
	    (define stack           (make-register 'stack)))))

      (define display (lambda ()
          (newline)
	  (princ "Sweep number ")
	  (princ *sweep-count*)
	  (mapc (lambda (p)
		  (print (display-processor p)))
		processor-list)))

      (define results (lambda ()
          (list (cons 'value               *final-value*)
		(cons 'sweeps              *sweep-count*)
		(cons 'tot-instructions  *total-instruction-count*)
		(cons 'processors          *processor-count*)
		(cons 'max-parallelism     *max-parallelism*)
		(cons 'avg-parallelism     (/ *total-instruction-count* *sweep-count*))
		)))

      (define record-value (lambda (v)
          (set! *final-value* v)
	  (princ "Value: ")
	  (user-print v)
	  (princ "; sweep: ")
	  (princ *sweep-count*)))

      (define sweep (lambda ()
	  (let
	        ((n (length processor-list))
	         (new-processor-list (threaded-map execute-instruction
						   processor-list)))

	    (set! *total-instruction-count* (+ *total-instruction-count* n))
	    (set! *max-parallelism* (max *max-parallelism* n))
	    (set! *sweep-count* (1+ *sweep-count*))
	    (set! *parallelism-profile* (cons (cons *sweep-count* n) *parallelism-profile*))

	    (set! processor-list new-processor-list)

	    (if (zero? (remainder *sweep-count* 10))
		(princ "."))
	    (if *trace* (display)))))

      (define run (lambda ()
	  (if (null? processor-list)
	      (results)
	      (sequence
		(sweep)
		(run)))))

      (define run-with-pauses (lambda (n)
          (set! n (- n 1))
	  (if (null? processor-list)
	      (results)
	      (sequence
		(if (<= n 0)
		    (sequence
		      (newline)
		      (princ "Pause : ")
		      (set! n (read))))
		(sweep)
		(run-with-pauses n)))))

      (define initialize-with-expr (lambda (expr)
          (init-stats)
	  (set! *main-processor* (make-processor))
	  (set! processor-list (list *main-processor*))
          (set! *final-value* 'unknown)
	  (set-register-contents! (get-processor-register *main-processor* 'stack)
				  (list TOP-LEVEL-EVAL expr the-empty-environment))
	  (set-register-contents! (get-processor-register *main-processor* 'program-counter)
				  *start*)
	  'initialized))

    )))


(define make-multiprocessor (lambda (controller-code)
  (let
        ((mp (make-raw-multiprocessor)))
    (build-instruction-list mp (cons '*start* controller-code))
    mp)))

(define build-instruction-list (lambda (machine op-list)
    (if (null? op-list)
	nil
	(let
	      ((rest-of-instructions
		 (build-instruction-list machine (cdr op-list))))
	  (if (label? (car op-list))
	      (sequence
                  (declare-label machine
				 (car op-list)
				 rest-of-instructions)
		  rest-of-instructions)
	      (cons (car op-list)
		    rest-of-instructions))))))

(define label? (lambda (expression) (symbol? expression)))

(define declare-label (lambda (machine label instruction-sequence)
    (let
          ((defined-labels (eval '*labels*  machine)))
      (if (memq label defined-labels)
          (error "Multiply-defined label" label)
	  (sequence
              (remote-define machine label instruction-sequence)
	      (remote-set machine
			  '*labels*
			  (cons label defined-labels)))))))

(define remote-define (lambda (machine variable value)
    (eval (list 'define variable (list 'quote value))
	  machine)))

(define remote-set (lambda (machine variable value)
    (eval (list 'set! variable (list 'quote value))
	  machine)))

(define load-mp (lambda (mp expr)
    (eval (list 'initialize-with-expr (list 'quote expr)) mp)))

(define run-mp (lambda (mp)
    (eval '(run) mp)))

(define run-mp-with-pauses (lambda (mp n)
    (eval (list 'run-with-pauses n) mp)))

;;; ****************************************************************
;;; Multiprocessor user interface.

;;; Load a multiprocessor with an expression and run it.

(define load-and-go (lambda (mp exp)
    (newline)
    (load-mp mp (pre-process exp))
    (run-mp mp)))

;;; Load a multiprocessor with an expression and run it, pausing after each
;;; sweep.  At each sweep, you get a ``Pause:'' message.  Type a number N,
;;; to continue for N more sweeps before pausing again.

(define load-and-go-with-pauses (lambda (mp exp)
    (newline)
    (load-mp mp (pre-process exp))
    (run-mp-with-pauses mp 1)))

;;; Show the result of a run, i.e., the value of the evaluated expression,
;;; after converting it back into Scheme forms.  This is because cons-cells
;;; produced in the parallel machine have a slightly different representation
;;; than raw Scheme cons-cells.  (e.g., they have synchronization flags,
;;; etc.).

(define show-value (lambda (mp)
    (newline)
    (user-print (convert-back (eval '*final-value* mp)))))

;;; Extract the parallelism profile from the multiprocessor.

(define get-pp (lambda (mp)
    (reverse (eval '*parallelism-profile* mp))))

;;; Show the parallelism profile stored in a multiprocessor (i.e., from its last run)
;;; averaging the parallelism profiles in STEPSIZE sweeps together.

(define show-pp (lambda (mp stepsize)
  (define loop (lambda (tot pp)
    (if (null? pp)
	'done
	(let
	      ((sweep (caar pp))
	       (n     (cdar pp)))
	  (if (zero? (remainder sweep stepsize))
	      (sequence
		(newline)
		(princ sweep)
		(princ " : ")
		(print-bar (round (/ (+ tot n) stepsize)))
		(loop 0 (cdr pp)))
	      (loop (+ tot n) (cdr pp)))))))
  (princ "Parallelism profile : ")
  (loop 0 (get-pp mp))))

(define print-bar (lambda (n)
   (if (zero? n)
       'done
       (sequence (princ "*") (print-bar (- n 1))))))

;;; ****************************************************************
