;;; 6.001, Nikhil, November 24, 1988
;;; Auxiliary definitions for simulation of explicit-control evaluators

;;; ****************************************************************
(define user-print (lambda (v)
  (if (compound-procedure? v)
      (print-compound-procedure v)
      (princ v))))

;;; ****************************************************************
;;; For EVAL-DISPATCH

(define self-evaluating? (lambda (exp)
  (or (number? exp) (null? exp))))

(define global-variable? (lambda (exp)
  (symbol? exp)))

(define define? (lambda (exp)
  (if (atom? exp)
      nil
      (eq? (car exp) 'define))))

(define quoted? (lambda (exp)
  (if (atom? exp)
      nil
      (eq? (car exp) 'quote))))

(define lookup? (lambda (exp)
  (if (atom? exp)
      nil
      (eq? (car exp) 'lookup))))

(define lambda? (lambda (exp)
  (if (atom? exp)
      nil
      (eq? (car exp) 'lambda))))

(define letrec? (lambda (exp)
  (if (atom? exp)
      nil
      (eq? (car exp) 'letrec))))

(define if? (lambda (exp)
  (if (atom? exp)
      nil
      (eq? (car exp) 'if))))

(define application? (lambda (exp)
  (not (atom? exp))))

;;; ****************************************************************
;;; For EV-GLOBAL and EV-DEFINE

(define lookup-global (lambda (symbol)
  (let
        ((x (assq symbol *the-global-environment*)))
    (if (null? x)
	(error "LOOKUP-GLOBAL: unbound symbol" symbol)
	(cdr x)))))

(define define-global! (lambda (symbol val)
  (let
        ((x (assq symbol *the-global-environment*)))
    (if (null? x)
	(set! *the-global-environment* (cons (cons symbol val) *the-global-environment*))
	(set-cdr! x val)))))

(define *the-global-environment*
  (list
    (cons 'nil      '())
    (cons 'car      '(primitive car))
    (cons 'cdr      '(primitive cdr))
    (cons 'cons     '(primitive cons))

    (cons 'atom?    '(primitive atom?))
    (cons 'null?    '(primitive null?))
    (cons 'truncate '(primitive truncate))

    (cons 'eq?      '(primitive eq?))
    (cons '+        '(primitive +))
    (cons '-        '(primitive -))
    (cons '*        '(primitive *))
    (cons '/        '(primitive /))
    (cons '<        '(primitive <))
    (cons '>        '(primitive >))
    (cons '=        '(primitive =))
  ))

(define the-empty-environment nil)

;;; ****************************************************************
;;; For EV-LOOKUP

(define find-frame-cell (lambda (exp env)
    (let
        ((i (cadr exp))
	 (j (caddr exp)))
      (var-in-frame j (frame-in-env i env)))))

(define frame-in-env (lambda (i env)
    (list-ref env i)))

(define var-in-frame (lambda (j frame)
    (list-ref frame (- (length frame) j 1))))

;;;representing environments

(define first-frame (lambda (env) (car env)))
(define rest-frames (lambda (env) (cdr env)))

;;; ****************************************************************
;;; For EV-LAMBDA
(define make-procedure (lambda (lambda-exp env)
  (list 'procedure lambda-exp env)))

;;; ****************************************************************
;;; For EV-APPLICATION

(define the-empty-stack nil)

;;; ****************************************************************
;;; For APPLY-DISPATCH

(define primitive-procedure? (lambda (proc)
  (if (atom? proc)
      nil
      (eq? (car proc) 'primitive))))

;;; Representation of compound procedures:
;;;     (lambda <body> <env>)
;;; Note that there is no formal-parameter list, because lambdas
;;; have been processed to use lexical addressing.

(define compound-procedure? (lambda (proc)
  (if (atom? proc)
      nil
      (eq? (car proc) 'procedure))))

(define print-compound-procedure (lambda (proc)
  (princ (list 'procedure
               (cadr proc)
               '<procedure-env>))))

;;; ****************************************************************
;;; For PRIMITIVE-APPLY

(define car?      (lambda (proc) (eq? (cadr proc) 'car)))
(define cdr?      (lambda (proc) (eq? (cadr proc) 'cdr)))
(define cons?     (lambda (proc) (eq? (cadr proc) 'cons)))

(define atom??    (lambda (proc) (eq? (cadr proc) 'atom?)))
(define null??    (lambda (proc) (eq? (cadr proc) 'null?)))
(define truncate? (lambda (proc) (eq? (cadr proc) 'truncate)))

(define eq??      (lambda (proc) (eq? (cadr proc) 'eq?)))
(define +?        (lambda (proc) (eq? (cadr proc) '+)))
(define -?        (lambda (proc) (eq? (cadr proc) '-)))
(define *?        (lambda (proc) (eq? (cadr proc) '*)))
(define /?        (lambda (proc) (eq? (cadr proc) '/)))
(define <?        (lambda (proc) (eq? (cadr proc) '<)))
(define >?        (lambda (proc) (eq? (cadr proc) '>)))
(define =?        (lambda (proc) (eq? (cadr proc) '=)))

;;; ****************************************************************
;;; For COMPOUND-APPLY

(define procedure-environment (lambda (proc)
  (caddr proc)))

(define procedure-body (lambda (proc)
  (cdr (cadr proc))))

(define adjoin-frame (lambda (frame env) (cons frame env)))

;;; ****************************************************************
;;; ``Frame cells'' for strict interpreter.

(define make-frame-cell (lambda () (cons 'full nil)))

(define frame-cell-value (lambda (f-cell)
  (cdr f-cell)))

(define set-frame-cell-value! (lambda (f-cell v)
  (set-cdr! f-cell v)))

(define decr-arg-count-and-fetch! (lambda (f-cell)
  (let
        ((n (car f-cell)))
    (set-car! f-cell (- n 1))
    (- n 1))))

;;; ****************************************************************
;;; ``I-structure cells'' for non-strict interpreter.

(define make-I-cell (lambda () (cons 'empty nil)))

(define I-cell-flag-set? (lambda (i-cell)
  (eq? (car i-cell) 'full)))

(define set-I-cell-flag! (lambda (i-cell)
  (set-car! i-cell 'full)))

(define I-cell-value (lambda (i-cell)
  (cdr i-cell)))

(define set-I-cell-value! (lambda (i-cell v)
  (if (eq? (car i-cell) 'empty)
      (set-cdr! i-cell v)
      (error "SET-I-CELL-VALUE!: multiple writes : " i-cell v))))

(define I-cell-waiting-list (lambda (i-cell)
  (cdr i-cell)))

(define add-to-I-cell-waiting-list (lambda (i-cell p)
  (set-cdr! i-cell (cons p (cdr i-cell)))))

;;; ****************************************************************
;;; Converting expressions with ``frame cells'' or ``I-cells'' back into
;;; ordinary expressions.

(define convert-back (lambda (term)
  (cond
    ((not (pair? term)) term)
    ((eq? (car term) 'full) (convert-back (cdr term)))
    (else (cons (convert-back (car term))
		(convert-back (cdr term)))))))

;;; ****************************************************************
;;; Pre-processor:
;;;   * Converts (DEFINE (f ..xs) bs) => (DEFINE f (LAMBDA (xs) bs))
;;;   * Move quotes inward onto atoms
;;;   * Convert LETs to LAMBDAs
;;;   * Convert lambda-bound variables into (LOOKUP i j) forms

(define pre-process (lambda (exp)
  (if (define? exp)
      (pre-process-define exp)
      (pre-process-expression exp nil))))

;;; Convert (DEFINE (f x1..xN) b1..bM)
;;; to      (DEFINE f (lambda (x1..xN) b1 .. bM))
;;; and pre-process b1..bM

(define pre-process-define (lambda (def)
  (if (pair? (cadr def))
      (list (car def)                    ; DEFINE
	    (caadr def)                  ; f
	    (pre-process-expression
	      (cons 'lambda              ; (LAMBDA
		    (cons (cdadr def)    ;   (x1..xN)
			  (cddr def)))   ;   b1..bM)
	      nil))
      (list (car def)                    ; DEFINE
	    (cadr def)                   ; f
	    (pre-process-expression
	      (caddr def)                ; e
	      nil)))))

(define pre-process-expression (lambda (exp env)
  (cond
    ((self-evaluating? exp) exp)
    ((symbol? exp) (find-variable exp env 0))
    ((quoted? exp) (convert-quoted (cadr exp)))
    ((lambda? exp) (cons (car exp)
			 (pre-process-sequence (cddr exp)
					       (cons (cadr exp) env))))
    ((let? exp) (pre-process-expression (convert-let-to-lambda exp)
					env))
    ((letrec? exp) (pre-process-letrec (cadr exp) (cddr exp) env))
    ((if? exp) (list (car exp)
		     (pre-process-expression (cadr exp) env)
		     (pre-process-expression (caddr exp) env)
		     (pre-process-expression (cadddr exp) env)))
    ((define? exp)
          (error "PRE-PROCESS-EXPRESSION: Internal DEFINEs not allowed:"
		 exp))
    ((application? exp) (mapcar (lambda (e) (pre-process-expression e env))
				exp))
    (else (error "PRE-PROCESS-EXPRESSION: Unknown expression type:"
		 exp)))))

(define pre-process-sequence (lambda (seq env)
    (if (null? seq)
	nil
        (cons (pre-process-expression (car seq) env)
	      (pre-process-sequence (cdr seq) env)))))

(define pre-process-letrec (lambda (bindings bodys env)
    (let
          ((names (mapcar car bindings))
	   (exprs (mapcar cadr bindings)))
      (let
	    ((new-env (cons names env)))
	(cons 'letrec
	      (cons (mapcar (lambda (e) (pre-process-expression e new-env))
			    exprs)
		    (pre-process-sequence bodys new-env)))))))

(define find-variable (lambda (symbol env i)
  (if (null? env)
      symbol
      (let
            ((j (frame-lookup symbol (car env) 0)))
	(if (null? j)
	    (find-variable symbol (cdr env) (1+ i))
	    (list 'lookup i j))))))

(define frame-lookup (lambda (symbol frame j)
  (cond
    ((null? frame) nil)
    ((eq? symbol (car frame)) j)
    (else (frame-lookup symbol (cdr frame) (1+ j))))))

(define let? (lambda (exp)
  (if (atom? exp)
      nil
      (eq? (car exp) 'let))))

(define convert-let-to-lambda (lambda (exp)
    (cons (cons 'lambda
		(cons (mapcar car (cadr exp))
		      (cddr exp)))
	  (mapcar cadr (cadr exp)))))

(define convert-quoted (lambda (exp)
    (cond
      ((null? exp) (list 'quote nil))
      ((number? exp) exp)
      ((atom? exp) (list 'quote exp))
      (else (list 'cons (convert-quoted (car exp))
		        (convert-quoted (cdr exp)))))))

;;; ****************************************************************
