
;;; INITIALIZATION AND DRIVER LOOP

;;; To start the metacircular evaluator, call INITIALIZE-EVALUATOR.
;;; This initializes the global environment, and starts the DRIVER-LOOP.
;;; Use INITIALIZE-EVALUATOR instead of DRIVER-LOOP if you want
;;; to erase any definitions you have accumulated and start fresh
;;; with a clean global environment


(define (initialize-evaluator)
  (set! the-global-environment (setup-environment))
  (driver-loop))

;;; The driver loop reads an expression, evaluates it in the global
;;; environment, and prints the result.  Note that the driver
;;; uses a prompt of "MC-EVAL==> " to help you avoid confusing typing to the
;;; metacircular evaluator with typing to the underlying SCHEME interpreter.
;;; (Also, we have called our evaluator procedures MINI-EVAL and MINI-APPLY
;;; to help you avoid confusing them with Scheme's EVAL and APPLY.)

;;; When/If your interaction with the evaluator bombs out in an error,
;;; you should restart it by calling DRIVER-LOOP again.

(define (driver-loop)
  (newline)
  (princ "MC-EVAL==> ")
  (user-print (mini-eval (read) the-global-environment))
  (driver-loop))
 
;;; We use a special PRINT here, which avoids printing the environment
;;; part of a compound procedure, since the latter is a very long (or
;;; even circular) list.

(define (user-print object)
  (cond ((compound-procedure? object)
         (print (list 'compound-procedure
                      (parameters object)
                      (procedure-body object)
                      '[procedure-env])))
        (else (print object))))

;;; THE CORE OF THE EVALUATOR -- from section 4.1.1

(define (mini-eval exp env)
  (cond ((self-evaluating? exp) exp)
        ((quoted? exp) (text-of-quotation exp))
        ((variable? exp) (lookup-variable-value exp env))
        ((definition? exp) (eval-definition exp env))
        ((assignment? exp) (eval-assignment exp env))
        ((lambda? exp) (make-procedure exp env))
        ((conditional? exp) (eval-cond (clauses exp) env))
        ((application? exp)
         (mini-apply (mini-eval (operator exp) env)
                (list-of-values (operands exp) env)))
        (else (error "Unknown expression type -- EVAL" exp))))


(define (mini-apply procedure arguments)
  (cond ((primitive-procedure? procedure)
         (apply-primitive-procedure procedure arguments))
        ((compound-procedure? procedure)
         (eval-sequence (procedure-body procedure)
                        (extend-environment
                         (parameters procedure)
                         arguments
                         (procedure-environment procedure))))
        (else
         (error "Unknown procedure type -- APPLY" procedure))))
