Object subclass: #MagnifyingGlass
	instanceVariableNames: 'radius magnification magnifierForm largeBlackHole backgroundForm mergedForm backgroundRectangle displayRectangle mergedRectangle relativeBackgroundRectangle relativeDisplayRectangle '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'JOOP Vol 1 #3, Aug 88'!
MagnifyingGlass comment:
'	A magnifying glass (magnifier for short) of specified radius and magnification can be created and subsequently activated. When activated, the cursor is replaced by the magnifying glass. As long as the red (standard) mouse button is depressed, the area under the magnifying glass is magnified and displayed over it. Depressing the yellow (option+mouse) button deactivates the magnifying glass and restores the cursor to what it was. Note that the magnifier has a round (as opposed to square) glass.

	For speed, two forms are initialized at creation time: (1) magnifierForm which contains the magnifier icon and (2) largeBlackHole, a circular mask to capture the portion of the magnified picture to be displayed.'!


!MagnifyingGlass class methodsFor: 'instance creation'!

new
	"Creates a default magnifying glass."
	^self radius: 20 magnification: 4!

radius: anInteger magnification: anotherInteger
	"Creates a magnifying glass of a specified size and magnification."
	^super new radius: anInteger magnification: anotherInteger! !

!MagnifyingGlass class methodsFor: 'examples'!

example
	"MagnifyingGlass new activate."
	"(MagnifyingGlass radius: 40 magnification: 5) activate."! !
!MagnifyingGlass methodsFor: 'instance initialization'!

radius: anInteger magnification: anotherInteger
	"Initializes the magnifying glass instance."

	"Save the radius and magnification."
	radius _ anInteger. 
	magnification _ anotherInteger. 

	"Create the utility forms."
	self makeMagnifierForm.
	self makeLargeBlackHole.
	self makeBackgroundAndMergedForms! !

!MagnifyingGlass methodsFor: 'instance activation'!

activate
	"Make the magnifier track the mouse and display a magnified version of the area underneath it if the mouse button is depressed; otherwise, display the magnifier itself. Depressing the yellow button (option + mouse down) deactivates the mouse and restores everything to its former state."
	| magnifying newCenter |

	"Initialize."
	magnifying _ false. backgroundRectangle _ Sensor cursorPoint extent: 0@0.

	"Hide the cursor."
	Cursor blank showWhile:[
	
		"Quit when a yellow button is pressed."
		[Sensor yellowButtonPressed] whileFalse: [
			
			"Determine if magnification status has changed."
			magnifying _ Sensor redButtonPressed.
			
			"Display either the magnifier or the magnified area."
			newCenter _ Sensor cursorPoint.
			magnifying
				ifTrue: [
					displayRectangle _ newCenter - (radius@radius*magnification)
						extent: largeBlackHole extent. 
					self restoreBackgroundAndDisplayUsing: 
						#displayMagnifiedAreaOnMergedForm]
				ifFalse: [
					displayRectangle _ newCenter - (radius@radius)
						extent: magnifierForm extent.
					self restoreBackgroundAndDisplayUsing:
						#displayMagnifierOnMergedForm]].

		self restoreBackgroundAndDisplayUsing: #isNil]! !

!MagnifyingGlass methodsFor: 'displaying'!

displayMagnifiedAreaOnMergedForm
	"In-place Magnifies a circular area centered at newCenter in aForm."	
	| imageRectangle imageForm magnifiedImage dot |

	"First, obtain a magnified square image of the area."
	imageRectangle _ 
		relativeDisplayRectangle center - (radius@radius) extent: (radius@radius)*2.
	imageForm _ Form new extent: imageRectangle extent. 
	imageForm copyBits: imageRectangle from: mergedForm at: 0@0
		clippingBox: (0@0 extent: imageRectangle extent) rule: Form over mask: nil.
 
	magnifiedImage _ imageForm magnifyBy: magnification@magnification.
	
	"Second, AND it with the black hole."
	largeBlackHole displayOn: magnifiedImage at: 0@0 
		clippingBox: magnifiedImage boundingBox rule: Form and mask: nil.
	
	"Third, create a white hole in the magnified area (erase changes black to white)."
	largeBlackHole displayOn: mergedForm at: relativeDisplayRectangle origin
		clippingBox: relativeDisplayRectangle rule: Form erase mask: nil.
	
	"Fourth, OR the magnified image onto the magnification area containing the white hole."
	magnifiedImage displayOn: mergedForm at: relativeDisplayRectangle origin 
clippingBox: relativeDisplayRectangle rule: Form under mask: nil.
	
	"Fifth, draw a circle around the magnified section to make it stand out."
	dot _ (Form extent: 3@3) black.
	(Circle new form: dot; radius: radius * magnification - 1; 
		center: relativeDisplayRectangle center - (1@1)) 
			displayOn: mergedForm at: 0@0 clippingBox: relativeDisplayRectangle
				rule: Form over mask: nil!

displayMagnifierOnMergedForm
	magnifierForm 
		displayOn: mergedForm at: relativeDisplayRectangle origin
		clippingBox: relativeDisplayRectangle rule: Form under mask: nil!

restoreBackgroundAndDisplayUsing: displayOnMergedFormSymbol 
	"If the background form were restored in one step followed by a display of the new information (either the magnifier or a circular magnification area) in a second step, flickering would occur when the two areas overlap. We can avoid this by first displaying the information on a temporary form called the merged form and then displaying this merged form on the screen using one display message."

	"First, obtain a copy of the screen that contains both the background and display areas. Note: if the background and/or display forms partially reside off the screen, we may need a larger merged form to avoid information loss by clipping."

		mergedRectangle _ backgroundRectangle merge: displayRectangle.

		mergedRectangle extent x > mergedForm extent x | 
		(mergedRectangle extent y > mergedForm extent y) 
			ifTrue: [mergedForm _ 
				Form new extent: (mergedRectangle extent max: mergedForm extent)].

		mergedForm copyBits: mergedRectangle from: Display at: 0 @ 0
			clippingBox: (0 @ 0 extent: mergedRectangle extent) rule: Form over mask: nil.

	"Second, compute background and display rectangles relative to the merged form origin."
relativeBackgroundRectangle _ 
	backgroundRectangle translateBy: 0 @ 0 - mergedRectangle origin.

relativeDisplayRectangle _ 
	displayRectangle translateBy: 0 @ 0 - mergedRectangle origin.

	"Third, restore the merged form to what it used to be and save the display area for later use."
		backgroundForm displayOn: mergedForm 
			at: relativeBackgroundRectangle origin
			clippingBox: relativeBackgroundRectangle 
			rule: Form over mask: nil.
		backgroundForm
			copyBits: relativeDisplayRectangle from: mergedForm 
			at: 0@0 clippingBox: (0 @ 0 extent: relativeDisplayRectangle extent)
			rule: Form over mask: nil.
		backgroundRectangle _ displayRectangle.

	"Fourth, display the display form onto the merged form (in-place modify)."
	self perform: displayOnMergedFormSymbol.

	"Finally, display the merged form onto the screen."
	mergedForm displayOn: Display at: mergedRectangle origin
		clippingBox: mergedRectangle rule: Form over mask: nil! !

!MagnifyingGlass methodsFor: 'private form initialization'!

makeBackgroundAndMergedForms
	"Creates forms for keeping track of the background and merged forms."
	| size extent |

	size _ radius * magnification * 2 max: radius * 3. 
	extent _ size@size.
	backgroundForm _ Form new extent: extent.
	mergedForm _ Form new extent: Display extent!

makeLargeBlackHole
	"Creates a black hole (a large black dot) and save it in the instance."
	| magnifiedDiameter |

	magnifiedDiameter _ radius * magnification * 2. 
	largeBlackHole _ (Form dotOfSize: magnifiedDiameter) offset: 0@0!

makeMagnifierForm
	"Creates and initializes the form containing the magnifying glass and handle."
	| diameter center dot |

	"Compute often used values."
	diameter _ 2*radius. center _ radius@radius.
	magnifierForm _ Form new extent: (radius@radius)*3.
	dot _ (Form extent: 3@3) black.

	"Draw a solid black handle."
	(Line from: center to: center*4 "off the form!!!!" 
		withForm: (Form new extent: (radius // 4) @ (radius // 4); black)) 
			displayOn: magnifierForm.

	"Draw a solid white circle over the top left part of the handle."
	(Form dotOfSize: 2 * radius + 1) 
		offset: 0@0; displayOn: magnifierForm at: 0@0 rule: Form erase.

	"Create the circle for the magnifying glass."
	(Circle new form: dot; radius: radius; center: center) displayOn: magnifierForm.

	"Create the horizontal line across the magnifying glass."
	(Line from: 0@radius to: diameter@radius withForm: dot) displayOn: magnifierForm.

	"Create the vertical line across the magnifying glass."
	(Line from: radius@0 to: radius@diameter withForm: dot) displayOn: magnifierForm! !
