;;;
;;;   P R O D U C T I O N S   T O O L B O X
;;;   = = = = = = = = = = =   = = = = = = =
;;;
;;; Depends on the following toolboxes:
;;;    Systems
;;;    Patterns
;;;    KnowledgeBases
;;;
;;; This toolbox contains routines to implement Production Systems.
;;; This toolbox is built upon the pattern matching and KB toolboxes.
;;;
;;; Knowledge-based systems store information as a collection of facts
;;; and rules. The package implements a simple forward chaining
;;; strategy for reasoning with rule-based knowledge bases implemented
;;; through production systems

; Data Structures
; ==== ==========
;

;  MEMORY             - a list of facts ("working memory" WM) or
;                       rules ("production memory" PM) implemented
;                       as a KB data type from the KB toolbox
;  FACT               - a list asserting a state of affairs in some world
;  RULE               - a list of the form:
;                       (Identifier aConditionList aConclusionList)
;  NEGATED-CONDITION  - a list of the form (~ PATTERN)
;  CONDITION-ELEMENT  - either a PATTERN or a NEGATED-CONDITION
;  CONDITION-LIST     - a list of CONDITION-ELEMENTs
;  ASSERTION          - a list (ASSERT fact)
;  RETRACTION         - a list (RETRACT fact)
;  EXECUTION          - a list (EXECUTE procedure parameters)
;  RETURN             - a list (RETURN procedure parameters)
;  CONCLUSION-ELEMENT - an assertion, retraction, execution or return
;  CONCLUSION-LIST    - a list of CONCLUSION-ELEMENTs

; ========== fns on MEMORY ==================================
; the WM is a list of facts and the PM is a list of rules. The
; procedures from the KB toolbox can be used to create and manipulate
; these

(define MakeWM MakeKB)

(define MakePM MakeKB)

(define PrintWM (lambda (aWM) (PrintKB aWM PrintFact)))

(define PrintPM (lambda (aPM) (PrintKB aPM PrintRule)))

; ========== fns on FACTs ===================================
; the facts that make up the WM are lists of unspecified structure

(define PrintFact (lambda (aFact) (DisplayList "Fact:" aFact)))

; ========== fns on RULEs ===================================
; a list of the form (identifier conditions conclusions)

(define RULE
  (lambda (aName aConditionList aConclusionList)
    (list aName aConditionList aConclusionList)))

(define GetName (lambda (aRule) (car aRule)))

(define GetConditionList (lambda (aRule) (cadr aRule)))

(define GetConclusionList (lambda (aRule) (caddr aRule)))

(define PrintRule
  (lambda (aRule) 
    (DisplayLine "+++ Rule:" (GetName aRule))
    (display "  Conditions : ")
    (PrintConditions (GetConditionList aRule))
    (newline)
    (display "  Conclusions: ")
    (PrintConclusions (GetConclusionList aRule))
    (newline) (newline)))

; ========= fns on NEGATED-CONDITIONs ======================
; lists of the form (~ pattern)

(define ~ (lambda (aPattern) (list '~ aPattern)))

(define GetNegatedPattern (lambda (aCondition) (cadr aCondition)))

(define NegatedCondition?
  (lambda (aCondition)
    ; is the condition of the form (~ pattern)
    (and (not (null? aCondition))
         (eq? (car aCondition) '~))))

; ========= fns on CONDITION-LISTs ==========================
; lists of condition elements. Each condition element is a pattern
; data type from the pattern toolbox

(define CONDITIONS (lambda conditionList conditionList))

(define CurrentCondition (lambda (aConditionList) (car aConditionList)))

(define RestOfConditions (lambda (aConditionList) (cdr aConditionList)))

(define NoMoreConditions? (lambda (aConditionList) (null? aConditionList)))

(define PrintConditions
  (lambda (aConditionList)
    (if (null? aConditionList)
        #f
        (begin (PrintPattern (car aConditionList))
               (display " ")
               (PrintConditions (cdr aConditionList))))))


; ========= fns on ASSERTIONs ===============================
; a list of the form: (ASSERT a-fact)

(define ASSERT (lambda (aFact) (list 'ASSERT aFact)))

(define GetAssertion
  (lambda (anAssertionElement) (cadr anAssertionElement)))

(define PrintAssertion
  (lambda (anAssertion)
    (display "(ASSERT ")
    (PrintPattern (GetAssertion anAssertion))
    (display ")")))

; ========= fns on RETRACTIONs ==============================
; a list of the form: (RETRACT a-fact)

(define RETRACT (lambda (aFact) (list 'RETRACT aFact)))

(define GetRetraction
  (lambda (anRetractionElement) (cadr anRetractionElement)))

(define PrintRetraction
  (lambda (aRetraction)
    (display "(RETRACT ")
    (PrintPattern (GetRetraction aRetraction))
    (display ")")))

; ========= fns on EXECUTIONs ===============================
; a list of the form: (EXECUTE aProc aParameterList)

(define EXECUTE
  (lambda (aProc . aParameterList)
    (list 'EXECUTE aProc aParameterList)))

(define GetProc (lambda (anExecution) (cadr anExecution)))

(define GetParameters (lambda (anExecution) (caddr anExecution)))

(define PrintExecution
  (lambda (anExecution)
    (DisplayList "(EXECUTE"
                 (GetProc anExecution)
                 (GetParameters anExecution)
                 ")")))
    
; ========= fns on RETURNs ====================================
; a list of the form: (RETURN aProc aParameterList)

(define RETURN
  (lambda (aProc . aParameterList)
    (list 'RETURN aProc aParameterList)))

(define PrintReturn
  (lambda (aReturn)
    (DisplayList "(RETURN"
                 (GetProc aReturn)
                 (GetParameters aReturn)
                 ")")))

; ========= fns on CONCLUSIONs ==============================
; either an ASSERTION, a RETRACTION, an EXECUTION or RETURN

(define Assertion?
  (lambda (aConclusionElement)
    (eq? (car aConclusionElement) 'ASSERT)))

(define Retraction?
  (lambda (aConclusionElement)
    (eq? (car aConclusionElement) 'RETRACT)))

(define Execution?
  (lambda (aConclusionElement)
    (eq? (car aConclusionElement) 'EXECUTE)))

(define Return?
  (lambda (aConclusionElement)
    (eq? (car aConclusionElement) 'RETURN)))

(define PrintConclusion
  (lambda (aConclusionElement)
    (cond ((Assertion? aConclusionElement)
           (PrintAssertion aConclusionElement))
          ((Retraction? aConclusionElement)
           (PrintRetraction aConclusionElement))
          ((Execution? aConclusionElement)
           (PrintExecution aConclusionElement))
          ((Return? aConclusionElement)
           (PrintReturn aConclusionElement))
          (else
            (Fatal-Error "PrintConclustion:"
                         "Bad conclusion"
                         aConclusionElement)))))

; ========= fns on CONCLUSION-LISTs =========================
; lists of CONCLUSIONs

(define CONCLUSIONS (lambda conclusionList conclusionList))

(define CurrentConclusion
  (lambda (aConclusionList) (car aConclusionList)))

(define RestOfConclusions
  (lambda (aConclusionList) (cdr aConclusionList)))

(define NoMoreConclusions?
  (lambda (aConclusionList) (null? aConclusionList)))

(define PrintConclusions
  (lambda (aConclusionList)
    (if (null? aConclusionList)
        #f
        (begin (PrintConclusion (car aConclusionList))
               (display " ")
               (PrintConclusions (cdr aConclusionList))))))

; ========= control fns on Deduction Machine ================

; The "forward-chaining" problem solver  uses the following strategy:
;
; Data structures: WM remembers facts,
;                  PM remembers rules.
;
;  A. FOR EACH rule in PM
;
;     (1) Match the first CONDITION in the rule against all facts
;         in the WM using an initially empty association list to
;         yield RL, a RETREIVEDLIST of fact/aList pairs, one for
;         each successful match against the pattern.
;     (2) FOR EACH remaining CONDITION in a rule:
;             (a) If RL is empty then this rule doesn't match so
;                 try the next rule at step A
;             (b) Set RLNEW to be empty
;             (c) FOR EACH retrievedElement in RL
;                     (i) Match condition against all facts in WM
;                         with aList as the initial association
;                         list extracted from retrievedElement
;                    (ii) Append the resulting RETREIVEDLIST to
;                         RLNEW.
;             (d) Set RL to be RLNEW
;     (3) FOR EACH conclusion in the rule
;             (a) If the conclusion is an Assertion/Retraction 
;                 then FOR EACH retrievedElement in RL
;                     (i) Replace all (<-? x) matching elements in
;                         the conclusion using the binding from AList
;                         extracted from retrievedElement
;                    (ii) Add/Remove the fact to/from theWM
;             (b) If the conclusion is an Execution then
;                 FOR EACH retrievedElement in RL
;                     (i) Replace all (<-? x) matching elements in
;                         the parameters using the binding from AList
;                         extracted from retrievedElement
;                    (ii) Apply the procedure to the substituted
;                         parameters
;             (c) If the conclusion is an Return  then
;                 FOR ONLY THE FIRST retrievedElement in RL
;                     (i) Replace all (<-? x) matching elements in
;                         the parameters using the binding from AList
;                         extracted from retrievedElement
;                    (ii) Apply the procedure to a list of aWM plus
;                         the substituted parameters
;
;  B. If any of the rules in the above causes an ASSERTION or
;     RETRACTION to change the WM then go back to step A.
;
;  C. Return the updated value of WM
;

(define TestCondition
  (lambda (aCondition aWM anAList)
    ; Return a RETRIVEDLIST of successful matches against aWM of
    ; the pattern in aCondition. If aCondition is a NEGATEDCONDITION
    ; then return a non-empty RETRIVEDLIST only if there is no match
    ; against aWM for the pattern in aCondition.
    (let ((pattern (if (NegatedCondition? aCondition)
                       (GetNegatedPattern aCondition)
                       aCondition))
          (aRL #f))
      (set! aRL (RetrieveAllByPattern aWM
                                      pattern
                                      (lambda (x) x)
                                      (if (eq? anAList #t)
                                          ()
                                          anAList)))
      (if (NegatedCondition? aCondition)
          (if (EmptyRetrievedList? aRL)
              (MakeRetrievedList
                (MakeRetrievedElement 'NEGATED-FACT (if anAList anAList #t)))
              (MakeRetrievedList))
          aRL))))

(define TestConditions
  (lambda (aConditionList aWM)
    ; If all the conditions in aConditionList are satisfied by the
    ; facts in aWM then return an updated RETRIEVEDLIST containing an
    ; entry for each successful match using the ALISTs obtained from
    ; oldRL. For a NEGATED-CONDITION the match succeeds only if an
    ; empty RETRIEVEDLIST is returned from the match
    
    (define testConditionsAux
      (lambda (aConditionList oldRL)
        
        (if (or (NoMoreConditions? aConditionList)
                (EmptyRetrievedList? oldRL))
            oldRL
            (let ((newRL (MakeRetrievedList))
                  (condition (CurrentCondition aConditionList)))
              (do ((restOfRL oldRL (cdr restOfRL))
                   (aList #f))
                  ((EmptyRetrievedList? restOfRL) #f)
                  (set! aList (GetAList (CurrentRetrievedElement restOfRL)))
                  (set! newRL 
                        (CombineRetrievedLists
                          newRL
                          (TestCondition condition aWM aList))))
              (testConditionsAux (cdr aConditionList) newRL)))))
  
    ; deal with first condition specially to build the initial
    ; RETRIEVEDLIST and then use testConditionsAux to deal with
    ; any further conditions
    (if (NoMoreConditions? aConditionList)
        (MakeRetrievedList)
        (testConditionsAux (cdr aConditionList)
                           (TestCondition (car aConditionList)
                                          aWM
                                          (MakeAList))))))

(define ForwardChainer
  (lambda (aPM aWM . verboseFlag)
    (call-with-current-continuation
     (lambda (returnContinuation)
       ; Cycle through all the rules in aPM while there are resulting
       ; changes in aWM. If verboseFlag is absent it defaults to #f,
       ; if present with value #t then each deduction, retraction,
       ; execution and return is output. Return updated aWM
       
       (define performConclusions
         (lambda (ruleName aConclusionList anRL)
           ; return #t if any Assertion in aConclusionList causes
           ; aWM to be altered
           (let ((changedWM? #f))
             
             (define performAssertion!
               (lambda (anAssertion)
                 ; add fact to aWM for each aList substitution in anRL
                 (do ((restOfRL anRL (cdr restOfRL))
                      (aList #f)
                      (newFact #f))
                     ((EmptyRetrievedList? restOfRL) #f)
                     (set! aList (GetAList (CurrentRetrievedElement restOfRL)))
                     (set! newFact (MakeVariableSubst
                                     (GetAssertion anAssertion)
				     (if (eq? aList #t) #f aList)))
                     (if (KnownFact? newFact aWM)
                         #f
                         (begin
                           (set! changedWM? #t)
                           (if verboseFlag
			       (begin
                                  (DisplayList "Rule" ruleName)
				  (PrintAssertion anAssertion)
				  (newline)
				  (display ".... adds fact ")
				  (PrintPattern newFact)
				  (newline))
			       #f)
			   (set! aWM (AddFact aWM newFact)))))))
             
             (define performRetraction!
               (lambda (aRetraction)
                 ; remove fact from aWM for each aList substitution
                 ; in anRL
                 (do ((restOfRL anRL (cdr restOfRL))
                      (aList #f)
                      (oldFact #f))
                     ((EmptyRetrievedList? restOfRL) #f)
                     (set! aList (GetAList (CurrentRetrievedElement restOfRL)))
                     (set! oldFact (MakeVariableSubst
                                    (GetRetraction aRetraction)
                                    aList))
                     (if (not (KnownFact? oldFact aWM))
                         #f
                         (begin
                          (set! changedWM? #t)
                          (if verboseFlag
                              (begin
                               (DisplayList "Rule" ruleName)
                               (PrintRetraction aRetraction)
                               (newline)
                               (display ".... removes fact ")
                               (PrintPattern oldFact)
                               (newline))
                              #f)
                          (set! aWM (RemoveFact aWM oldFact)))))))
             
             (define substParams
               (lambda (aParamList anAList)
                 ; substitute for (<-? x) type elements in possible PATTERNs
                 (map (lambda (aParam)
                        (if (pair? aParam)
                            (MakeVariableSubst aParam anAList)
                            aParam))
                      aParamList)))
                          
             (define performExecution
               (lambda (anExecution)
                 ; call the procedure for each AList with substituted
                 ; parameters
                 (if verboseFlag
                     (begin (DisplayList "Rule" ruleName)
                            (PrintExecution anExecution) (newline))
                     #f)
                 (do ((restOfRL anRL (cdr restOfRL))
                      (aList #f)
                      (newParams #f))
                     ((EmptyRetrievedList? restOfRL) #f)
                     (set! aList (GetAList (CurrentRetrievedElement restOfRL)))
                     (set! newParams
                           (substParams
                            (GetParameters anExecution)
                            (if (eq? aList #t) #f aList)))
                     (apply (GetProc anExecution) newParams))))
             
             (define performReturn
               (lambda (aReturn)
                 ; call the procedure with parameters aWM and AList as
                 ; substituted parameters. Use only the first
                 ; RetrievedElement if there is more than one. Return the
                 ; result from ForwardChainer
                 ; WARNING: this fn causes a jump out of the ForwardChainer
                 (let ((finalResult #f)
                       (aList #f)
                       (newParams #f))
                   (if verboseFlag
                       (begin (DisplayList "Rule" ruleName)
                              (PrintReturn aReturn) (newline))
                       #f)
                   (set! aList (GetAList (CurrentRetrievedElement anRL)))
                   (set! newParams
                         (substParams (GetParameters aReturn)
                                      (if (eq? aList #t) #f aList)))
                   (set! finalResult
                         (apply (GetProc aReturn)
                                (append (list aWM) newParams)))
                   (returnContinuation finalResult))))
             
             (do ((restOfConclusions aConclusionList
                                    (cdr restOfConclusions))
                  (conclusion #f))
                 ((NoMoreConclusions? restOfConclusions) changedWM?)
	         (set! conclusion (CurrentConclusion restOfConclusions))
                 (cond ((Assertion? conclusion)
                        (performAssertion! conclusion))
                       ((Retraction? conclusion)
                        (performRetraction! conclusion))
                       ((Execution? conclusion)
                        (performExecution conclusion))
                       ((Return? conclusion)
                        (performReturn conclusion))
                       (else
                        (Fatal-Error "ForwardChainer:"
                                     "Invalid conclusion"
                                     conclusion))))
             changedWM?)))
       
       (if (null? verboseFlag)
           #f
           (set! verboseFlag (car verboseFlag)))
       (do ((changedWM? #t))
           ((not changedWM?) aWM)
           (set! changedWM? #f)
           (do ((rulesToTry aPM (RestOfKB rulesToTry))
                (rule #f))
               ((EmptyKB? rulesToTry))
               (let* ((rule (FirstInKB rulesToTry))
                      (name (GetName rule))
                      (conditions (GetConditionList rule))
                      (conclusions (GetConclusionList rule))
                      (RL #f))
                 (set! RL (TestConditions conditions aWM))
                 (if RL
                     (set! changedWM?
                           (performConclusions name conclusions RL))
                     #f))))))))

(define FireRule
  (lambda (aRule aWM . verboseFlag)
    ; try a single rule by constructing a PM containing the
    ; rule and invoking ForwardChainer with aWM
    (if (null? verboseFlag)
        #f
        (set! verboseFlag (car verboseFlag)))
    (ForwardChainer (MakePM aRule) aWM verboseFlag)))
