;;;
;;;  A S S O C I A T I V E   N E T W O R K S   T O O L B O X
;;;  = = = = = = = = = = =   = = = = = = = =   = = = = = = =
;;;
;;; Depends on the following toolboxes:
;;;    Systems

; This toolbox supports Associative Networks (ANs) among concepts
; which may be linked. Special property inheritance links labelled
; by AKO (A Kind Of) are recognised and other user defined links may
; also be supplied. These links can be traversed by query functions.
; Simple "certainty" factors may optionally be attached to the user-
; defined links with values ranging from 0.0 (highly improbable) to
; 1.0 (virtually certain). Queries may then supply a threshold
; (value 0.0 to 1.0) specifying a minimum "strength" before which a
; path is considered to be present. Any factor that is not suppled
; defaults to 1.0.
 
; The following data types are used:

; AKO-SLOT  :a list of (AKO aListOfConceptNames)
; SLOT      :a list of (aLinkName aConceptName aCertaintyFactor)
; SLOT-LIST :a list of SLOTs
; CONCEPT   :a list of (aSymbol aSlotList)
; AN        :a list of CONCEPTs                                

; ========== fns on SLOTs ===================================
; either an AKO slot, a list of the form:
;        (AKO aConceptList)
; or a user defined slot, a list of the form
;        (aLinkName aConceptName aCertaintyFactor)
; or a list with an implied certainty factor of 1.0 of the form
;        (aLinkName aConceptName)

(define AKO (lambda aConceptList (list 'AKO aConceptList)))

(define LINK
  (lambda (aLinkName aConceptName . aCertaintyFactor)
    ; aCertaintyFactor defaults to 1.0 if not present and is not
    ; stored if 1.0
    (cond ((or (null? aCertaintyFactor) (= (car aCertaintyFactor) 1.0))
           (list aLinkName aConceptName))
          ((and (positive? (car aCertaintyFactor))
                (<= (car aCertaintyFactor) 1.0))
           (list aLinkName aConceptName (car aCertaintyFactor)))
          (else
           (Fatal-Error "LINK:" "Certainty factor" aCertaintyFactor
                        "not valid")))))

(define GetLinkName (lambda (aSlot) (car aSlot)))

(define AKOSlot?
  (lambda (aSlot)
    (if (pair? aSlot)
        (eq? 'AKO (GetLinkName aSlot))
        (Fatal-Error "AKOSlot?:" "Slot" aSlot "is illegal"))))

(define GetConceptList
  (lambda (anAKOSlot)
    (if (AKOSlot? anAKOSlot)
        (cadr anAKOSlot)
        (Fatal-Error "GetConceptList:" "Slot" anAKOSlot "not an AKO slot"))))

(define GetConceptName
  (lambda (aSlot)
    (if (AKOSlot? aSlot)
        (Fatal-Error "GetConceptName:" "Not valid on AKO slot" aSlot)
        (cadr aSlot))))

(define HasCertaintyFactor?
  (lambda (aSlot) (not (null? (cddr aSlot)))))

(define GetCertaintyFactor
  (lambda (aSlot)
    ; return 1.0 if has none
    (cond ((AKOSlot? aSlot)
           (Fatal-Error "GetCertaintyFactor:" "Not valid on AKO slot" aSlot))
          ((HasCertaintyFactor? aSlot)
           (caddr aSlot))
          (else 1.0))))

(define PrintANSlot
  (lambda (aSlot)
    (if (AKOSlot? aSlot)
        (DisplayList "(AKO" (GetConceptList aSlot) ")")
        (DisplayList "(LINK" (GetLinkName aSlot)
                             (GetConceptName aSlot)
                             (if (HasCertaintyFactor? aSlot)
                                 (GetCertaintyFactor aSlot)
                                 "")
                             ")"))))

; ========== fns on SLOT-LISTs ==============================
; lists of SLOTs

(define NoMoreSlots? (lambda (aSlotList) (null? aSlotList)))

(define FirstSlot (lambda (aSlotList) (car aSlotList)))

(define RestOfSlots (lambda (aSlotList) (cdr aSlotList)))

; ========== fns on CONCEPTs ================================
; list of the form (aConceptSymbol aSlotList)

(define CONCEPT
  (lambda (aConceptName . aSlotList) (list aConceptName aSlotList)))

(define MakeConceptWithList
  (lambda (aConceptName  aSlotList) (list aConceptName aSlotList)))

(define GetConceptSymbol (lambda (aConcept) (car aConcept)))

(define GetSlots (lambda (aConcept) (cadr aConcept)))

(define PrintConcept
  (lambda (aConcept)
    (display (GetConceptSymbol aConcept)) (display ": ")
    (for-each (lambda (aSlot) (PrintANSlot aSlot) (display " "))
              (GetSlots aConcept))))

; ========== fns on ANs =====================================
; lists of CONCEPTs 

(define MakeAN (lambda aConceptList aConceptList))

(define EmptyAN? (lambda (anAN) (null? anAN)))

(define FirstConcept (lambda (anAN) (car anAN)))

(define RestOfConcepts (lambda (anAN) (cdr anAN)))

(define AddConcept
  (lambda (anAN aConcept)
    (append (RemoveConcept anAN (GetConceptSymbol aConcept))
            (list aConcept))))

(define RemoveConcept
  (lambda (anAN aConceptSymbol)
    (cond ((EmptyAN? anAN) ())
          ((eq? aConceptSymbol
                (GetConceptSymbol (FirstConcept anAN)))
           (RestOfConcepts anAN))
          (else (cons (FirstConcept anAN)
                      (RemoveConcept (RestOfConcepts anAN)
                                     aConceptSymbol))))))

(define RemoveLink
  (lambda (anAN aConceptSymbol aLinkSymbol)
    
    (define removeAux
      (lambda (aSlotList)
        ; remove LINK with name aLinkSymbol from list
        (cond ((not aSlotList) ())
              ((eq? aLinkSymbol (GetLinkName (car aSlotList)))
               (cdr aSlotList))
              (else (cons (car aSlotList)
                          (removeAux (cdr aSlotList)))))))
    
    (let ((aConcept (FindConcept anAN aConceptSymbol)))
      (if (not aConcept)
          anAN
          (AddConcept anAN
                      (MakeConceptWithList
                        aConceptSymbol
			(removeAux (GetSlots aConcept))))))))

(define AddLink
  (lambda (anAN aConceptSymbol aLink)
    (let* ((newAN (RemoveLink anAN aConceptSymbol
                                   (GetLinkName aLink)))
           (aConcept (FindConcept newAN aConceptSymbol)))
      (AddConcept newAN
                  (MakeConceptWithList aConceptSymbol
                                       (if (not aConcept)
                                           (list aLink)
                                           (append
                                             (GetSlots aConcept)
					     (list aLink))))))))

(define RemoveAKOLink
  (lambda (anAN aConceptSymbol superClassConceptSymbol)
    
    (define removeAux
      (lambda (aConceptNameList)
        ; remove superClassConceptSymbol from list
        (cond ((null? aConceptNameList) ())
              ((eq? superClassConceptSymbol (car aConceptNameList))
               (cdr aConceptNameList))
              (else (cons (car aConceptNameList)
                          (removeAux (cdr aConceptNameList)))))))
    
    (let ((oldLinks (GetAKOLinks anAN aConceptSymbol)))
      (AddLink anAN
               aConceptSymbol
               (list 'AKO (removeAux oldLinks))))))

(define AddAKOLink
  (lambda (anAN aConceptSymbol superClassConceptSymbol)
    (let ((oldLinks (GetAKOLinks anAN aConceptSymbol)))
      (if (member superClassConceptSymbol oldLinks)
          anAN
          (AddLink anAN
                   aConceptSymbol
                   (list 'AKO
                         (append
                            oldLinks
			    (list superClassConceptSymbol))))))))

(define PrintAN
  (lambda (anAN)
    (DisplayLine "*** Associative Network ***")
    (for-each (lambda (aConcept)
                (display "  ")
                (PrintConcept aConcept)
                (newline))
              anAN)
    (newline)))
               
; ========== query fns on ANs ===============================

(define RecurseThruLists
  (lambda (anAN nodesToSearch nodesSeen aGeneratorFN aCmpFN)
    ; general searching procedure for finding all nodes linked from an
    ; initial node by some type of links. Initial call is of the form:
    ; (RecurseThruLists (list initialNode)
    ;                   ()
    ;                   aGeneratorFN
    ;                   aCmpFN)
    ; where aGeneratorFN returns a list of nodes immediately linked to
    ; a given node and aCmpFN can be used to test nodes for equality.
    ; Algorithm:
    ;   Repeatably remove one node N from nodesToSearch, use
    ;   aGeneratorFN to generate a new list NEW and add N to
    ;   nodesSeen. Then for each node in NEW not in nodesSeen
    ;   add it to nodesToSearch. Repeat while there are still
    ;   nodes in nodesToSearch
    
    (define onList?
      (lambda (aNode aList)
        (cond ((null? aList) #f)
              ((aCmpFN aNode (car aList)) #t)
              (else (onList? aNode (cdr aList))))))
            
    (if (null? nodesToSearch)
        nodesSeen
        (let* ((n (car nodesToSearch))
               (rest (cdr nodesToSearch))
               (new (aGeneratorFN anAN n)))
          (set! nodesSeen (append nodesSeen (list n)))
          (for-each (lambda (aNode)
                      (if (onList? aNode nodesSeen)
                          #f
                          (set! rest (append rest (list aNode)))))
                    new)
          (RecurseThruLists anAN
                            rest
                            nodesSeen
                            aGeneratorFN
                            aCmpFN)))))

(define FindAllConcepts
  (lambda (anAN) (map GetConceptSymbol anAN)))

(define FindAllLinkNames
  (lambda (anAN)
    (let ((result ()))
      (for-each
       (lambda (aConcept)
         (for-each (lambda (aSlot)
                     (if (or (AKOSlot? aSlot)
                             (member (GetLinkName aSlot) result))
                         #f
                         (set! result (cons (GetLinkName aSlot) result))))
                   (GetSlots aConcept)))
       anAN)
      result)))

(define FindConcept (lambda (anAN aConceptSymbol) (assoc aConceptSymbol anAN)))

(define GetAKOSlot
  (lambda (anAN aConceptSymbol)
    (let ((aConcept (FindConcept anAN aConceptSymbol)))
      (if (not aConcept)
          #f
          (assoc 'AKO (GetSlots aConcept))))))

(define GetAKOLinks
  (lambda (anAN aConceptSymbol)
    (let ((aSlot (GetAKOSlot anAN aConceptSymbol)))
      (if (not aSlot)
          ()
          (GetConceptList aSlot)))))

(define GetAKOChain
  (lambda (anAN aConceptSymbol)
    ; return a list of all concepts linked via AKO links
    ; starting from concept aConceptSymbol
    (if (member aConceptSymbol (FindAllConcepts anAN))
        (RecurseThruLists anAN
                          (list aConceptSymbol)
                          ()
                          GetAKOLinks
                          equal?)
        ())))

(define GetLinkSymbolSlots
  (lambda (anAN aConceptSymbol aLinkSymbol)
    
    (define getAux
      (lambda (aSlotList aResultList)
        (if (NoMoreSlots? aSlotList)
            aResultList
            (let ((aSlot (FirstSlot aSlotList)))
              (getAux (RestOfSlots aSlotList)
                      (if (eq? aLinkSymbol (GetLinkName aSlot))
                          (cons aSlot aResultList)
                          aResultList))))))
    
    ; return all slots of aConceptSymbol labelled with
    ; aConceptSymbol
    (let ((aConcept (FindConcept anAN aConceptSymbol)))
      (if (not aConcept)
          ()
          (getAux (GetSlots aConcept) ())))))

(define FindAllLinkedConcepts
  (lambda (anAN aConceptSymbol aLinkSymbol . aThresholdValue)
    ; return a list of all concepts linked via aLinkSymbol links
    ; starting from concept aConceptSymbol that exceed the optional
    ; certainty factor aThresholdValue (which defaults to 1.0 if not
    ; supplied). The nodes during the call to RecurseThruLists are
    ; tuples with the first element either a ConceptName derived from
    ; AKO links or slots with a link name of aLinkSymbol and the
    ; second element is the current multiplicative weight.
    
    (define makeTuple (lambda (anElement aWeight) (list anElement aWeight)))
    (define getElement (lambda (aNode) (car aNode)))
    (define getWeight (lambda (aNode) (cadr aNode)))
    
    (define getBothLinks
      (lambda (anAN aNode)
        ; get both AKO and all aLinkSymbol links
        (let* ((anElement      (getElement aNode))
               (aWeight        (getWeight aNode))
               (aConceptSymbol (if (pair? anElement)
                                   (GetConceptName anElement)
                                   anElement))
               (akoLinks (GetAKOLinks anAN aConceptSymbol))
               (symbolLinks (GetLinkSymbolSlots anAN
                                                aConceptSymbol
                                                aLinkSymbol)))
          (append (map (lambda (akoElement)
                         (makeTuple akoElement aWeight))
                       akoLinks)
                  (map (lambda (symbolElement)
                         (makeTuple symbolElement
                                    (* aWeight
                                       (GetCertaintyFactor
                                        symbolElement))))
                       symbolLinks)))))
    
    (define cmpNodes
      (lambda (aNode1 aNode2)
        ; decide if aNode1 is different from aNode2 and should
        ; be added to the list of links. Return #f only if they
        ; are different and it should be added.
        ; Don't add it if it has a weight less than the threshold.
        (let* ((element1 (getElement aNode1))
               (element2 (getElement aNode2))
               (aConceptSymbol1 (if (pair? element1)
                                    (GetConceptName element1)
                                    element1))
               (aConceptSymbol2 (if (pair? element2)
                                    (GetConceptName element2)
                                    element2))
               (aWeight1 (getWeight aNode1))
               (aWeight2 (getWeight aNode2)))
          (or (< aWeight1 aThresholdValue)
              (and (eq? aConceptSymbol1 aConceptSymbol2)
                   (= aWeight1 aWeight2))))))
    
    (define removeExtraneousSymbols
      (lambda (aList)
        ; remove concept symbols and if weight above
        ; aThresholdValue use this weight for link symbol elements
        (cond ((null? aList) ())
              ((or (not (pair? (getElement (car aList))))
                   (< (getWeight (car aList)) aThresholdValue))
               (removeExtraneousSymbols (cdr aList)))
              (else
               (let ((element (getElement (car aList))))
                 (cons (LINK (GetLinkName element)
                             (GetConceptName element)
                             (getWeight (car aList)))
                       (removeExtraneousSymbols (cdr aList))))))))
    
    (set! aThresholdValue
          (if (null? aThresholdValue) 1.0 (car aThresholdValue)))
    (removeExtraneousSymbols
     (RecurseThruLists anAN
                       (list (makeTuple aConceptSymbol 1.0))
                       ()
                       getBothLinks
                       cmpNodes))))

(define FindAllLinks
  (lambda (anAN aConceptSymbol1 aConceptSymbol2 . aThresholdValue)
    ; return a list of all possible links that can connect the two
    ; symbols in the sence of the function FindAllLinkedConcepts.
    ; If there are pure AKO links connecting them then AKO is
    ; also prepended to the list.
    ; The optional certainty factor aThresholdValue defaults to
    ; 1.0 if not supplied.
    
    (define filterOnConceptSymbol2
      (lambda (aConceptList)
        ; remove those without aConceptSymbol2 as their concept name
        (cond ((null? aConceptList) ())
              ((eq? (GetConceptName (car aConceptList)) aConceptSymbol2)
               (cons (car aConceptList)
                     (filterOnConceptSymbol2 (cdr aConceptList))))
              (else (filterOnConceptSymbol2 (cdr aConceptList))))))
    
    (set! aThresholdValue
          (if (null? aThresholdValue) 1.0 (car aThresholdValue)))
    (let ((result ()))
      (for-each
       (lambda (aLinkSymbol)
         (set! result
               (append result
                       (filterOnConceptSymbol2
                         (FindAllLinkedConcepts anAN
						aConceptSymbol1
						aLinkSymbol
						aThresholdValue)))))
       (FindAllLinkNames anAN))
      (if (member aConceptSymbol2 (GetAKOChain anAN aConceptSymbol1))
          (cons 'AKO result)
          result))))

(define Linked?
  (lambda (anAN aConceptSymbol1 aConceptSymbol2 . aThresholdValue)
    ; is there aany possible linksymbols that can connect the two
    ; symbols in the sence of the functions FindAllLinkedConcepts
    ; The optional certainty factor aThresholdValue defaults to
    ; 1.0 if not supplied.
    (set! aThresholdValue
          (if (null? aThresholdValue) 1.0 (car aThresholdValue)))
    (not (null? (FindAllLinks anAN
                              aConceptSymbol1
                              aConceptSymbol2
                              aThresholdValue)))))
