Object subclass: #Tablet
	instanceVariableNames: 'tabletForm owner drawPen erasePen sprayForm setNibMenu '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'warwick'!
Tablet comment:
'The Tablet contains a form for drawing on, and all the associated drawing
routines. It also contains display and update routines, to cutthe resulting
bitmap fromthe screen.'!

!Tablet methodsFor: 'updating'!
resetTabletForm: aForm
	"Change the form which we are editing"

	tabletForm _ aForm!
setEraseTip
	" Chooses a new tip size for the eraser"

	| answer |
	answer _ setNibMenu startUpWithCaption: 'Choose Tip size'.
	answer > 0 ifTrue: [erasePen defaultNib: answer]!
setPenNib
	"Set the pens nib to the desired width; note that the maximum pen size is 9"

	| answer |
	answer _ setNibMenu startUpWithCaption: 'Choose Nib size'.
	answer > 0 ifTrue: [drawPen defaultNib: answer]!
updateTablet
	"The bitmap is 'cut' from the screen whenever the user makes changes.
	The old bitmap is used if it is the same size."

	tabletForm _ Form fromDisplay: self tabletSize using: tabletForm! !

!Tablet methodsFor: 'orientation'!
owner
	"Returns the top view of the model."

	^ owner!
owner: t1
	"Sets the top view" 

	owner _ t1!
tabletExtent
	"Return the bottom right corner of the form."

	^ tabletForm extent!
tabletSize
	"Returns the size of the form, which is its owner's size inset by the border"

	| formSize |
	formSize _ owner insetDisplayBox.
	^ formSize insetBy: 3! !

!Tablet methodsFor: 'drawing'!
drawOnTablet
	"Set the pen for using on the tablet"

	self penOnTablet: drawPen!
eraseOnTablet
	self penOnTablet: erasePen!
penOnTablet: aPen
	"Draw or erase on the tablet while we have not chosen another option or closed down the window"

	aPen frame: self tabletSize.
	[Sensor yellowButtonPressed not and: [Sensor blueButtonPressed not]]
		whileTrue: 
			[Sensor redButtonPressed
				ifTrue: [aPen down]
				ifFalse: [aPen up].
			aPen goto: Sensor mousePoint].
	self updateTablet!
repeatOnTablet: aForm
	"Scribble with the user defined form on the screen"
 
	| centerX centerY |
	centerX _ aForm width / 2.	"Calculate where to display so that the cursor is centered"
	centerY _ aForm height / 2.
	[Sensor yellowButtonPressed not and: [Sensor blueButtonPressed not]]
		whileTrue: [Sensor redButtonPressed
				ifTrue: [aForm
						displayOn: Display
						at: Sensor mousePoint - (centerX @ centerY)
						clippingBox: self tabletSize
						rule: Form paint
						mask: Form black]].
	self updateTablet!
ruleOnTablet
	"Rules a line between two consecutive mouseclicks using the current pen"

	drawPen frame: self tabletSize.
	[Sensor yellowButtonPressed not and: [Sensor blueButtonPressed not]]
		whileTrue: [Sensor redButtonPressed
				ifTrue: 
					[Sensor waitNoButton.	"Till have released from the first click"
					drawPen up.
					drawPen goto: Sensor mousePoint.
					Sensor waitClickButton.
					drawPen down.
					drawPen goto: Sensor mousePoint]].
	self updateTablet!
sprayOnTablet
	"Sprays the sprayForm on the screen"

	[Sensor yellowButtonPressed not and: [Sensor blueButtonPressed not]]
		whileTrue: [Sensor redButtonPressed
				ifTrue: [sprayForm
						displayOn: Display
						at: Sensor mousePoint + (10 @ 4)
						clippingBox: self tabletSize
						rule: Form paint
						mask: Form black]].
	self updateTablet! !

!Tablet methodsFor: 'display'!
displayTablet: aMask
	"Redraw the tablet using the argument mask, used to blank out and redraw the tablet" 

	tabletForm
		displayOn: Display
		at: 0 @ 0
		clippingBox: self tabletSize
		rule: Form over
		mask: aMask!
grabTablet
	"Reposition the form within the current window, reset it to its original position
	if the user has another go at it"

	| origMousePosition |
	[Sensor yellowButtonPressed not and: [Sensor blueButtonPressed not]]
		whileTrue: [Sensor redButtonPressed
				ifTrue: 
					[self displayTablet: Form white.		"Erase the old image"
					tabletForm offset: self tabletSize origin + (Sensor mousePoint - origMousePosition).
					self displayTablet: Form black]		"Redraw in new position"
				ifFalse: [origMousePosition _ Sensor mousePoint]].
	self updateTablet!
showTablet
	"Called to reset the origin of the tablet  and redisplay at the new window location"

	tabletForm offset: self tabletSize origin.
	self displayTablet: Form black.! !

!Tablet methodsFor: 'initialisation'!
initPens
	"Set up the instance pens, erase has to be over the top of everything else"

	drawPen _ Pen new black; defaultNib: 3.
	erasePen _ Pen new white; combinationRule: Form over; defaultNib: 6!
initPopupmenu
	"Menu for choosing nib sizes"

	setNibMenu _ PopUpMenu labels: '1
2
3
4
5
6
7
8
 9 '!
initSpraycan
	"The spray form"

	sprayForm _ Form
				extent: 16 @ 16
				fromArray: #(8452 2080 33153 12300 6168 36873 12684 960 12684 36873 6168 34849 12300 33153 1040 8452 )
				offset: -14 @ -14!
initTablet
	"Set the tablet to a null form"

	| aForm |
	aForm _ Rectangle origin: 0 @ 0 corner: 0 @ 0.
	tabletForm _ Form new fromDisplay: aForm! !

!Tablet methodsFor: 'retrieval'!
returnBitmap
	"For returning the edited form to the TabletEditor which owns this Tablet" 

	^ tabletForm! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Tablet class
	instanceVariableNames: ''!

!Tablet class methodsFor: 'creation'!
create
	| newTablet |
	newTablet _ self new.
	^ newTablet! !


MouseMenuController subclass: #TabletController
	instanceVariableNames: 'MyCursor PenCursor EraseCursor SprayCursor RuleCursor RepCursor GrabCursor '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'warwick'!
TabletController comment:
'The controller handles the specialised menus and cursors which are used while
editing a Tablet.'!

!TabletController methodsFor: 'initialization'!
initCursors
	"Initialise the cursor constants from arrays which represent binary bitmaps"

	self initPen.
	self initSpraycan.
	self initRule.
	self initEraser.
	self initRep.
	self initGrab!
initEraser
	EraseCursor _ Cursor
				extent: 16 @ 16
				fromArray: #(15872 25344 49536 50368 51296 53552 27160 13388 6808 3376 1632 960 384 0 0 0 )
				offset: -14 @ -14!
initGrab
	GrabCursor _ Cursor
				extent: 16 @ 16
				fromArray: #(384 3696 4686 4681 29257 37449 37449 37449 37449 37449 16385 8194 4100 2052 1028 0 )
				offset: -14 @ -14!
initialize
	"Initialize the mouse controller, it's menus and it's yellow button menu"

	| aMenu actionList |
	super initialize.
	MyCursor _ Cursor new.
	self initCursors.
	aMenu _ PopUpMenu labels: 'draw
erase
ruler
spraycan
replicate
grab
set erase tip
set pen nib' lines: #(4 6 ).
	actionList _ #(draw eraser ruler spraycan replicate grab eraseTip penNib ).
	self yellowButtonMenu: aMenu yellowButtonMessages: actionList!
initPen
	PenCursor _ Cursor
				extent: 16 @ 16
				fromArray: #(32768 49152 24576 14336 7168 3584 1856 944 472 236 118 62 30 14 3 0 )
				offset: -14 @ -14!
initRep
	RepCursor _ Cursor
				extent: 16 @ 16
				fromArray: #(384 384 384 384 384 384 384 65535 384 384 384 384 384 384 384 0 )
				offset: -14 @ -14!
initRule
	RuleCursor _ Cursor
				extent: 16 @ 16
				fromArray: #(65504 49248 49344 49536 49920 50688 52224 55296 61440 57344 0 0 0 0 0 0 )
				offset: -14 @ -14!
initSpraycan
	SprayCursor _ Cursor
				extent: 16 @ 16
				fromArray: #(960 960 3696 14364 57351 51171 51219 51203 51171 49171 51219 51171 49155 65535 0 0 )
				offset: -14 @ -14! !

!TabletController methodsFor: 'set cursor shapes'!
setEraser
	^ EraseCursor!
setGrab
	^ GrabCursor!
setPen
	^ PenCursor!
setRep
	^ RepCursor!
setRuler
	^ RuleCursor!
setSpraycan
	^ SprayCursor! !

!TabletController methodsFor: 'menu options'!
draw
	"Set the cursor and send the appropiate message to the tablet"
 
	MyCursor _ self setPen.
	Cursor currentCursor: MyCursor.
	self model drawOnTablet!
eraser
	"Set the cursor and send the appropiate message to the tablet"

	MyCursor _ self setEraser.
	Cursor currentCursor: MyCursor.
	self model eraseOnTablet!
eraseTip
	"Reset the erase pen and then put into erase mode"

	self model setEraseTip.
	self eraser!
grab
	"Set the cursor and send the appropiate message to the tablet"

	MyCursor _ self setGrab.
	Cursor currentCursor: MyCursor.
	self model grabTablet!
penNib
	"Reset the pen nib then put into draw mode"

	self model setPenNib.
	self draw.!
replicate
	"Get a form from the user to draw on the screen, set the cursor then send control to the model"

	| drawForm |
	drawForm _ Form fromUser.
	MyCursor _ self setRep.
	Cursor currentCursor: MyCursor.
	self model repeatOnTablet: drawForm!
ruler
	"Set the cursor and send the appropiate message to the tablet"

	MyCursor _ self setRuler.
	Cursor currentCursor: MyCursor.
	self model ruleOnTablet!
spraycan
	"Set the cursor and send the appropiate message to the tablet"

	MyCursor _ self setSpraycan.
	Cursor currentCursor: MyCursor.
	self model sprayOnTablet! !


Object subclass: #TabletEditor
	instanceVariableNames: 'tablet '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'warwick'!
TabletEditor comment:
'The TabletEditor has a tablet associated with it, on which it can open an editing view.
The tablets form can then be retrieved, reset re-edited etc.'!

!TabletEditor methodsFor: 'tablet access'!
returnTablet
	^ tablet returnBitmap! !

!TabletEditor methodsFor: 'setting tablet'!
setTablet: aForm
	"We are about to start editing a new Form, set the Tablet to the new Form"

	tablet resetTabletForm: aForm!
tablet: aTablet 
	tablet _ aTablet! !

!TabletEditor methodsFor: 'edit tablet'!
editTablet
	"Open a view on the TabletEditor's Tablet. Note that this command cannot be executed in
	the middle of a sequence as it returns control back to the system."

	TabletView open: tablet! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TabletEditor class
	instanceVariableNames: ''!

!TabletEditor class methodsFor: 'creation'!
create
	"Creates a Tablet Editor which must have a tablet associated with it.
	This Tablet's form can then be altered (by sending the message 'edit' etc)"

	| aTablet aTabletEditor |
	aTabletEditor _ TabletEditor new.
	aTablet _ Tablet create.
	aTabletEditor tablet: aTablet.
	aTablet initTablet; initPens; initSpraycan; initPopupmenu.
	^ aTabletEditor! !

!TabletEditor class methodsFor: 'examples'!
example1
	"This simple example uses a Tablet Editor to return a Form to a global variable,
	this form then being displayed on the screen. The globals must be declared before
	executing the example and the commands must be executed as two sequences of 
	commands (as grouped) as control is returned to the system after the editTablet.

	ATabletEditor _ TabletEditor create.
	ATabletEditor editTablet.
	
	AForm _ ATabletEditor returnTablet.
	AForm displayOn: Display at: Sensor waitButton.  "!
example2
	"This demonstrates the use of an editor to re-edit a form,

	ATabletEditor _ TabletEditor create.
	ATabletEditor editTablet.
	
	'Now display the resulting form and then edit the tablet again, starting with the previous form'
	AForm _ ATabletEditor returnTablet.
	AForm displayOn: Display at: Sensor waitButton.
	ATabletEditor editTablet.

	AForm _ ATabletEditor returnTablet.
	AForm displayOn: Display at: Sensor waitButton.    "!
example3
	"This example demonstrates the use of a single Tablet Editor to edit two different
	forms. The first one is the original Tablet, the second is one which is obtained from the
	user. Again these need to be executed in these groupings.
 
	ATabletEditor _ TabletEditor create.
	ATabletEditor editTablet.
	
	'Now get a form from the user and use the Tablet Editor to edit it'
	AForm _ ATabletEditor returnTablet.
	AnotherForm _ Form fromUser.
	ATabletEditor setTablet: AnotherForm.
	ATabletEditor editTablet.
	
	'Display the two edited forms, (be carefull with the mouse clicks!!)'
	AnotherForm _ ATabletEditor returnTablet.
	AnotherForm displayOn: Display at: Sensor waitButton.
	AForm displayOn: Display at: Sensor waitButton.    "! !


View subclass: #TabletView
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'warwick'!
TabletView comment:
'A view upon a Tablet, used for editing and displaying a Tablet'!

!TabletView methodsFor: 'display'!
displayView
	"This is used by the view to update the model, for example when the window
	is resized. This redefines the superclass displayView."

	model showTablet! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TabletView class
	instanceVariableNames: ''!

!TabletView class methodsFor: 'creation'!
open: aTablet
	"Set up a window to display, edit and control the Tablet"
 
	| aStandardView aTabletView |
	aTabletView _ TabletView new.
	aStandardView _ StandardSystemView new.
	aTablet owner: aStandardView.
	aTabletView model: aTablet; controller: TabletController new; borderWidth: 3; insideColor: Form white.
	aStandardView label: 'Tablet'; minimumSize: 100 @ 100; addSubView: aTabletView.
	aStandardView controller open! !
