;;;-*-Mode:LISP;Syntax: Common-Lisp;Package:ib;Base:10-*-
;;;
;;; Copyright 1990 GMD (German National Research Center for Computer Science)
;;;
;;; Permission to use, copy, modify, distribute, and sell this software and its
;;; documentation for any purpose is hereby granted without fee, provided that
;;; the above copyright notice appear in all copies and that both that
;;; copyright notice and this permission notice appear in supporting
;;; documentation, and that the name of GMD not be used in advertising or
;;; publicity pertaining to distribution of the software without specific,
;;; written prior permission.  GMD makes no representations about the
;;; suitability of this software for any purpose.  It is provided "as is"
;;; without express or implied warranty.
;;;
;;; GMD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
;;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL GMD
;;; BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
;;; WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
;;; OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
;;; CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;; Authors: Project GINA (spenke@gmd.de)
;;;          P.O. Box 1316
;;;          D-5205 Sankt Augustin 1
;;;          Germany

(in-package :ib)

(setq *sccs-id* "@(#)lisp-code.lisp	1.20	11/9/92")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; details record for lisp code
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

'(defclass lisp-code-details (general-code-details)
  ((detail-name    :reader detail-name      :initform :lisp-code-details
                   :allocation :class)
   (what-package   :accessor what-package   :initform "ib-demo"))
  (:documentation "contains information needed to generate lisp code"))

(defun make-lisp-code-details ()
  (make-instance 'lisp-code-details))

(defmethod write-code-details :after ((details lisp-code-details) stream)
  (format stream "~s~%" (what-package details)))

(defmethod read-code-details :after ((details lisp-code-details) stream)
  (setf (what-package details) (read stream)))

(defmethod write-code ((details lisp-code-details) doc default-pathname)
  (with-open-file (stream
                   (make-pathname :type "lisp" :defaults default-pathname)
		   :direction :output
		   :if-exists :new-version :if-does-not-exist :create)
    (write-lisp-code doc details stream)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; writing the lisp source file
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod write-lisp-code ((doc ib-document) details stream)
  (write-code-header details doc stream)
  (terpri stream)
  (write-code-for-dialog-class details doc stream)
  (terpri stream)
  (write-code-for-constructor details doc stream)
  (terpri stream))

(defmethod write-code-header ((details lisp-code-details) doc stream)
  (declare (ignore doc))
  (format stream "~a~a~%~%"
          ";;;-*-Mode:LISP; Syntax: Common-Lisp;"
          (format nil "Package:~a;-*-"
                  (what-package details)))
  (format stream ";; !!! Do not edit !!!~%")
  (format stream 
          ";; This code was generated by the GINA Interface Builder.~%~%")
  (format stream "(in-package :~a)~%" (what-package details))
  ;;(format stream "(use-package '(gina lisp))~%")
  )

(defun pp-to-stream (stream string)
  (write (read-from-string string) :stream stream :pretty t :level nil
	 :length nil :case :downcase))

(defmethod write-code-for-dialog-class ((details lisp-code-details) doc stream)
  (pp-to-stream stream (code-for-dialog-class doc details)))

(defmethod write-code-for-constructor ((details lisp-code-details) doc stream)
  (pp-to-stream stream (code-for-dialog-constructor doc details)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; individual code generators
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod code-for-dialog-class ((doc ib-document) (details lisp-code-details) 
                                  &aux slot-code)
  "generate the code for the new dialog class"
  (setf slot-code (loop for info in (find-all-widget-infos (main-info doc))
		   collect (format nil "(~a :accessor ~a)"
                                   (widget-name info)
                                   (widget-name info))))
  (format nil "(defclass ~a (~a) (~{~a~}  ) (:documentation \"\"))"
	  (dialog-class (main-info doc))
          (widget-class (main-info doc))
	  slot-code))

(defmethod code-for-dialog-constructor ((doc ib-document) 
                                        (details lisp-code-details)
                                        &aux single non-form parent size)
  "generate the code for the dialog constructor function"
  (setq single (eql (length (children (main-info doc))) 1))
  (setq size (bounding-box (children (main-info doc))))
  (setq non-form (eql (widget-class (main-info doc)) 'document-shell))
  (setq parent (if (and non-form (not single)) "main-form" "box"))
  (format nil "(defun make-~a (~a &aux ~a) (setq box ~a) ~{~a ~} box)"
          (dialog-class (main-info doc))
          (param-definitions doc details non-form)
          (if (and non-form (not single)) "box main-form" "box")
          (generate-code-for-shell (main-info doc) details non-form
              (if single (width (first (children (main-info doc))))
                                 (third size))
              (if single (height (first (children (main-info doc))))
                                 (fourth size)))
	  (append
              (when (and non-form (not single))
                  '("(setq main-form (make-form box))"))
              (apply 'append
		 (loop for info in (children (main-info doc))
		  collect (generate-code-for-widgets doc details info parent)))
              (unless (and non-form single)
                 (default-constraints (children (main-info doc)) single
                    (eql (widget-class (main-info doc)) 'tool-dialog-box)))
              (callback-definitions (main-info doc) details))))

(defmethod param-definitions ((doc ib-document) (details lisp-code-details) 
                              non-form 
                              &aux result)
  (declare (ignore non-form))
  (setq result 
        (loop for view-info in (find-all-of-a-class (main-info doc) 'view)
              collect (format nil "(~a-constructor 'make-~a)"
                                  (widget-name view-info)
                                  (dialog-class view-info))))
  (concatenate 'simple-string
               "doc &key (initargs nil) (motif-resources nil) "
               (format nil "(class '~a)~{ ~a~}" 
                           (dialog-class (main-info doc)) result)))

(defmethod generate-code-for-shell ((info widget-info) 
                                    (details lisp-code-details) 
                                    non-form width height)
  (declare (ignore width height))
  (concatenate 'simple-string
    (format nil "(~a ~a ~{~a ~} :class class :initargs initargs ~{~s ~a ~} "
	    (create-function info)
            (if non-form "doc" "")
	    (find-positional-representations info details)
            (find-key-representations info details
                             :resource-list
                                 (remove (find-resource info :class)
                                         (key-resources info))))
    (if non-form
        ":motif-resources motif-resources)"
        ":document doc :motif-resources motif-resources)")
  ;; absolute size no longer used in 1.1
  ;  (format nil 
  ;          (if non-form 
  ;              ":width ~d :height ~d)"
  ;      ;;        "(append '(:width ~d :height ~d) motif-resources))"
  ;              "motif-resources)"   ;; works in 1.1
  ;              )
  ;          width
  ;          (+ height
  ;             (if (or (eql (widget-class info) 'modeless-dialog-box) 
  ;                     (eql (widget-class info) 'modal-dialog-box))
  ;                  0 30)))
  ))

(defun default-constraints (children single tool &aux size)
  (setq size (bounding-box children))
  (if single
      (list (format nil "(define-form-constraint (~a box) ~a ~a ~a ~a)"
                    (widget-name (first children))
                    ":left-attachment :form"
                    ":right-attachment :form"
                    (if tool
                        ":top-attachment :widget :top-widget (main-menu box)"
                        ":top-attachment :form")
                    ":bottom-attachment :form"))
      (loop for info in children
            collect 
              (format nil "(define-form-constraint (~a box) ~a ~d ~a ~d)"
              (widget-name info)
              ":left-attachment :position :left-position "
              (round (/ (* (- (x-pos info) (first size)) 100) (third size)))
              ":top-attachment :position :top-position "
              (round (/ (* (- (y-pos info) (second size)) 100) (fourth size)))
  ))))

(defmethod generate-code-for-widgets ((doc ib-document) 
                                      (details lisp-code-details) info parent)
  (cons (format nil "(setf (~a box) ~a) "
		(widget-name info)
		(code info details parent))
	(append
            (apply 'append
	       (loop for child in (children info)
		collect (generate-code-for-widgets
			 doc details child (format nil "(~a box)"
					               (widget-name info)))))
            (loop for child in (children info)
             collect (constraint-code child details
                             (format nil "(~a box)" (widget-name child)))))))

(defmethod code ((info widget-info) (details lisp-code-details) parent 
                 &aux motif-values)
  "build the code to create this widget"
  (setq motif-values (find-motif-representations info details))
  (format nil "(~a ~a ~{~a ~} :name ~s ~{~s ~a ~}~?)"
	  (create-function info)
	  parent
	  (find-positional-representations info details)
	  (widget-name info)
          (find-key-representations info details)
	  (if motif-values ":motif-resources '(~{~s ~a ~})" "")
	  (if motif-values (list motif-values) '(()))))

;(defun quote-it (value-list)
;  (loop for element in value-list
;        collect (if (and element (listp element))
;                    `',element
;                    element)))

(defun quote-list (element)
  (if (and element (listp element))
                    `',element
                    element))

(defmethod constraint-code ((info widget-info) (details lisp-code-details) 
                            slot-string)
  (if (and (constraints info) (is-constraint-child info))
      (constraint-code (constraints info) details slot-string)
      "")) 

(defmethod code ((info view-info) (details lisp-code-details) parent 
                 &aux motif-values)
  "build the code to create this widget"
  (setq motif-values (find-motif-representations info details))
  (format nil "(funcall ~a-constructor ~a ~a~{~a ~} ~{~s ~a ~}~?)"
          (widget-name info)
          parent
          (if (needs-doc info) "doc " "")
          (find-positional-representations info details)
	  (find-key-representations info details
                               :resource-list
                                 (remove (find-resource info :class)
                                   (remove (find-resource info :needs-doc)
                                           (key-resources info))))
          (if motif-values ":motif-resources '(~{~s ~a ~})" "")
          (if motif-values (list motif-values) '(()))))

(defmethod find-positional-representations ((info widget-info) 
                                            (details lisp-code-details))
  (loop for resource in (positional-resources info)
	collect (resource-representation resource details info)))

(defmethod find-key-representations ((info widget-info) 
                                     (details lisp-code-details)
                            &key (resource-list (key-resources info)))
  (loop for resource in resource-list
        when (or (resource-modified resource info)
		 (member (resource-name resource)
			 (required-key-names info)))
        collect (resource-name resource) and
        collect (resource-representation resource details info)))

(defmethod find-motif-representations ((info widget-info)
                                       (details lisp-code-details))
  (loop for resource in (motif-resources info)
	when (or (resource-modified resource info)
		 (member (resource-name resource)
			 (required-key-names info)))
	collect (resource-name resource) and
        collect (resource-representation resource details info)))

(defmethod resource-representation ((resource motif-resource)
                                    (details lisp-code-details) info)
  (format nil "~s" (quote-list (resource-value resource info))))

(defmethod callback-definitions ((info widget-info) (details lisp-code-details))
  (loop for child in (find-all-widget-infos info)
      append
        (loop for resource in (callback-resources child)
              when (resource-modified resource child)
              collect (format nil 
                          "(setf (~a (~a box)) (make-callback '~a ~{~a ~}))"
                          (resource-name resource)
                          (widget-name child)
                          (first (resource-value resource child))
                          (callback-params resource child)))))

(defun callback-params (resource info)
  (loop for item in (rest (resource-value resource info))
        unless (equal item "<Constructor>")
        collect (if (equal item (widget-name (main-info (document info))))
                  "box"
                  (if (equal item "<Document>") 
                    "doc"
                    (if (alpha-char-p (char item 0))
                      (format nil "(~a box)" item)
                      (format nil "~a" item))))))

;;;;;;;;;;;;;;;;;;;;;;;;;; Constraints ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod constraint-code ((constraints ib-form-constraints) 
                            (details lisp-code-details) slot-string)
  (concatenate 'simple-string
      "(define-form-constraint " slot-string " "
      (attachment-code (top constraints) "top") " "
      (attachment-code (left constraints) "left") " "
      (attachment-code (right constraints) "right") " "
      (attachment-code (bottom constraints) "bottom") ")"))

(defun attachment-code (attachment name)
  (case (atype attachment)
    ((:none)
        (format nil ":~a-attachment :none" 
                    name))
    ((:position :self)
        (format nil ":~a-attachment ~s :~a-position ~s" 
                    name (atype attachment)
                    name (form-pos attachment)))
    ((:widget :opposite-widget)
        (format nil ":~a-attachment ~s :~a-widget (~a box) :~a-offset ~d"
                    name (atype attachment)
                    name (widget-name (widget attachment))
                    name (offset attachment)))
    (t
        (format nil ":~a-attachment ~s :~a-offset ~d" 
                    name (atype attachment)
                    name (offset attachment)))))

(defmethod constraint-code ((constraints ib-pane-constraints) 
                            (details lisp-code-details) slot-string)
  (format nil "(define-pane-constraint ~a ~{~s ~})" slot-string
           (append `(:minimum ,(minimum constraints))
                   `(:maximum ,(maximum constraints))
                   `(:skip-adjust ,(skip-adjust constraints))
                   `(:allow-resize ,(allow-resize constraints)))))
