;;; -*- Mode:LISP;Syntax: Common-Lisp;Package: GINA ;Base:10-*-
;;;
;;; Copyright 1990 GMD (German National Research Center for Computer Science)
;;;
;;; Permission to use, copy, modify, distribute, and sell this software and its
;;; documentation for any purpose is hereby granted without fee, provided that
;;; the above copyright notice appear in all copies and that both that
;;; copyright notice and this permission notice appear in supporting
;;; documentation, and that the name of GMD not be used in advertising or
;;; publicity pertaining to distribution of the software without specific,
;;; written prior permission.  GMD makes no representations about the
;;; suitability of this software for any purpose.  It is provided "as is"
;;; without express or implied warranty.
;;;
;;; GMD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
;;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL GMD
;;; BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
;;; WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
;;; OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
;;; CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;; Authors: Project GINA (spenke@gmd.de)
;;;          P.O. Box 1316
;;;          D-5205 Sankt Augustin 1
;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;; some OS-dependent utilities
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(in-package :GINA)

(setq *sccs-id* "@(#)OS-dependent.lisp	1.27  11/9/92")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;   PCL patches
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; tell the PCL code-walker to walk through SCL:LETF* just like through LET*
;; SCL:LETF* is generated by the WITH-GCONTEXT macro of CLX !!
#+(and Genera pcl)
(walker:define-walker-template SCL:LETF* walker::walk-let*)

;; (real-)get-method existiert, find-method nicht
;; mir unbekannt, ob beide wirklich gleich sind
#+pcl (setf (symbol-function 'find-method)
	    (symbol-function 'pcl::real-get-method))


;; without the patch this form cannot be macro-expanded!!
#+(and Genera pcl)
'(with-slots (xxx) y
  (xlib:with-gcontext (gc :line-style :dash)))
;; ==>
#+(and Genera pcl)
'(LET ((#:G73385 Y))
  (DECLARE (PCL:VARIABLE-REBINDING #:G73385 Y))
  #:G73385
  (MULTIPLE-VALUE-BIND (#:G73386 #:G73387 #:G73389 #:G73388)
      (XLIB::COPY-GCONTEXT-LOCAL-STATE GC '(5))
    (DECLARE (TYPE XLIB:GCONTEXT #:G73386)
             (TYPE XLIB::GCONTEXT-STATE #:G73387)
             (TYPE XLIB::XGCMASK #:G73389)
             (TYPE (OR NULL XLIB:RESOURCE-ID) #:G73388))
    (LET ((#:G73392 (AREF #:G73386 6)))
      (DECLARE (TYPE XLIB::GCONTEXT-STATE #:G73392))
      (UNWIND-PROTECT (SCL:LETF* (((AREF #:G73392 5) (AREF #:G73387 5)) ((AREF #:G73392 26) 0))
                        (LET ((#:G73390 :DASH))
                          (XLIB::SET-GCONTEXT-LINE-STYLE #:G73386 #:G73390)))
                      (ZL:ASET 0 #:G73392 26)
                      (IF #:G73388
                          (PROGN (XLIB::RESTORE-GCONTEXT-TEMP-STATE #:G73386 #:G73389 #:G73388)
                                 )
                          NIL)
                      (XLIB::DEALLOCATE-GCONTEXT-STATE #:G73387)))))

#+Genera
(progn
  (zwei:defindentation (defginavar 1 1))
  (zwei:defindentation (defginamethod 2 1))
  (zwei:defindentation (defginaclass 2 1))
  (zwei:defindentation (defginafun 2 1))
  (zwei:defindentation (defginamacro 2 1))
  (zwei:defindentation (defcallback 2 1)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;   file-system interface
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(ginachapter "Some Utilities for File System Handling")

;; The cmu version should work for any implementation with CLtL2/ANSI
;; pathnames.

(defginafun name-and-type (pathname "pathname object or string"
				    &aux filetype)
  (description "extract name and type components of a pathname"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "a string")
  (example (name-and-type "w1:>gina>documents>four-rects.hello.3")
	   "returns \"FOUR-RECTS.HELLO\""
	   (name-and-type "xpssun:/home/spenke")
	   "returns \"SPENKE\"")
  (setq filetype (pathname-type pathname))
  (cond #+cmu
	((directoryp pathname)
	 (let ((path (pathname-directory pathname)))
	   ;; Since pathname-directory returns nil for some things
	   ;; directoryp thinks is a directory. (like "." or "")
	   (cond ((not path)
		  "")
		 ;; :RELATIVE or :ABSOLUTE
		 ((= (length path) 1)
		  (if (eq :RELATIVE (car path))
		      "."
		      "/"))
		 (t (car (last path))))))
	((or (null filetype) (eq :unspecific filetype))
	 (format nil "~a"    (pathname-name pathname)))
	(t 
	 (format nil "~a.~a" (pathname-name pathname) (pathname-type pathname)))))

(defginafun current-directory-wildcard (&key (alternative :root) ":root or :homedir")
  (description "return wildcard for the current directory or a sensible alternative"
	       :called-by-gina :sometimes
	       :called-by-application :sometimes
	       :result "a pathname object")
  #+(or excl lucid cmu) (declare (ignore alternative))
  #+excl  (directory-wildcard (excl:current-directory))
  #+lucid (directory-wildcard (working-directory))
  #+cmu (multiple-value-bind (okay dir)
	    (unix:unix-current-directory)
	  (if okay
	      (subdirectory-wildcard dir)
	      (error "Current directory could not be detected")))
  ;; there is no working directory on Genera!
  #+Genera (if (eq alternative :root)
	       (root-directory-wildcard)
	       (directory-wildcard (user-homedir-pathname))))
  
(defginafun directory-wildcard (file-pathname "pathname object or string"
				 &optional (file-type :wild))
  (description "build a wildcard for the directory where the file is located"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "a pathname object")
  (example
    (directory-wildcard "w1:>gina>documents>four-rects.hello.3" "xxx")
    "returns #P\"W1:>gina>documents>*.XXX.newest\""
    (directory-wildcard "w1:>gina>documents>four-rects.hello.3")
    "returns #P\"W1:>gina>documents>*.*.newest\""
    (directory-wildcard "xpssun:/home/spenke/four-rects.hello")
    "returns #P\"XPSSUN:/home/spenke/*\"")

  (make-pathname :name :wild :type file-type :version :newest
		 :defaults file-pathname))

(defginafun subdirectory-wildcard (pathname "pathname object or string"
				    &optional (file-type :wild) &aux subdir)
  (description "build a wildcard for the directory this file denotes"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "a pathname object")
  (example
    (subdirectory-wildcard "w1:>gina>documents.directory.1")
    "returns #P\"W1:>gina>documents>*.*.newest\""
    (parent-directory-wildcard "xpssun:/GINA/documents/*.lisp")
    "returns #P\"XPSSUN:/GINA/*.lisp\""
    (subdirectory-wildcard "w1:>gina>documents" "hello")
    "returns #P\"W1:>gina>documents>*.HELLO.newest\"")
    
  #-cmu
  (setq subdir (append (when (listp (pathname-directory pathname))
			 (pathname-directory pathname))
		       (list (#-Genera name-and-type 
				       #+Genera pathname-name pathname))))
  #+cmu
  (setq subdir (append (pathname-directory pathname)
		       (when (pathname-name pathname)
			 (list (pathname-name pathname)))))

  #+excl (setq subdir (remove :root subdir))

  (make-pathname :name      :wild
		 :type      file-type
		 :version   :newest
		 :directory subdir
		 :host (pathname-host pathname)))

(defginafun parent-directory-wildcard (wildcard "pathname object or string")
  (description "build a similar wildcard for the parent directory of this wildcard"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "a pathname object")
  (example (parent-directory-wildcard "w1:>gina>documents>*.*.newest")
	   "returns #P\"W1:>gina>*.*.newest\"")
  (example (parent-directory-wildcard "/usr/local/lisp/*.lisp")
	   "returns #P\"/usr/local/*.lisp\"")
  (make-pathname :directory #-cmu (or (butlast (pathname-directory wildcard)) 
				      #-lucid :root
				      #+lucid '(:root))
		 #+cmu (butlast (pathname-directory wildcard))
		 :defaults #-cmu wildcard #+cmu (file-namestring wildcard)))

'(parent-directory-wildcard "w1:>gina>*.hello.newest")
'(parent-directory-wildcard "w1:>*.*.newest")
'(parent-directory-wildcard "tom:/GINA/documents/*.lisp")
'(parent-directory-wildcard "tom:/GINA/*")
'(parent-directory-wildcard "tom:/*.*")

(defginafun parent-directory-wildcards (pathname "pathname object or string")
  (description "build a list of wildcards for the parent directories"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "a list of pathname objects")
  (example (parent-directory-wildcards "w1:>gina>*.*.newest")
	   "returns (#P\"W1:>gina>*.*.newest\" #P\"W1:>*.*.newest\")")
  (cons (directory-wildcard pathname)
	(loop while (not (root-directory-p pathname))
	      do (setq pathname (parent-directory-wildcard pathname))
	      collect pathname)))

(defginafun parent-directory (pathname "pathname object or string")
  (description "build a pathname for the parent directory"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "a pathname object")
  (example (parent-directory "w1:>gina>documents>*.*.newest")
	   "returns #P\"W1:>gina>documents.DIRECTORY\"")
  
  (make-pathname :host (pathname-host pathname)
		 :directory (or (butlast (pathname-directory pathname)) 
				#-(or lucid cmu) :root
				#+(or lucid cmu) '(:root))
		 :name (car (last (pathname-directory pathname)))
		 :type #+Genera "directory" #+(or lucid excl cmu) nil))

(defginafun parent-directories (pathname "pathname object or string")
  (description "build a list of pathnames for the parent directories"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "a list of pathname objects")
  (example (parent-directories "w1:>gina>documents>*.gredit.newest")
	   "returns (#P\"W1:>gina>documents.DIRECTORY\" #P\"W1:>gina.DIRECTORY\")"
	   (parent-directories "w1:>*.gredit.newest")
	   "returns nil")
  
  (loop while (not (root-directory-p pathname))
	do (setq pathname (parent-directory pathname))
	collect pathname))

'(parent-directories "tom:/GINA/documents/*.lisp")
 

(defginafun root-directory-p (pathname "pathname object or string")
  (description "check if directory component denotes the root directory"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "T or NIL")
  #+Genera (eq :root (pathname-directory pathname)) 
  #+lucid  (equal '(:root) (pathname-directory pathname))
  #+excl   (equal '(:absolute :root) (pathname-directory pathname))
  #+cmu    (equal '(:absolute) (pathname-directory pathname))
  )

'(root-directory-p "w1:>gina>documents>*.*.newest")
'(root-directory-p "w1:>*.*.newest")
'(root-directory-p "w1:>mike")
'(root-directory-p "tom:/*.*")

(defginafun root-directory-wildcard ()
  (description "build a wildcard for the root directory"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "a pathname object")
  (make-pathname :directory #+Genera :root
		            #+(or lucid excl) nil
			    #+cmu `(:absolute)
		 ;;:host "Wisdom-1"
		 :type :wild :name :wild))
'(root-directory-wildcard)

(defginafun directoryp (pathname "pathname object or string")
  (description "check if pathname denotes a directory"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "T or NIL"
	       :bugs '("tape listings are reported as directories on Symbolics"))
  (example (directoryp "w1:>mike.directory")
	   "returns t"
	   (directoryp "w1:>documents>four-rects.gredit.newest")
	   "returns nil"
	   (directoryp "tom:/home/spenke")
	   "returns t")
  #+Genera
  (let ((file-type (pathname-type pathname)))
    (or (null file-type) ;; later: call C-subroutine
	(eq file-type :unspecific)
	(string-equal file-type "directory")))
  #+(and unix (or lucid excl)) (xtk::directoryp (namestring pathname))
  #+cmu (xtk::directoryp (ext:unix-namestring pathname)))

(defginafun wildp (file-type-or-name)
    (description "is a given type or name a wild card?")
    (and (not (null file-type-or-name))
	 (not (stringp file-type-or-name))))	 
        
(defginafun pathname-equal (pathname1 pathname2)
  (description "determine equality of two pathnames"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "T or NIL")
  (example
    (pathname-equal "w1:>documents>four-rects.gredit.newest"
		    "w1:>documents>four-rects.gredit.2")
    "returns t"
    (pathname-equal "W1:>DOCUMENTS>four-rects.GREDIT" "w1:>documents>four-rects.gredit.2")
    "returns t"
    (pathname-equal "W1:>DOCUMENTS>four-rects.GREDIT" nil)
    "returns nil")
  
  (when (and pathname1 pathname2)
    (setq pathname1 (pathname pathname1))
    (setq pathname2 (pathname pathname2))
    (or (eq pathname1 pathname2)
	(and (eq (pathname-host pathname1) (pathname-host pathname2))
	     (equal
		       (pathname-directory pathname1) 
		       (pathname-directory pathname2))
	     (or (and (wildp (pathname-name pathname1))
		      (wildp (pathname-name pathname2)))
		 (#-genera equal #+genera string-equal
			   (pathname-name pathname1) 
			   (pathname-name pathname2)))
	     (or (and (wildp (pathname-type pathname1))
		      (wildp (pathname-type pathname2)))
		 (#-genera equal #+genera string-equal
			   (pathname-type pathname1) 
			   (pathname-type pathname2)))))))

(defginafun pathname-name< (pathname1 pathname2)
  (description "determine lexicographical order of the name component"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes)
  (example (pathname-name< "/home/spenke/aaa.hello" "/home/spenke/bbb.hello")
	   (print (sort (directory "/home/spenke/") #'pathname-name<)))
  (string-lessp (pathname-name pathname1) (pathname-name pathname2)))

;;(defginamacro with-errors-ignored (&body body)
;;  (description "execute body ignoring errors"
;;	       :called-by-application :sometimes)
;;  (comment "In case of an error, NIL is returned, otherwise the value of the last form.")
;;  (example (with-errors-ignored (/ 5 0)))
;;  ;; return NIL in case of an error
;;  #+Genera `(scl:ignore-errors ,(append '(progn) body))
;;;;  #+excl (let ((var (gensym)))
;;;;	   `(let (,var)
;;;;	      (setq ,var (multiple-value-list
;;;;			   (excl:errorset ,(append '(progn) body))))
;;;;	      (when (first ,var) (second ,var)))) 
;;  #+excl  `(ignore-errors ,(append '(progn) body))
;;  #+lucid `(ignore-errors ,(append '(progn) body)) 
;;  )

;; soon replace with-errors-ignored by ignore-errors in the GINA code !
;;#+Genera (setf (symbol-function 'ignore-errors)
;;	       (symbol-function 'scl:ignore-errors))
;;#+excl   (setf (symbol-function 'ignore-errors)
;;	       (symbol-function 'ignore-errors))
#+allegro-v3.1
(import 'excl::ignore-errors)


(defginafun file-exists (pathname)
  (description "check if pathname denotes an existing file"
	       :called-by-gina "in open/save dialogs"
	       :called-by-application :sometimes
	       :result "T or NIL")
  (example (file-exists "fdhgdfgfdg")
	   "returns nil")
  (ignore-errors (probe-file pathname)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;   basics for Inspector
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; this is OS-dependent because it uses PCL functions
#+pcl 'pcl::*pcl-system-date* 

#+pcl
(defun slots (object)
  "return list of all slot-descriptions of an object"
  (loop for slot-description in (pcl::class-slots (class-of object))
	as slot-name = (slot-value slot-description 'pcl::name)
	collect (list slot-name
		      (if (slot-boundp object slot-name) 
			  (slot-value object slot-name)
			  :unbound))))

#+(and clos (not pcl))
(defun slots (object)
  "return list of all slot-descriptions of an object"
  (loop for slot-description in (clos::class-slots (class-of object))
	as slot-name = (slot-value slot-description 'clos::name)
	collect (list slot-name
		      (if (slot-boundp object slot-name) 
			  (slot-value object slot-name)
			  :unbound))))

#+genera
(defun slots (object)
  "return list of all slot-descriptions of an object"
  (loop for slot-description in (clos::class-slots (class-of object))
	as slot-name = (slot-value slot-description 'clos-internals::name)
	collect (list slot-name
		      (if (slot-boundp object slot-name) 
			  (slot-value object slot-name)
			  :unbound))))

'(let ((*print-level* 2))
   (print (slots *application*)))

'(let ((*print-level* 2))
   (print (slots (first (document-list *application*)))))

#+pcl
(defun clos-object-p (object)
  "check-if object is a clos-object"
  (or (eq 'pcl::std-instance (type-of object)) ;; in cmu this type was given
      (eq 'pcl::iwmc-class (type-of object))))

#+(or genera (and clos (not pcl)))
(defun clos-object-p (object)
  "check-if object is a clos-object"
  (eq 'clos::standard-class (type-of (class-of object))))

#+pcl '(clos-object-p *application*)
#+pcl '(clos-object-p '(1 2 3))

#+pcl
(defun clos-class-of (object)
  "find CLOS class of an object as a symbol"
  (slot-value (class-of object) 'pcl::name))

#+(and clos (not pcl))
(defun clos-class-of (object)
  "find CLOS class of an object as a symbol"
  (slot-value (class-of object) 'clos::name))

#+genera
(defun clos-class-of (object)
  "find CLOS class of an object as a symbol"
  (slot-value (class-of object) 'clos-internals::name))


#+pcl
(defun gina-class-of (object)
  "find the gina class of an object"
  (loop for class in (#-cmu pcl::class-class-precedence-list
		      #+cmu pcl::class-precedence-list
				 (class-of object))
      as found = (find-gina-class (slot-value class 'pcl::name))
      when found do (return found)))

#+(and clos (not pcl))
(defun gina-class-of (object)
  "find the gina class of an object"
  (loop for class in (clos:class-precedence-list (class-of object))
      when (find-gina-class (slot-value class 'clos::name))
	return it))

#+genera
(defun gina-class-of (object)
  "find the gina class of an object"
  (loop for class in (clos:class-precedence-list (class-of object))
      when (find-gina-class (slot-value class 'clos-internals::name))
	return it))

'(describe (gina-class-of *application*))
'(describe (gina-class-of (first (document-list *application*))))

#+pcl '(clos-class-of *application*)

#+Genera 
(defginafun replace-returns (string)
  (description "replace return for use in text widgets")
  (substitute (code-char 10) #\return string))

#-Genera
(defginafun replace-returns (string)
  (description "replace return for use in text widgets")
  string)

#+Genera
(defginafun unreplace-returns (string)
  (description "unreplace return for use in text widgets")
  (substitute #\return (code-char 10) string))

#-Genera
(defginafun unreplace-returns (string)
  (description "unreplace return for use in text widgets")
  string)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;   Miscellaneous
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defginafun find-bitmap (bitmap-name 
			 &key (for-clm nil) "T if result is passed to CLM")
  (description 
   "search for a bitmap in gina:*bitmap-directories*"
   :called-by-gina "when setting label bitmaps"
   :called-by-application :sometimes
   :result "the absolute pathname as a string or bitmap-name when not found")
  (comment "The second value (T or NIL) indicates whether found.")
  (unless (and for-clm (toolkit-host *application*))
    ;; pathnames for remote clm-servers cannot be checked !!
    ;; instead user must set XBMLANGPATH of clmd
    (loop for dir in *bitmap-directories*
     for pathname = (merge-pathnames bitmap-name 
				     (subdirectory-wildcard 
				      dir
				      #-genera nil #+genera "bitmap"))
     when (file-exists pathname)
     do (return-from find-bitmap
	       (values #-cmu (namestring pathname)
		       #+cmu (ext:unix-namestring pathname)
		       t))))
  (values bitmap-name nil))

(defginafun shell-command (command-string)
  (description "execute a string as a UNIX command"
	       :called-by-application :sometimes)
  #+excl (excl:shell command-string)
  #+lucid (shell command-string)
  #+cmu (ext:run-program "/bin/sh" (list "-c" command-string))
  #+Genera
  (let (box frame)
    (setq box (make-modal-dialog-box "Please copy into an xterm"))
    (setq frame (make-frame box :margin-width 50 :margin-height 20))
    (make-text frame
	       :value command-string
	       :columns (max 20 (length command-string))
	       :editable nil)
    (pop-up box)))


;; undefine this for any system that has handler-bind
#-(or allegro-v4.0 allegro-v4.1 allegro-v3.1 lcl4.0 genera cmu)
(defmacro handler-bind (bindings &body body)
  (declare (ignore bindings))
  (append '(progn) body))

#+allegro-v3.1
(import 'excl::handler-bind)

#-(or allegro-v4.0 allegro-v4.1 genera cmu)
(defun print-backtrace (cond)
  (declare (ignore cond)))

#+(or allegro-v4.0 allegro-v4.1)
(defun print-backtrace (cond)
  (declare (ignore cond))
  (terpri)
  (let ((frame (debug:next-older-frame (debug:newest-frame))))
     (loop while frame do
       (when (listp (debug:frame-name frame))
         (when (eq (second (debug:frame-name frame)) 'execute) (return))
         (when (eq (first (debug:frame-name frame)) 'clos::method)
           (format t "=> ~s~{ ~s~}~%" (second (debug:frame-name frame))
                     (debug:frame-get-actuals frame))))
       (when (symbolp (debug:frame-name frame))
         (when (member (debug:frame-name frame) 
		       '(gina-callback widget-callback gina-binding-wrapper
			 xtk::run-motif-clx-application))
	   (return))
         (let ((p-name 
                    (package-name (symbol-package (debug:frame-name frame)))))
           (unless (or (equal p-name "COMMON-LISP")
                       (equal p-name "EXCL")
                       (equal p-name "CLOS"))
             (format t "=> ~s~{ ~s~}~%" (debug:frame-name frame)
                     (debug:frame-get-actuals frame)))))
       (setq frame (debug:next-older-frame frame)))))

#+genera
(defun print-backtrace (cond)
  (dbg:with-erring-frame (frame1 cond)
    (loop with frame = (dbg:frame-previous-active-frame frame1)
	  repeat 20 ;; has to be enough
	  while frame
	  do (format t "~&=> ")
	     (write (dbg:get-frame-function-and-args frame))
	     (setq frame (dbg:frame-previous-active-frame frame)))))

#+cmu
(defun print-backtrace (cond)
  (declare (ignore cond))
  (debug:backtrace))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;   Macro for calling GINA functions from the editor
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(ginachapter "Calling GINA Functions from the Editor")

;; using this macro, code to create and modify windows can be executed from the editor
(defginamacro with-application-stopped (&body body)
   (description "stop an application temporarily and use its connections"
		:called-by-gina nil
		:called-by-application "to execute some code from the editor")
  (comment "Within the body you can use *application*, *display* and xtk:*motif-connection*.")
  (comment
    "Using this macro, code to create and modify windows can be executed from the editor.")
  (example (with-application-stopped
	     (pop-up (make-modeless-dialog-box "Test Dialog" :resize t)))
	   "pops up an empty dialog box")
   `(progn
      (when (null *running-applications*)
	(error "there is no application running!!"))
      (unwind-protect
	  (progn
	    ;; make connections accessible
	    (setq *application* (first *running-applications*))
	    (setq *display* (display *application*))
	    (setq xtk:*x-display* *display*)
	    (setq xtk:*motif-connection* (xtk-connection *application*))
	    ;; stop the application process to prevent two processes from
	    ;; reading events and callbacks
	    (format t "Using Application ~a ...~%" (name *application*))
	    (#+Genera si:process-disable
	     #+excl   mp:process-disable
	     #+lucid  deactivate-process
     	     #+cmu    identity (process *application*))
	    ,(append '(progn) body))
	;; send of generated X requests
	(xlib:display-force-output *display*)
	;; restart the application process again
	(#+Genera si:process-enable
	 #+excl   mp:process-enable
	 #+lucid  activate-process
	 #+cmu	  identity (process *application*)))))

(defginafun rotate-applications ()
  (description "bring the next application to front of *running-applications*"
	       :called-by-gina nil
	       :called-by-application
	       "to use another application by with-application-stopped")
  (setq *running-applications*
	(append (cdr *running-applications*) (list (first *running-applications*)))))

(defginafun focus-application (&optional (number 1))
  (description "bring another application to front of *running-applications*"
	       :called-by-gina nil
	       :called-by-application
	       "to set *application* or use another application by with-application-stopped")
  (example (focus-application 3)
	   "focusses on the third running application")
  (loop repeat (1- number)
	do (rotate-applications))
  (setq *application* (first *running-applications*)))

(defginafun focus-this-application (app doc)
  (description "bring the given application to front of *running-applications*")
  (when (member app *running-applications*)
    (loop while (not (eq app (first *running-applications*)))
	do (rotate-applications))
    (setf *document* doc)
    (format t "Application ~a will now be used by macro with-application-stopped~%*document* is now ~s~%"
	    (name (first *running-applications*))
	    doc)))

(defginamacro defginapackage (package &rest defpackage-options)
  (description "define a new package using at least GINA and CL"
	       :called-by-gina "only in the demo applications"
	       :called-by-application 
	       "once at the beginning of your application")
  (comment "This macro calls defpackage with implementation dependent parms.")
  #-(or allegro-v4.0 allegro-v4.1 lucid genera cmu)
  `(in-package ,package :use '(:GINA :lisp))
  #+genera
  `(defpackage ,package (:use :GINA :future-common-lisp) ,@defpackage-options)
  #+(or allegro-v4.0 allegro-v4.1 cmu)
  `(defpackage ,package (:use :GINA :common-lisp) ,@defpackage-options)
  #+lucid
  `(eval-when (compile load eval)
     (defpackage ,package (:use :GINA lisp lcl) ,@defpackage-options)))

