;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: steer-tutorial.lisp
;;;  Author: Heeger
;;;  Description:
;;;  Creation Date: 6/93
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; This file is just an introduction to the steerable-basis and the
;; steerable-pyramid in OBVIUS.  For details on what these things are,
;; see the following papers:
;; * Freeman and Adelson (1991), The design and use of steerable
;;   filters.  IEEE Trans. on Pattern Anal. and Mach. Intelligence,
;;   13:891-906.
;; * Simoncelli, Freeman, Adelson, & Heeger (1992) Shiftable
;;   multi-scale transforms.  IEEE Transactions on Information
;;   Theory,38:587-607.

(obv-require "steer")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Orientation-Selective, Steerable Filters:

;; Make a pair of orientation selective filters and look at
;; their orientation selective responses:

(setq hfilt (make-separable-filter
	     '(0.233 0.534 0.233)
	     '(-0.459 0.0 0.459)))
(setq vfilt (make-separable-filter
	     '(-0.459 0.0 0.459)
	     '(0.233 0.534 0.233)))
(setq disc (make-disc '(64 64)))
(setq hresponse (apply-filter hfilt disc))
(setq vresponse (apply-filter vfilt disc))

;; The apply-filter method does a convolution of the filter kernel
;; with the image.  Apply-filter is also smart about subsampling and
;; handling edges (take a look at the OBVIUS documentation for
;; details).

;; Let's look at the frequency responses of these two filters:

(power-spectrum hfilt :dimensions '(64 64))
(power-spectrum vfilt :dimensions '(64 64))

;; These two filters were designed to tile (evenly cover) orientation.
;; The sum of their frequency responses is an annulus of spatial
;; frequencies.

(+. (power-spectrum hfilt :dimensions '(64 64))
    (power-spectrum vfilt :dimensions '(64 64)))

;; A zone plate is an image of a radially symmetric frequency sweep,
;; cos(r-sqrd).  Zone plates are another way to look at orientation
;; and frequency selectivity.

(setq zone (make-zone-plate '(64 64)))
(square (apply-filter hfilt zone))
(square (apply-filter vfilt zone))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; "Steering" the filter responses:

;; Hfilt and vfilt are made up of two one-dimensional (sampled)
;; functions.  One of those functions is a low-pass filters and the
;; other is the first derivative of that low-pass filter.  So hfilt
;; and vfilt are directional derivative operators, hfilt computes the
;; derivative in the x-direction and vfilt computes the derivative in
;; the y-direction.

;; To take derivatives in the other directions, we can use simple
;; combinations of hfilt and vfilt.  For example, the derivative
;; operators for the two diagonal orientations are:

(setq pfilt (mul (/ (sqrt 2) 2) (add hfilt vfilt)))
(setq qfilt (mul (/ (sqrt 2) 2) (sub hfilt vfilt)))

;; Let's look at their frequency responses:

(power-spectrum pfilt :dimensions '(64 64))
(power-spectrum qfilt :dimensions '(64 64))

;; And the responses of these filters:

(setq presponse (apply-filter pfilt disc))
(setq qresponse (apply-filter qfilt disc))

;; Here we've convolved the original image with these two new filters.
;; The two new filters are just linear sums/differences of hfilt and
;; vfilt.  We actually didn't have to do these convolutions since we
;; had already computed hresponse and vresponse (the responses to
;; hfilt and vfilt).  The responses to the diagonal filters are linear
;; sums/differences of the responses to hfilt and vfilt:

(mean-square-error presponse (mul (/ (sqrt 2) 2) (add hresponse vresponse)))
(mean-square-error qresponse (mul (/ (sqrt 2) 2) (sub hresponse vresponse)))

;; Why does this work?  It's because convolution distributes across
;; addition:
;;    (hfilt * image) + (vfilt * image) = (hfilt + vfilt) * image
;; where * means convolution.

;; These first derivative filters are examples of what we call
;; "steerable" filters.  We start with a pair of "basis filters"
;; (hfilt and vfilt) to compute a pair of "basis images" (hresponse
;; and vresponse).  Then we can get derivatives in any other
;; orientation by taking linear sums of hresponse and vresponse.
;; Compile the following function (using C-c c):

(defun steer1 (hresponse vresponse angle)
  (add (mul (cos angle) hresponse)
       (mul (sin angle) vresponse)))

;; Then evaluate this expression to make a sequences of rotating
;; derivatives.  Use C-M-right to display the sequences:

(make-image-sequence (loop for i from 0 below 10
			   collect
			   (steer1 hresponse vresponse (* 2-pi (/ i 10)))))

(load-image (merge-pathnames "images/einstein"
			     obv::*obvius-directory-path*))
(progn
  (setq hresp (apply-filter hfilt einstein))
  (setq vresp (apply-filter vfilt einstein))
  (make-image-sequence (loop for i from 0 below 10
			     collect
			     (steer1 hresp vresp (* 2-pi (/ i 10))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; 2nd order derivatives:

;; The "steering" property is not restricted to first derivative
;; filters.  We can also steer 2nd derivative filters (and their
;; responses), but for 2nd derivatives we need to start with three
;; basis filters (recall that we had only two 1st derivative filters).
;; With higher-order derivatives (3rd order, etc) we need more and
;; more basis filters.  We call the basis filters a basis set because
;; they span the space of all rotations of each basis filter.  As is
;; typical of a linear basis, we have some freedom about how to choose
;; the basis filters.  For 2nd derivatives we could choose three
;; oriented directional derivatives, rotated 60 degrees from each
;; other.  Instead we choose to use three separable filters:

(setq g2-filters (make-g2-steerable-filters))
(first g2-filters)
(second g2-filters)
(third g2-filters)

;; Their power spectra:

(power-spectrum (first g2-filters) :dimensions '(64 64))
(power-spectrum (second g2-filters) :dimensions '(64 64))
(power-spectrum (third g2-filters) :dimensions '(64 64))

;; Make separable steerable basis set for default steerable filter
;; (2nd deriv of Gaussian, "G2")
(setq zone-sb (make-steerable-basis zone))

;; Steer to 45 degrees: this could be done by convolving with a new
;; filter.  But we compute it here using the steer function that takes
;; an appropriate linear sum of the the basis images.  The arguments
;; to steer are a steerable-basis and an angle specification.  The
;; angle specification can be either a number or an image of angles.
(setq zone-0 (steer zone-sb 0.0))
(setq zone-45 (steer zone-sb (/ pi 4)))

;; Make a steerable basis set for a quadrature pair of filters.  The
;; default filters are 2nd deriv of Gaussian (G2), and it's Hilbert
;; transform (H2).  We haven't written a display routine for the
;; quadrature steerable basis object (for the time being) so nothing
;; gets displayed.
(setq zone-qsb (make-quadrature-steerable-basis zone))

;; To display the G2 (even phase) basis set:
(even-steerable-basis zone-qsb)   

;; To display the H2 (odd phase) basis set (note there are 4 of them):
(odd-steerable-basis zone-qsb)    

;; Steer the odd phase basis set to 30 degrees:
(steer (odd-steerable-basis zone-qsb) (/ pi 6))

;;; Compute "energy", square-root[ G2^2 + H2^2 ], steered to pi/6
;;; radians:
(directional-magnitude zone-qsb (/ pi 6))

;; Make a steerable basis set for einstein
(setq al (gauss-out einstein))
(setq basis (make-quadrature-steerable-basis al))

;; Steer it to different orientations (use C-M-right to display the
;; sequence):
(make-image-sequence (loop for i from 0 below 10
			   collect
			   (steer (even-steerable-basis basis) (* i (/ 2-pi 10)))))

;; Compute energy at different orientations:
(directional-magnitude basis 0.0)
(directional-magnitude basis (/ pi 2))

;; Steer it to different phases and display it as "motion without
;; movement" illusion (use C-M-right to display the sequence):
(progn
  (setq even (steer (even-steerable-basis basis) 0.0))
  (setq odd (steer (odd-steerable-basis basis) 0.0))
  (make-image-sequence (loop for i from 0 below 8
			     for phase = (* i (/ 2-pi 8))
			     collect
			     (add (mul (cos phase) even)
				  (mul (sin phase) odd)))))

;; Space-variant filtering.  Remember, the angle arguement to the
;; steer function can be an image, in which case the basis is steered
;; to a different angle at each pixel.
(setq angle-image (make-synthetic-image
		   '(128 128)
		   #'(lambda (y x) (atan y x))))
(setq even (steer (even-steerable-basis basis) angle-image))
(setq odd (steer (odd-steerable-basis basis) angle-image))
(make-image-sequence (loop for i from 0 below 8
			   for phase = (* i (/ 2-pi 8))
			   collect
			   (add (mul (cos phase) even)
				(mul (sin phase) odd))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Orientation Map:

;;; Compile this function (using C-c c) that computes, for each pixel,
;;; the eigenvector corresponding to the smallest eigenvalue of a 2x2
;;; image matrix:
(defmethod min-eigenvector ((2x2-matrix image-matrix) &key ->)
  (unless (and (= (row-dim 2x2-matrix) 2)
	       (= (col-dim 2x2-matrix) 2))
    (error "min-eigenvector only implemented for 2x2 image matrices"))
  (with-result ((res ->) (list :class 'image-matrix
			       :size '(1 2)
			       :sub-viewable-spec
			       (list :class 'image
				     :dimensions (dimensions 2x2-matrix))
			       :display-type 'pasteup))
    (let ((mat00 (aref (data 2x2-matrix) 0 0))
	  (mat01 (aref (data 2x2-matrix) 0 1))
	  (mat10 (aref (data 2x2-matrix) 1 0))
	  (mat11 (aref (data 2x2-matrix) 1 1))
	  (res0 (aref (data res) 0 0))
	  (res1 (aref (data res) 0 1)))
      (with-local-viewables
	  ((norm (similar mat00))
	   (tmp1 (similar mat00))
	   (tmp2 (similar mat00))
	   (trace (add mat00 mat11))
	   (determinant (sub (mul mat00 mat11 :-> tmp1)
			     (mul mat01 mat10 :-> tmp2)))
	   (descriminant (square-root (sub (square trace :-> tmp1)
					   (mul 4.0 determinant :-> tmp2)
					   :-> tmp1)))
	   (eigenvalue (mul 0.5 (sub trace descriminant :-> tmp1))))
	;;(display (copy eigenvalue :-> "eigenvalue"))
	(fill! res0 1.0)
	(div (sub eigenvalue mat00 :-> res1) mat01 :-> res1)
	(square-root (add (square res0 :-> tmp1) (square res1 :-> tmp2)
			  :-> norm) :-> norm)
	(div res0 norm :-> res0)
	(div res1 norm :-> res1)))
    res))

;;; Now, evaluate the following expressions to compute an orientation
;;; map of a disc image:

(setq im (make-disc '(64 64)))

(progn
  (setq hfilt (make-separable-filter
	       '(0.233 0.534 0.233)
	       '(-0.459 0.0 0.459)
	       :edge-handler :dont-compute))
  (setq vfilt (make-separable-filter
	       '(-0.459 0.0 0.459)
	       '(0.233 0.534 0.233)
	       :edge-handler :dont-compute))

  (setq fx (apply-filter hfilt im))
  (setq fy (apply-filter vfilt im))

  (setq mat (make-image-matrix (list (list (square fx) (mul fx fy))
				     (list (mul fx fy) (square fy)))))
  (setq bmat (blur mat :edge-handler :dont-compute))
  (setq eigenvect (min-eigenvector bmat))
  (setq gradient-mag (square-root (add (square fx) (square fy))))
  (mul eigenvect (>. gradient-mag *tolerance*) :-> eigenvect)
  (display
   (setq orientation-map (make-image-pair (list (aref (matrix eigenvect) 0 1)
						(aref (matrix eigenvect) 0 0))))
   'vector-field)
  (setp :skip 2 :scale 1 :zoom 4 :arrow-heads nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; *** Steerable Pyramid (not done yet) ***

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Local Variables:
;;; buffer-read-only: t 
;;; fill-column: 79
;;; End:
