;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: 2d-filters-tutorial.lisp
;;;  Author: Heeger
;;;  Description:
;;;  Creation Date: 6/93
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; You should run this tutuorial only after running:
;;;   <obvius>/tutorials/signal-processing/linear-systems-tutorial.lisp
;;;   <obvius>/tutorials/signal-processing/sampling.lisp

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; 2D DFT:

;; First we set up our familiar Fourier transform matrices.  The rows
;; of these matrices are sines (imaginary part) and cosines (real
;; part) that make up the projection functions of the DFT for a finite
;; length, 1D signal (see linear-systems-tutorial.lisp).

(progn
  (setq Fourier-real (make-array '(32 32) :element-type 'single-float))
  (loop for k from 0 below 32 do
	(loop for n from 0 below 32 do
	      (setf (aref Fourier-real k n)
		    (* (sqrt 1/32) (cos (* 2/32 pi k n))))))
  (setq Fourier-imag (make-array '(32 32) :element-type 'single-float))
  (loop for k from 0 below 32 do
	(loop for n from 0 below 32 do
	      (setf (aref Fourier-imag k n)
		    (* (sqrt 1/32) (sin (* 2/32 pi k n)))))))

(make-complex-image (list (make-image Fourier-imag)
			  (make-image Fourier-real))
		    :-> "DFT projection functions")

;; In Obvius, a "complex-image" is a data structure that contains two
;; sub-images, one is the real-part and the other is the imaginary
;; part.  The make-complex-image function takes a list of two images
;; (first one is the imaginary part and the second one is the real
;; part).  The imaginary part is displayed on the left and the real
;; part is displayed on the right.

;; These very same matrices are used to compute the DFT of 2D images,
;; simply by applying the 1D DFT to both the rows and the cols of the
;; 2D image.  Let's try it on a sinusoidal grating test image:

(setq sin (make-sin-grating '(32 32) :x-freq (/ (* 2 pi 8) 64)
			    :y-freq (/ (* 2 pi 8) 64)))

;; First we multiply by the Fourier matrices on the left, that is,
;; computing the 1D DFT of each of the columns of the image:

(setq DFT-cols
      (make-complex-image
       (list
	(make-image (matrix-mul Fourier-imag (data sin)))
	(make-image (matrix-mul Fourier-real (data sin))))))

;; The first column of the real part of DFT-cols is the real part of
;; the DFT of the first column of sin:

(make-slice (real-part DFT-cols) :x 1 :-> "1st col of DFT")
(make-image (matrix-mul Fourier-real (columnize (data (make-slice sin :x 1))))
	    :-> "DFT of 1st col")

;; Likewise for the 2nd col (and all the other cols).  And likewise
;; for the imaginary part of each col:

(make-slice (imaginary-part DFT-cols) :x 1 :-> "1st col of DFT")
(make-image (matrix-mul Fourier-imag (columnize (data (make-slice sin :x 1))))
	    :-> "DFT of 1st col")

;; We use the same Fourier matrix to compute DFTs of each of the rows
;; of the image.  To compute the DFT of the cols we took the image
;; data, D, and the Fourier matrix, F, and multiplied on the left:
;;          DFT cols = F D
;; Now we multiply on the right to get the DFT of the rows:
;;          DFT rows = D F

(setq DFT-rows
      (make-complex-image
       (list
	(make-image (matrix-mul (data sin) Fourier-imag))
	(make-image (matrix-mul (data sin) Fourier-real)))))

;; We get the full 2D DFT by putting the two steps together.  First we
;; compute the DFT of the cols of an image.  Then we take that result
;; and compute the DFT of its rows:

(progn
  (setq DFT-real (sub (matrix-mul (data (real-part DFT-cols)) Fourier-real)
		      (matrix-mul (data (imaginary-part DFT-cols)) Fourier-imag)))
  (setq DFT-imag (add (matrix-mul (data (real-part DFT-cols)) Fourier-imag)
		      (matrix-mul (data (imaginary-part DFT-cols)) Fourier-real)))
  (setq DFT (make-complex-image
	     (list (make-image DFT-imag)
		   (make-image DFT-real)))))

And the magnitude:

(magnitude DFT)

;; What do the various transform coefficients mean?  The top left
;; corner of the DFT is the dc component (i.e., the average of the
;; image), computed by taking the average of the averages of each of
;; the columns.  The entire first row is the DFT of the averages of
;; the columns.  That is, we could have first computed the average of
;; each column to get a 1D signal and then taken the DFT of that 1D
;; signal.  For the second row, we could have first computed the 1
;; cycle per image component of each column to get a 1D signal and
;; then taken the DFT of that 1D signal.  For the first column, we
;; could have first computed the average of each row to get a 1D
;; signal, and then taken the DFT of that 1D signal.  Likewise every
;; other row and column.

;; In Obvius, all of the DFT can be computed efficiently using the fft
;; routine:

(setq fft (fft sin))
(mean-square-error DFT (fft sin))

;; Let's look at the DFT of some real images:

(load-image (merge-pathnames "images/einstein" obv::*obvius-directory-path*))
(setq fft-einstein (fft (gauss-out einstein) :post-center t))
(setq mag-einstein (magnitude fft-einstein))

;; Note that you don't see much.  That's because the dc component (and
;; low frequency components dominate).  Often, it is useful to display
;; the log of the magnitude of the DFT:

(setq log-mag-einstein (point-operation mag-einstein #'log))

;; Load some other images and look at their DFTs.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; 2D DFT Examples:

;; Here we synthesize a number of test images and look at their power
;; spectra.  In each case, try to guess what the power spectrun will
;; look like before you compute it.

;; We'll zoom these images so they're easier to see:
(set-default 'gray :zoom 2)

(display (setq constant (fill! (make-image '(64 64)) 1.0))
	 'gray :pedestal 0 :scale 1)
(setq constant-ft (power-spectrum constant :center t))

(setq impulse (make-impulse '(64 64)))
(display (setq impulse-power (power-spectrum impulse :center t))
	 'gray :pedestal 0.0 :scale 2.5e-4)

(setq sin-x8 (make-sin-grating '(64 64) :x-freq (/ (* 2 pi 8) 64) :y-freq 0))
(setq sin-x8-power (power-spectrum sin-x8 :center t))

(setq sin-x4 (make-sin-grating '(64 64) :x-freq (/ (* 2 pi 4) 64) :y-freq 0))
(setq sin-x4-power (power-spectrum sin-x4 :center t))

(setq sin-x16 (make-sin-grating '(64 64) :x-freq (/ (* 2 pi 16) 64) :y-freq 0))
(setq sin-x16-power (power-spectrum sin-x16 :center t))

(setq sin-y8 (make-sin-grating '(64 64) :y-freq (/ (* 2 pi 8) 64) :x-freq 0))
(setq sin-y8-power (power-spectrum sin-y8 :center t))
 
(setq sin-x8-y8 (make-sin-grating '(64 64)
				  :x-freq (/ (* 2 pi 8) 64)
				  :y-freq (/ (* 2 pi 8) 64)))
(setq sin-x8-y8-power (power-spectrum sin-x8-y8 :center t))

(let ((sigma-x 4)
      (sigma-y 4))
  (setq gaussian1 (make-synthetic-image
		   '(64 64)
		   #'(lambda (y x)
		       (exp (- (+ (/ (sqr x) (* 2 (sqr sigma-x)))
				  (/ (sqr y) (* 2 (sqr sigma-y)))))))
		   :x-range '(-32 31)
		   :y-range '(-32 31))))
(setq gaussian1-power (power-spectrum gaussian1 :center t))

(let ((sigma-x 2)
      (sigma-y 2))
  (setq gaussian2 (make-synthetic-image
		   '(64 64)
		   #'(lambda (y x)
		       (exp (- (+ (/ (sqr x) (* 2 (sqr sigma-x)))
				  (/ (sqr y) (* 2 (sqr sigma-y)))))))
		   :x-range '(-32 31)
		   :y-range '(-32 31))))
(setq gaussian2-power (power-spectrum gaussian2 :center t))

(let ((sigma-x 2)
      (sigma-y 1))
  (setq gaussian3 (make-synthetic-image
		   '(64 64)
		   #'(lambda (y x)
		       (exp (- (+ (/ (sqr x) (* 2 (sqr sigma-x)))
				  (/ (sqr y) (* 2 (sqr sigma-y)))))))
		   :x-range '(-32 31)
		   :y-range '(-32 31))))
(setq gaussian3-power (power-spectrum gaussian3 :center t))

(let ((sigma-x 4)
      (sigma-y 4)
      (k 8))
  (setq gabor1 (make-synthetic-image
		'(64 64)
		#'(lambda (y x)
		    (* (exp (- (+ (/ (sqr x) (* 2 (sqr sigma-x)))
				  (/ (sqr y) (* 2 (sqr sigma-y))))))
		       (sin (/ (* 2 pi k x) 64))))
		:x-range '(-32 31)
		:y-range '(-32 31))))
(setq gabor1-power (power-spectrum gabor1 :center t))

(let ((sigma-x 2)
      (sigma-y 2)
      (k 8))
  (setq gabor2 (make-synthetic-image
		'(64 64)
		#'(lambda (y x)
		    (* (exp (- (+ (/ (sqr x) (* 2 (sqr sigma-x)))
				  (/ (sqr y) (* 2 (sqr sigma-y))))))
		       (sin (/ (* 2 pi k x) 64))))
		:x-range '(-32 31)
		:y-range '(-32 31))))
(setq gabor2-power (power-spectrum gabor2 :center t))

(let ((sigma-x 2)
      (sigma-y 1)
      (k 8))
  (setq gabor3 (make-synthetic-image
		'(64 64)
		#'(lambda (y x)
		    (* (exp (- (+ (/ (sqr x) (* 2 (sqr sigma-x)))
				  (/ (sqr y) (* 2 (sqr sigma-y))))))
		       (sin (/ (* 2 pi k x) 64))))
		:x-range '(-32 31)
		:y-range '(-32 31))))
(setq gabor3-power (power-spectrum gabor3 :center t))

(let ((sigma-x 2)
      (sigma-y 1)
      (k 16))
  (setq gabor4 (make-synthetic-image
		'(64 64)
		#'(lambda (y x)
		    (* (exp (- (+ (/ (sqr x) (* 2 (sqr sigma-x)))
				  (/ (sqr y) (* 2 (sqr sigma-y))))))
		       (sin (/ (* 2 pi k x) 64))))
		:x-range '(-32 31)
		:y-range '(-32 31))))
(setq gabor4-power (power-spectrum gabor4 :center t))

(let ((sigma-x 1)
      (sigma-y 1/2)
      (k 16))
  (setq gabor5 (make-synthetic-image
		'(64 64)
		#'(lambda (y x)
		    (* (exp (- (+ (/ (sqr x) (* 2 (sqr sigma-x)))
				  (/ (sqr y) (* 2 (sqr sigma-y))))))
		       (sin (/ (* 2 pi k x) 64))))
		:x-range '(-32 31)
		:y-range '(-32 31))))
(setq gabor5-power (power-spectrum gabor5 :center t))

(setq separable-function
      (make-synthetic-image
       '(64 64)
       #'(lambda (y x)
	   (* y (exp (- (/ (sqr y) (sqr 1/12))))
	      x (exp (- (/ (sqr x) (sqr 1/12))))
	      ))))
(setq separable-magnitude (sqrt. (power-spectrum separable-function :center t)))

(setq non-separable-function
      (make-synthetic-image
       '(64 64)
       #'(lambda (y x)
	   (* (+ x y)
	      (exp (- (+ (/ (sqr x) (sqr 1/12)) (/ (sqr y) (sqr 1/12)))))))))
(setq non-separable-magnitude (sqrt. (power-spectrum non-separable-function :center t)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; 2D Filtering:

;; Make a pair of orientation selective filters and look at
;; their orientation selective responses:

(setq hfilt (make-filter '((-0.107 0.0 0.107)
			   (-0.245 0.0 0.245)
			   (-0.107 0.0 0.107))))

(setq vfilt (make-filter '((-0.107 -0.245 -0.107)
			   ( 0.0    0.0    0.0)
			   ( 0.107  0.245  0.107))))

(setq disc (make-disc '(64 64)))
(apply-filter hfilt disc)
(apply-filter vfilt disc)

;; The apply-filter method does a convolution of the filter kernel
;; with the image.  Apply-filter is also smart about subsampling and
;; handling edges (take a look at the OBVIUS documentation for
;; details).

;; Let's look at the frequency responses of these two filters:

(power-spectrum hfilt :dimensions '(64 64))
(power-spectrum vfilt :dimensions '(64 64))

;; These two filters were designed to tile (evenly cover) orientation.
;; The sum of their frequency responses is an annulus of spatial
;; frequencies.

(+. (power-spectrum hfilt :dimensions '(64 64))
    (power-spectrum vfilt :dimensions '(64 64)))

;; A zone plate is an image of a radially symmetric frequency sweep,
;; cos(r-sqrd).  Zone plates are another way to look at orientation
;; and frequency selectivity.

(setq zone (make-zone-plate '(64 64)))
(square (apply-filter hfilt zone))
(square (apply-filter vfilt zone))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Separable Filters:

;; A separable function is one that can be written as the product of
;; two functions, each of which depends on only one variable:

;;        f(x,y) = f1(x) f2(y)

;; An NxN non-separable filter requires N^2 multiplications for each
;; pixel.

;; For separable filters, we can do the convolutions more efficiently.
;; We convolve with each of one-dimensional filters, one of them
;; applied to the rows and the other applied to the columns.  This
;; requires only 2N multiplications.  When N is big (the filter
;; kernels are big), this savings is significant.

;; The two filters defined above can be expressed as separable
;; filters.  Let's just consider the horizonal filters (hfilt).  The
;; two dimensional convolution kernel for that filter is a 3x3 matrix
;; of values that can be expressed as the outer product of two
;; vectors:

(setq hfilt-kernel (make-matrix '((-0.107 0.0 0.107)
				  (-0.245 0.0 0.245)
				  (-0.107 0.0 0.107))))
(setq hfilt-sep-kernel (outer-product (make-matrix '(0.233 0.534 0.233))
				      (make-matrix '(-0.459 0.0 0.459))))
(mean-square-error hfilt-kernel hfilt-sep-kernel)

;; Consequently, we can do the convolutions separably.  The :direction
;; keyword specifies which direction to apply a filter in (0 means
;; apply the filter across the rows and 1 means apply it down the
;; columns):

(progn
  (setq filt1 (make-filter '(0.233 0.534 0.233)))
  (setq filt2 (make-filter '(-0.459 0.0 0.459)))
  (setq convolved-rows (apply-filter filt2 disc :direction 0))
  (setq convolved-row-and-cols (apply-filter filt1 convolved-rows :direction 1)))
(mean-square-error convolved-row-and-cols (apply-filter hfilt disc))

;; Why does this work?  It's because convolution is associative:
;;      (filt1 * filt2) * image = filt1 * (filt2 * image)
;; where * means convolution.

;; We can redefine hfilt and vfilt as separable-filters so that OBVIUS
;; will keep track of which kernel to apply to the rows and which to
;; apply to the columns.  The function make-separable-filter takes two
;; arguments.  They can be two filters or two lists.  The first one is
;; the y- (column) filter and the second one is the x- (row) filter.

(setq hfilt (make-separable-filter
	     '(0.233 0.534 0.233)
	     '(-0.459 0.0 0.459)))
(setq vfilt (make-separable-filter
	     '(-0.459 0.0 0.459)
	     '(0.233 0.534 0.233)))
(setq hresponse (apply-filter hfilt disc))
(setq vresponse (apply-filter vfilt disc))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Separable Filtering and Matrix Multiplication:

;; Convolution with a separable filter can be expressed as matrix
;; multiplication using two Toeplitz (circulant) matrices:
;;           C D R
;; where D is the image data, C and R are Toeplitz matrices.  The rows
;; of C filter the columns of D and the columns of R filter the rows
;; of D.

;; Here's an example, using the separable hfilt defined above:

(setq hfilt (make-separable-filter
	     '(0.233 0.534 0.233)
	     '(-0.459 0.0 0.459)))

(progn
  (setq Cmatrix (make-array '(64 64) :element-type 'single-float))
  (setq Cmatrix-row (make-matrix (append '(0.233 0.534 0.233) (list-of-length 61 0.0))))
  (loop for i from 0 below 64
	for shift = (- i 1)
	for row = (displaced-row i Cmatrix)
	do
	(circular-shift Cmatrix-row :x-shift shift :-> row))
  (setq Rmatrix (make-array '(64 64) :element-type 'single-float))
  (setq Rmatrix-row (make-matrix (append '(-0.459 0.0 0.459) (list-of-length 61 0.0))))
  (loop for i from 0 below 64
	for shift = (- i 1)
	for row = (displaced-row i Rmatrix)
	do
	(circular-shift Rmatrix-row :x-shift shift :-> row))
  (matrix-transpose Rmatrix :-> Rmatrix))

(display (make-image Cmatrix) 'gray :zoom 4 :pedestal -.1 :scale .2)
(display (make-image Rmatrix) 'gray :zoom 4 :pedestal -.1 :scale .2)

(progn
  (setq new-hresponse (make-image '(64 64)))
  (matrix-mul (matrix-mul Cmatrix (data disc)) Rmatrix :-> (data new-hresponse))
  new-hresponse)

(mean-square-error hresponse new-hresponse)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Non-Separable Filters:

;; Separable filters aren't sufficient for certain applications.  For
;; example, here's a a diagonal filter:

(setq dfilt (make-filter '((-0.2139 -0.2451 0.0)
			    (-0.2451 0.0 0.2451)
			    (0.0 0.2451 0.2139))))

;; Frequency response:

(power-spectrum dfilt :dimensions '(64 64))
(square (apply-filter dfilt zone))

;; Response to disc:

(apply-filter dfilt disc)

;; Using non-separable filters is less efficient.  In the above
;; example it doesn't make much of a difference because the filter is
;; very small (3x3).  But it can make a substantial difference for
;; larger filters.

;; Certain nonseparable filters can be expressed as a linear sum of
;; separable filters.  For example, our diagonal filter is just the
;; sum of hfilt plus vfilt:

(setq new-dfilt (add hfilt vfilt))
(mean-square-error new-dfilt dfilt)

;; We can compute the response of dfilt using two separable
;; convolutions:

(setq hresponse (apply-filter hfilt disc))
(setq vresponse (apply-filter vfilt disc))
(setq d1response (add hresponse vresponse))

;; So far, this looks like a silly thing to do.  Computing dresponse
;; directly (using a nonseparable filter as above) took 9 multiplies
;; per pixel.  The new way (adding hresponse plus vresponse) takes a
;; total of 12 multiplies per pixel.  But there is an advantage when
;; we are interested in other orientations as well.

;; For example, the other diagonal is just the difference between
;; hfilt and vfilt:

(setq other-dfilt (sub hfilt vfilt))
(power-spectrum other-dfilt :dimensions '(64 64))

;; And the response to this filter is just the difference between the
;; responses, hresponse minus vresponse:

(setq d2response (sub hresponse vresponse))

;; Now let's get all four responses (horizontal, vertical, and the two
;; diagonals) for the einstein image:

(set-default 'gray :zoom 1)
(setq hresponse (apply-filter hfilt einstein))
(setq vresponse (apply-filter vfilt einstein))
(setq d1response (add hresponse vresponse))
(setq d2response (sub hresponse vresponse))

;; Using the separable filters in this way is a bit more efficient
;; than doing nonseparable convolutions.  If the filters were bigger,
;; then it would make even more of a difference.  We'll use this trick
;; (computing nonseparable responses as linear sums of separable
;; responses) a lot in the tutorial on steerable filters and steerable
;; pyramids.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Quadrature Pairs and Energy Mechanisms:

;; A quadrature pair (also called a Hilbert transform pair) is a pair
;; of filters that have the same frequency response magnitude, but
;; that differ by 90 degrees in phase.  The Hilbert transform is a 90
;; deg phase shift: it keeps the Fourier magnitude the same but shifts
;; the phase of each frequency component by 90 degrees.  For example,
;; the Hilbert transform of a cosinusoid is a sinusoid.

;; Here we construct two separable filters that respond to vertical
;; image features.  The y-component of each filter is the same
;; low-pass.  The x-components are different from one another.  One of
;; them is even symmetric (0 phase) and the other one is odd symmetric
;; (90 deg phase).

(setq 1d-low-pass (make-filter '(7.598E-4 0.01759 0.1660 0.6383
				 1.0
				 0.6383 0.1660 0.01759 7.5987E-4)))
(setq 1d-even-filt (make-filter '(0.009356 0.1148 0.3964 -0.06010
				  -0.9213
				  -0.06010 0.3964 0.1148 0.009356)))
(setq 1d-odd-filt (make-filter '(-0.01045 -0.06578 0.1063 0.8042
				 0.0
				 -0.8042 -0.1063 0.06578 0.01045)))

(display (setq even-filt (make-separable-filter 1d-low-pass 1d-even-filt
						:edge-handler :reflect1))
	 'gray :zoom 8)
(display (setq odd-filt (make-separable-filter 1d-low-pass 1d-odd-filt
					       :edge-handler :reflect1))
	 'gray :zoom 8)

;; Let's look at the frequency responses:

(set-default 'gray :zoom 2)
(setq even-power (power-spectrum even-filt :center t :dimensions '(64 64)))
(setq odd-power (power-spectrum odd-filt :center t :dimensions '(64 64)))

;; Note that the power spectra of the two filters are very similar.
;; To get a better comparision, let's take a horizontal slice through
;; the middle of each power spectrum, and overlay the two plots.

(display (make-viewable-sequence
	  (list (make-slice even-power :y 32)
		(make-slice odd-power :y 32)))
	 'overlay)

;; These two filters are perfect Hilbert transforms of one another but
;; they are pretty close.

;; Let's look at the responses to a zone plate:

(setq zone-even (apply-filter even-filt zone))
(setq zone-odd (apply-filter odd-filt zone))
(setq zone-energy (add (square zone-even) (square zone-odd)))

;; The last step computed the sum of the squares of the even and odd
;; phase responses.  We call this an "energy mechanism", because it
;; responds to the "local Fourier energy" of the image, regardless of
;; the "local phase".

;; To demonstrate why these energy mechanisms are useful, let's
;; consider the problem of analyzing orientation in an image.  We'd
;; like to measure the local orientation for each small patch of the
;; image.  As simple examples, let's consider a vertical bar and a
;; vertical line.

(setq edge (make-left '(64 64)))
(setq line (mul edge (circular-shift edge :x 31)))

;; Both of these images have the same (vertical) orientation, but
;; their phases are different.  We want to measure the orientation of
;; these image features without having to worry about what the phase
;; is.  The energy mechanism gives us a response that is more or less
;; phase independent:

(progn
  (setq edge-even (apply-filter even-filt edge))
  (setq edge-odd (apply-filter odd-filt edge))
  (setq edge-energy (add (square edge-even) (square edge-odd))))

(progn
  (setq line-even (apply-filter even-filt line))
  (setq line-odd (apply-filter odd-filt line))
  (setq line-energy (add (square line-even) (square line-odd))))

;; Let's look at the vertical energy on a real image:

(set-default 'gray :zoom 1)
(display
 (setq vert-energy (add (square (apply-filter even-filt einstein))
			(square (apply-filter odd-filt einstein))))
 'gray :pedestal 0 :scale 1e4 :zoom 1)

;; And the horizontal energy:

(display (setq hor-even-filt (make-separable-filter 1d-even-filt 1d-low-pass
						    :edge-handler :reflect1))
	 'gray :zoom 8)
(display (setq hor-odd-filt (make-separable-filter 1d-odd-filt 1d-low-pass
						   :edge-handler :reflect1))
	 'gray :zoom 8)
(display
 (setq hor-energy (add (square (apply-filter hor-even-filt einstein))
		       (square (apply-filter hor-odd-filt einstein))))
 'gray :pedestal 0 :scale 1e4 :zoom 1)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; 2D Subsampling:

;; Everything we've learned about 1D subsampling applies in 2D as
;; well.  We can perfectly reconstruct a subsampled image as long as
;; the image is appropriately (according to the Nyquist sampling
;; theorem) bandlimited.

;; Here's a simple example:

(setq gaussian (make-synthetic-image
		'(64 64)
		#'(lambda (y x) (exp (- (/ (+ (sqr (- x 32)) (sqr (- y 32)))
					   (sqr 12)))))
		:x-range '(0 63)))
(setq dft-gaussian (fft gaussian :center t))
(setq mag-gaussian (magnitude dft-gaussian))
(setq impulses (make-synthetic-image
		'(64 64)
		#'(lambda (y x) (if (and (zerop (mod y 4)) (zerop (mod x 4)))
				    1.0
				    0.0))
		:x-range '(0 63)))
(setq gauss-impulses (mul gaussian impulses))
(setq dft-gauss-impulses (fft gauss-impulses :center t))
(setq mag-gauss-impulses (magnitude dft-gauss-impulses))

;; We get replicas of the original Fourier transform when we multiply
;; by an impulse train.  We can reconstruct the original image by
;; pulling out the correct replica:

(setq box (make-synthetic-image
	   '(64 64)
	   #'(lambda (y x) (if (and (< 23 x 40) (< 23 y 40))
			       16.0
			       0.0))
	   :x-range '(0 63)))
(setq reconstructed-gauss (real-part (fft (mul box dft-gauss-impulses) :center t :inverse t)))
(mean-square-error reconstructed-gauss gaussian)

;; In Obvius, we can use the downsample function to do subsampling:

(setq down-gauss (downsample gaussian :start-vector '(0 0) :step-vector '(4 4)))

;; This returns a 16x16 image taking every fourth sample of the
;; original.  Obvius also has an upsample function that expands the
;; size of an image filling the in between pixels with zeros:

(setq down-then-up-gauss (upsample down-gauss :start-vector '(0 0) :step-vector '(4 4)))

;; This is the same as we got above, multiplying by the impulses:

(mean-square-error down-then-up-gauss gauss-impulses)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Filtering and Subsampling.

;; When we use a pre-filter to bandlimit an image, the filtered image
;; can be subsampled without aliasing.

;; First look at the dft of an image that is not band limited:
(setq al (crop einstein :x 60 :y 50 :x-dim 128 :y-dim 128))
(setq dft-al (fft al :center t))
(display (magnitude dft-al) 'gray :pedestal 0.0 :scale 10)

;; Now blur the image to produce a (more or less) band limited result:
(setq filt (make-separable-filter '(1/16 1/4 3/8 1/4 1/16)
				  '(1/16 1/4 3/8 1/4 1/16)))
(setq blur-al (apply-filter filt (apply-filter filt (apply-filter filt al))))
(setq dft-blur-al (fft blur-al :center t))
(display (magnitude dft-blur-al) 'gray  :pedestal 0.0 :scale 10)

;; Now multiply the blurred image by impulses:
(setq impulses (make-synthetic-image
		'(128 128)
		#'(lambda (y x) (if (and (zerop (mod y 2)) (zerop (mod x 2)))
				    1.0
				    0.0))
		:x-range '(0 127)))
(setq sampled-blur-al (mul blur-al impulses))
(setq dft-sampled-blur-al (fft sampled-blur-al :center t))
(display (magnitude dft-sampled-blur-al) 'gray  :pedestal 0.0 :scale 10)

;; Notice that the replicas in the freq domain don't overlap (at
;; least, the overlap is insignificant).  So we can reconstruct
;; blur-al from sampled-blur-al:

(setq box (make-synthetic-image
	   '(128 128)
	   #'(lambda (y x) (if (and (< 32 x 97) (< 33 y 97))
			       4.0
			       0.0))
	   :x-range '(0 127)))
(setq reconstructed-blur-al (real-part (fft (mul box dft-sampled-blur-al)
					    :center t :inverse t)))
(mean-square-error reconstructed-blur-al blur-al)

;; Now we downsample (equivalent to multiplying by the impulses and
;; then throwing away the zero values in between).

(setq sub-blur-al (downsample blur-al :step-vector '(2 2)))
(setq dft-sub-blur-al (fft sub-blur-al :center t))
(display (magnitude dft-sub-blur-al) 'gray  :pedestal 0.0 :scale 10)

;; And reconstruct:

(setq reconstructed-dft-blur-al
      (mul 2 (paste dft-sub-blur-al (make-complex-image '(128 128))
		    :y 32 :x 32)))
(setq new-reconstructed-blur-al (real-part (fft reconstructed-dft-blur-al
						:center t :inverse t)))
(mean-square-error reconstructed-blur-al blur-al)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Local Variables:
;;; buffer-read-only: t 
;;; fill-column: 79
;;; End:

