;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: x-mouse.lisp
;;;  Author: David Heeger
;;;  Description: X mouse events using LispView
;;;  Package: 'OBVIUS
;;;  Creation Date: summer, 1990
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)

#|
;;; new mouse bindings (as of 2/91)
\begin{tabular}{|l|ccc|}
        & left button       & middle button  & right  button       \\
\hline
Raw	& select-viewable   & refresh        & select-pane         \\
Control & previous-picture  & next-picture   & move-to-here        \\
C-Sh	& pop-picture       & center         & drag                \\
Shift 	& zoom-in           & zoom-out       & numerical-magnifier \\
M	& describe-viewable & set-parameters & picture-menu        \\
C-Sh-M	& destroy-viewable  & unbound        & hardcopy            \\

FLIPBOOKS:
C-M	& previous-frame  & next-frame  & display-sequence    \\

GRAY/PASTEUP/BITMAP:
C-M	& boost-contrast    & reduce-contrast & histogram      \\
Sh-M	& x-slice           & y-slice         & crop           \\
\hline
\end{tabular}
|#

#|
;;; Example: add temp mouse binding

(defclass test-left-mouse (documented-mouse-interest) ()
	  (:default-initargs
	      :event-spec '(() (:left :down))))

(defmethod lispview:receive-event ((pane X-pane) (interest test-left-mouse) event)
  (status-message "~a received a test~%" pane))

(push (make-instance 'test-left-mouse) (lispview:interests *current-pane*))
(remove-interest *current-pane* 'test-left-mouse)
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Useful function to remove interests of a particular type from pane
(defun remove-interest (pane interest-class)
  (setf (lispview:interests pane)
	(remove-if #'(lambda (i) (typep i interest-class))
		   (lispview:interests *current-pane*))))

;;; Execute body only if pane is unlocked.  Use this to keep immediate
;;; mouse events that may require pane locking from hanging.
(defmacro when-pane-unlocked (pane . body)
  `(if (locked ,pane)
     (status-message "Locked pane: mouse event ignored.")
     (progn ,@body)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; key bindings, and interest classes

;;; Raw
(defclass select-viewable-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Select Viewable"
	      :event-spec '(() (:left :down))))
(defclass refresh-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Refresh"	      
	      :event-spec '(() (:middle :down))))
(defvar *select-pane-interests*
  (let* ((begin-function #'(lambda (pane xs ys)
			     (set-selected-pane pane)
			     (when (picture-stack pane)
			       (let ((pic (car (picture-stack pane))))
				 (position-message pic (viewable pic) pane ys xs)))))
	 (drag-function #'(lambda (pane xs ys x0 y0 x1 y1)
			    (declare (ignore xs ys x0 y0))
			    (when (picture-stack pane)
			      (let ((pic (car (picture-stack pane))))
				(position-message pic (viewable pic) pane y1 x1)))))
	 (null-function #'(lambda (window xs ys x0 y0 x1 y1)
			    (declare (ignore window xs ys x0 y0 x1 y1))
			    nil)))
    (make-drag-interests begin-function drag-function null-function
			 :drag-cursor :XC-right-ptr
			 :doc-string "Select Pane"
			 :permanent-p t
			 :buttons :right
			 :modifiers '())))

;;; Control (picture/pane manipulation)
(defclass previous-picture-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Previous Picture"
	      :event-spec '((:control) (:left :down))))
(defclass next-picture-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Next Picture"
	      :event-spec '((:control) (:middle :down))))
(defclass move-to-here-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Move to Here"
	      :event-spec '((:control) (:right :down))))

;;; Shift (zooming)
(defclass zoom-in-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Zoom in"
	      :event-spec '((:shift) (:left :down))))
(defclass zoom-out-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Zoom Out"
	      :event-spec '((:shift) (:middle :down))))
;;; *** Add numerical magnifier '((:shift) (:right :down))

;;; Meta  (help)
(defclass describe-viewable-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Describe Viewable"
	      :event-spec '((:meta) (:left :down))))
(defclass set-parameter-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Set Picture Parameters"
	      :event-spec '((:meta ) (:middle :down))))
(defclass picture-menu-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Picture Menu"
	    :event-spec '((:meta ) (:right :down))))

;;; C-Sh (moving picture, popping)
(defclass pop-picture-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Pop Picture"
	      :event-spec '((:control :shift) (:left :down))))
(defclass center-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Center"
	      :event-spec '((:control :shift) (:middle :down))))
(defvar *drag-picture-interests*
  (let* ((null-function #'(lambda (&rest args) args))
	 (drag-function #'(lambda (pane xs ys x0 y0 x1 y1)
			    (if (and x0 y0)
				(obvius::drag-picture
				 (car (obvius::picture-stack pane))
				 (- y1 y0) (- x1 x0))
				(obvius::drag-picture
				 (car (obvius::picture-stack pane))
				 (- y1 ys) (- x1 xs))))))
    (make-drag-interests null-function drag-function null-function
			 :drag-cursor :XC-fleur
			 :doc-string "Drag picture"
			 :permanent-p t
			 :buttons :right
			 :modifiers '(:control :shift))))

;;; C-Sh-M
(defclass destroy-viewable-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Destroy Viewable"
	      :event-spec '((:control :meta :shift) (:left :down))))
(defclass hardcopy-interest (documented-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Hardcopy"
	      :event-spec '((:control :meta :shift) (:right :down))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; Picture-specific bindings:

(defclass picture-specific-mouse-interest (documented-mouse-interest) ())

;;; Default method returns nil.
(defmethod picture-specific-mouse-interests ((pic t))
  (declare (ignore pic))
  nil)

;;; C-M (picture-specific)
(defclass x-slice-interest (picture-specific-mouse-interest) ()
	  (:default-initargs
	      :doc-string "X Slice"
	      :event-spec '((:shift :meta) (:left :down))))
(defclass y-slice-interest (picture-specific-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Y Slice"
	      :event-spec '((:shift :meta) (:middle :down))))

;;; *** Should clip coords to size of image.  
(defvar *crop-interests*
  (let* ((op boole-eqv)
	 (left 0) (top 0)
	 (begin-function #'(lambda (window xs ys)
			     (let* ((pic (car (picture-stack window)))
				    (vbl (viewable pic)))
			       (multiple-value-bind (y x)
				   (pane-coord-to-viewable-coord pic ys xs)
				 (setq top (clip y 0 (y-dim vbl))
				       left (clip x 0 (x-dim vbl)))
				 (status-message "cropping... start: (~A ~A)" y x)))))
	 (drag-function #'(lambda (window xs ys x0 y0 x1 y1)
			    (let ((clr (lispview:find-color :pixel 0)))
			    (when (and x0 y0)
			      (let ((x-pos (min xs x0))
				    (y-pos (min ys y0))
				    (wd (abs (- xs x0)))
				    (ht (abs (- ys y0))))
				(lispview:draw-rectangle window x-pos y-pos wd ht
							 :operation op :foreground clr)))
			    (multiple-value-bind (y x)
				(pane-coord-to-viewable-coord
				 (car (picture-stack window)) y1 x1)
			      (status-message "cropping... start: (~A ~A). size: (~A ~A)"
					      top left (- y top) (- x left))
			      (lispview:draw-rectangle
			       window 
			       (min xs x1) (min ys y1) (abs (- xs x1)) (abs (- ys y1))
			       :operation op :foreground clr)))))
	 (end-function #'(lambda (window xs ys x0 y0 x1 y1)
			   (when (and x0 y0)
			     (let ((x-pos (min xs x0))
				   (y-pos (min ys y0))
				   (wd (abs (- xs x0)))
				   (ht (abs (- ys y0)))
				   (clr (lispview:find-color :pixel 0)))
			       (lispview:draw-rectangle window x-pos y-pos wd ht
							:operation op :foreground clr)))
			   (when (and (/= y1 ys) (/= x1 xs))
			     (let* ((pic (car (picture-stack window)))
				    (vbl (viewable pic)))
			       (multiple-value-bind (y x)
				   (pane-coord-to-viewable-coord pic y1 x1)
				 (push-onto-eval-queue
				  `(display (crop ,vbl
					     :y ,(min top y)
					     :x ,(min left x)
					     :y-dim ,(abs (- top y))
					     :x-dim ,(abs (- left x)))))))))))
    (make-drag-interests begin-function drag-function end-function
			 :permanent-p t
			 :buttons :right
			 :doc-string "Crop"
			 :drag-cursor :xc-plus
			 :modifiers '(:shift :meta))))

(defclass boost-contrast-interest (picture-specific-mouse-interest) ()
	  (:default-initargs
	    :doc-string "Boost Contrast"
	    :event-spec '((:control :meta) (:left :down))))
(defclass reduce-contrast-interest (picture-specific-mouse-interest) ()
	  (:default-initargs
	    :doc-string "Reduce Contrast"
	    :event-spec '((:control :meta) (:middle :down))))
(defclass histogram-interest (picture-specific-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Histogram"
	      :event-spec '((:control :meta) (:right :down))))

(defclass reverse-sequence-interest (picture-specific-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Previous Frame"
	      :event-spec '((:control :meta) (:left :down))))
(defclass step-sequence-interest (picture-specific-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Next Frame"
	      :event-spec '((:control :meta) (:middle :down))))
(defclass display-sequence-interest (picture-specific-mouse-interest) ()
	  (:default-initargs
	      :doc-string "Display Sequence"
	      :event-spec '((:control :meta) (:right :down))))

;;; any button click or release with any modifier keys will stop sequence display
(defclass stop-display-sequence-interest (documented-mouse-interest)
  ((flipbook :initform nil :accessor flipbook :initarg :flipbook))
  (:default-initargs
      :doc-string "Stop Sequence"
    :event-spec '(((:others (or :up :down)))
		  ((or :left :middle :right) (or :up :down)))))

;;; *** This is a crufty way to do this: it would be more efficient to
;;; leave (eg) a single meta-shift-left interest on the pane and have
;;; it call a method on the top picture.  But then it's hard to get
;;; mouse-documentation!
(defun set-picture-specific-mouse-bindings (pane)
  (remove-interest pane (find-class 'picture-specific-mouse-interest))
  (setf (lispview:interests pane)
	(append (picture-specific-mouse-interests (car (picture-stack pane)))
		(lispview:interests pane))))


(defvar *gray-mouse-interests*
  (list (make-instance 'x-slice-interest)
	(make-instance 'y-slice-interest)
	(make-instance 'boost-contrast-interest)
	(make-instance 'reduce-contrast-interest)
	(make-instance 'histogram-interest)))
(defvar *image-mouse-interests*
  (nconc *gray-mouse-interests* *crop-interests*))
(defvar *overlay-mouse-interests*
  (list (make-instance 'reverse-sequence-interest)
	(make-instance 'step-sequence-interest)))
(defvar *sequence-mouse-interests*
  (list (make-instance 'x-slice-interest)
	(make-instance 'y-slice-interest)
	(make-instance 'reverse-sequence-interest)
	(make-instance 'step-sequence-interest)
	(make-instance 'display-sequence-interest)
	))

(defmethod picture-specific-mouse-interests ((pic flipbook))
  *sequence-mouse-interests*)

(defmethod picture-specific-mouse-interests ((pic overlay))
  *overlay-mouse-interests*)

(defmethod picture-specific-mouse-interests ((pic gray))
  *image-mouse-interests*)

;;; pasteups can't do crop
(defmethod picture-specific-mouse-interests ((pic pasteup))
  *gray-mouse-interests*)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; This is used as a default-initarg for X-panes.
(defvar *default-pane-interests*
  (append
   (list (make-instance 'lispview::damage-interest)
	 (make-instance 'mouse-entry-interest)
	 (make-instance 'mouse-exit-interest)
	 (make-instance 'bucky-change-interest)
	 (make-instance 'select-viewable-interest)
	 (make-instance 'refresh-interest))
   *select-pane-interests*
   (list (make-instance 'previous-picture-interest)
	 (make-instance 'next-picture-interest)
	 (make-instance 'move-to-here-interest)
	 (make-instance 'zoom-in-interest)
	 (make-instance 'zoom-out-interest)
	 (make-instance 'pop-picture-interest)
	 (make-instance 'center-interest)
	 (make-instance 'end-drag-interest))
   *drag-picture-interests*
   (list (make-instance 'destroy-viewable-interest)
	 (make-instance 'describe-viewable-interest)
	 (make-instance 'set-parameter-interest)
	 (make-instance 'picture-menu-interest)
	 (make-instance 'hardcopy-interest))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Receive-event methods: these are called when the pane receives an
;;;; event matched by the interest.

#|
;;; default mouse event handler
(defmethod lispview:receive-event (pane interest event)
  (declare (ignore pane event))
  (warn "No receive-event method defined for ~A" interest))
|#

;;; select-viewable
(defmethod lispview:receive-event ((pane X-pane) (interest select-viewable-interest) event)
  (declare (ignore event))
  (insert-sexpr-for-evaluation (get-viewable pane)))

;;; refresh
(defmethod lispview:receive-event ((pane X-pane) (interest refresh-interest) event)
  (declare (ignore event))
  (let ((pic (car (picture-stack pane))))
    (if (and pic (not (current-p pic)))
	(push-onto-eval-queue `(refresh ,pane))	;do later if involves computing pic
	(when-pane-unlocked pane (refresh pane)))))

#|
;;; select-pane: old version without echo.
(defmethod lispview:receive-event ((pane X-pane) (interest select-pane-interest) event)
  (set-selected-pane pane)
  (when (picture-stack pane)
    (let* ((pic (car (picture-stack pane)))
	   (vbl (viewable pic)))
      (position-message pic vbl pane
			(lispview:mouse-event-y event)
			(lispview:mouse-event-x event)))))
|#

;;; previous-picture
(defmethod lispview:receive-event ((pane X-pane) (interest previous-picture-interest) event)
  (declare (ignore event))
  (when-pane-unlocked pane
     (set-selected-pane pane)
     (cycle-pane pane 1)))

;;; next-picture
(defmethod lispview:receive-event ((pane X-pane) (interest next-picture-interest) event)
  (declare (ignore event))
  (when-pane-unlocked pane
     (set-selected-pane pane)
     (cycle-pane pane -1)))

;;; move-to-here
(defmethod lispview:receive-event ((pane X-pane) (interest move-to-here-interest) event)
  (declare (ignore event))
  (when-pane-unlocked pane
     (when (and (picture-stack *current-pane*) (not (eq pane *current-pane*)))
       (move-picture (car (picture-stack *current-pane*)) pane))
     (set-selected-pane pane)))

;;; zoom-in
(defmethod lispview:receive-event ((pane X-pane) (interest zoom-in-interest) event)
  (when-pane-unlocked pane
     (zoom-picture (car (picture-stack pane)) 2
		   (lispview:mouse-event-y event) (lispview:mouse-event-x event))))

;;; zoom-out
(defmethod lispview:receive-event ((pane X-pane) (interest zoom-out-interest) event)
  (when-pane-unlocked pane
     (zoom-picture (car (picture-stack pane)) 1/2
	   	   (lispview:mouse-event-y event) (lispview:mouse-event-x event))))


;;; pop-picture
(defmethod lispview:receive-event ((pane X-pane) (interest pop-picture-interest) event)
  (declare (ignore event))
  (when-pane-unlocked pane
     (pop-picture pane)))

;;; center
(defmethod lispview:receive-event ((pane X-pane) (interest center-interest) event)
  (declare (ignore event))
  (when-pane-unlocked pane
     (set-selected-pane pane)
     (when (picture-stack pane)
       (drag-picture (car (picture-stack pane)) nil nil))))

;;; destroy-viewable
(defmethod lispview:receive-event ((pane X-pane) (interest destroy-viewable-interest) event)
  (declare (ignore event))
  (when (picture-stack pane)
    (let ((vbl (viewable (car (picture-stack pane)))))
      (push-onto-eval-queue `(destroy ,vbl))
      (when-pane-unlocked pane (pop-picture pane)))))

;;; hardcopy
(defmethod lispview:receive-event ((pane X-pane) (interest hardcopy-interest) event)
  (declare (ignore event))
  (set-selected-pane pane)
  (let ((pic (car (picture-stack pane))))
    (push-onto-eval-queue
     `(hardcopy (viewable ,pic) ,pic)))) ;pass pic itself

;;; describe-viewable
(defmethod lispview:receive-event ((pane X-pane) (interest describe-viewable-interest) event)
  (declare (ignore event))
  ;(set-selected-pane pane)
  (let ((pic (car (picture-stack pane))))
    (push-onto-eval-queue
     `(when ,pic 
       (describe (viewable ,pic))
       (print-top-level-values nil)))))

#| 
;;; FOr describing viewalbe, want to do something like this, but
;;; notice-prompt centers the lines!
(let ((stream (make-string-output-stream)))
  (describe (viewable pic) stream)
  (lispview:notice-prompt :message (get-output-stream-string stream)
			  :choices '((:yes "OK" t))))
|#

;;; set-parameter
(defmethod lispview:receive-event ((pane X-pane) (interest set-parameter-interest) event)
  (declare (ignore event))
  ;(set-selected-pane pane)
  (let ((pic (car (picture-stack pane))))
    (push-onto-eval-queue
     `(when ,pic
       (make-slot-value-dialog
	,pic :slot-names (settable-parameters ,pic)
	:label "obvius slot dialog"
	:update-function 'picture-slot-update-function)))))

;;; Picture menu
(defmethod lispview:receive-event ((pane X-pane) (interest picture-menu-interest) event)
  (when (> (list-length (picture-stack pane)) 1)
    (lv:menu-show
     (make-instance 'lv:menu :label nil
		    :choices #'(lambda () (make-picture-menu-items pane)))
     pane :x (lv:mouse-event-x event) :y (lv:mouse-event-y event))))

(defun make-picture-menu-items (pane)
  (loop for pic in (picture-stack pane)
	for num from 0
	collect
	(make-instance 'lv:command-menu-item
		       :label (format nil "~A" pic)
		       :command `(lambda () (cycle-pane ,pane ,num)))))

;;;; ---------------------------------------------------------------------------
;;;; GRAY mouse bindings

;;; x-slice
(defmethod lispview:receive-event ((pane X-pane) (interest x-slice-interest) event)
  (when (picture-stack pane)
    (let* ((pic (car (picture-stack pane)))
	   (vbl (viewable pic)))
      (multiple-value-bind (y x)
	  (pane-coord-to-viewable-coord pic (lispview:mouse-event-y event)
					    (lispview:mouse-event-x event))
	(declare (ignore x))
	(push-onto-eval-queue
	 `(display (make-slice ,vbl :y ,y) t :pane ,pane))))))

;;; y-slice
(defmethod lispview:receive-event ((pane X-pane) (interest y-slice-interest) event)
  (when (picture-stack pane)
    (let* ((pic (car (picture-stack pane)))
	   (vbl (viewable pic)))
      (multiple-value-bind (y x)
	  (pane-coord-to-viewable-coord pic (lispview:mouse-event-y event)
					    (lispview:mouse-event-x event))
	(declare (ignore y))
	(push-onto-eval-queue
	 `(display (make-slice ,vbl :x ,x) t :pane ,pane))))))

;;; Change contrast, keeping pixel value which maps to 0.5 (middle gray) fixed.
(defmethod lispview:receive-event ((pane X-pane) (interest boost-contrast-interest) event)
  (declare (ignore event))
  (when-pane-unlocked pane
     (let* ((pic (car (picture-stack pane)))
	    (new-scale (/ (slot-value pic 'scale) 1.5))) ;factor of 1.5
       (reinitialize-instance pic
			      :scale new-scale
			      :pedestal (+ (slot-value pic 'pedestal)
					   (/ (- (slot-value pic 'scale) new-scale) 2)))
       (draw-pane pane :clear nil))))

(defmethod lispview:receive-event ((pane X-pane) (interest reduce-contrast-interest) event)
  (declare (ignore event))
  (when-pane-unlocked pane
     (let* ((pic (car (picture-stack pane)))
	    (new-scale (* (slot-value pic 'scale) 1.5)))
       (reinitialize-instance pic
			      :scale new-scale
			      :pedestal (+ (slot-value pic 'pedestal)
					   (/ (- (slot-value pic 'scale) new-scale) 2)))
       (draw-pane pane :clear nil))))

(defmethod lispview:receive-event ((pane X-pane) (interest histogram-interest) event)
  (declare (ignore event))
  (when (picture-stack pane)
    (let ((vbl (viewable (car (picture-stack pane)))))
      (push-onto-eval-queue `(display (make-histogram ,vbl) t :pane ,pane)))))

;;;; ---------------------------------------------------------------------------
;;;; FLIPBOOK mouse bindings

;;; reverse-sequence
(defmethod lispview:receive-event ((pane X-pane) (interest reverse-sequence-interest) event)
  (declare (ignore event))
  (when-pane-unlocked pane
     (when (or (typep (car (picture-stack pane)) 'flipbook)
	       (typep (car (picture-stack pane)) 'overlay))
       (single-step (car (picture-stack pane)) -1))))

;;; step-sequence
(defmethod lispview:receive-event ((pane X-pane) (interest step-sequence-interest) event)
  (declare (ignore event))
  (when-pane-unlocked pane
     (when (or (typep (car (picture-stack pane)) 'flipbook)
	       (typep (car (picture-stack pane)) 'overlay))
       (single-step (car (picture-stack pane))))))

;;; display-sequence *** Would be nice to do this with a process (see
;;; commented code), but this may cause the mouse process to hang
;;; since display-seq locks the pane.  This is a serious problem, since then the
;;; user has no way to stop the sequence display!
(defmethod lispview:receive-event ((pane X-pane) (interest display-sequence-interest) event)
  (declare (ignore event))
  (let ((pic (car (picture-stack pane))))
    (when (typep pic 'flipbook)
      ;;(make-process :name "Display Seuence"
      ;;              :function #'display-seq :args (list pic))
      (push-onto-eval-queue `(display-seq ,pic))
      (push (make-instance 'stop-display-sequence-interest :flipbook pic)
	    (lispview:interests pane))
      )))

(defmethod lispview:receive-event
    ((pane X-pane) (interest stop-display-sequence-interest) event)
  (declare (ignore event))
  (let ((pic (slot-value interest 'flipbook)))
    (setf (slot-value pic 'displaying-p) nil)
    (setf (lispview:interests pane)
	  (remove interest (lispview:interests pane)))))

;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
