;;; -*- Package: OBVIUS; Syntax: Common-lisp -*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: steer-pyramid.lisp
;;;  Author: David Heeger and Bill Freeman
;;;  Description: pyramid-steerable-filter image-transforms.
;;;            Program which came up with the filter taps:  
;;;                      ~freeman/lisp/steer/pyramid-alias.lisp
;;;  Creation Date: summer '89
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; History:
;;; NOTE:  Sept. 22, 1989  the non-separable pyramid tiling steering filters
;;;  steer backwards from the way they should.  ie, for them, positive angle = clockwise.
;;;  For all the other filters, including the separable versions of the 
;;;  pyramid tiling steering filters, positive angle = counterclockwise.  Eventually,
;;;  I'll fix the non-separable ones.  wtf.   See wtf labbook p. 52.
;;; April 19, 1990  freeman changed edge handling from  nil (wrap around) to "reflect1".
;;; June 20, 1990  Freeman modified to make an end-filter to stop the recursion.
;;;  So, I added an  end-filter  to the definition of steerable-pyramid,
;;;  and changed even-collapse, odd-collapse, and added *default-end-steerable-pyramid-filter*.
;;;  So the old default filters, stored as   bpeven0 ...  are still there on disk, but the
;;;  new filters are read in from   bpevena0 ...    
;;; july 3, 1991 modified this to read-in the newest version filters,  bpevenb0...
;;;  See /u/freeman/lisp/steer/pyramid-alias.lisp for general comments about the steerable
;;;  pyramid;  search for "july 3".  The x-y separable filters in this file are out of date.
;;; July 9, 1991  changed the edge-handler from reflect1 back to nil.  That significantly
;;;  reduced the reconstruction errors;  see ~/lisp/steer/pyramid-alias.lisp.
;;;  OLDEST FILTERS, designed summer, 1989:  bpeven0, bpeven1, ...
;;;  NEWER FILTERS, designed summer, 1990:  bpevena0, bpevena1, ...
;;;  NEWEST FILTERS, designed summer, 1991:  bpevenb0, bpevenb1, ...
;;;  Still newest filters, oriented filters:  July 1991:  bpevenc0, ... (oriented filters only.
;;;   the two low-pass filters are the same as version b).
;;; July 20, 1991.  Use even filters which have NO dc response, even though
;;; they give slightly higher maximum freq error.  Therefore, use bpevenb0,1,2,3,4  with
;;; bpoddc0,1,2,3.  


(in-package 'obvius)
(export '(steerable-pyramid make-steerable-pyramid 
	  low-filter end-filter even-filters odd-filters low-images steerables
	  access-low access-qsb build even-collapse steer-even
	  odd-collapse steer-odd
	  steer-odd steer sum-even sum-odd magnitude square-magnitude
	  dimensions
	  complex-phase *default-odd-steerable-pyramid-filters*
	  *default-even-steerable-pyramid-filters*
	  *default-low-steerable-pyramid-filter*
	  *default-end-steerable-pyramid-filter*
	  *odd-separable-steerable-pyramid-filters*
	  *even-separable-steerable-pyramid-filters*))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; STEERABLE PYRAMID CLASS

;;; low-filter is the low-pass filter
;;; end-filter is the recursion ending filter.
;;; even-filters is a list of even steerable filters
;;; odd-filters is a list of odd steerable filters
;;; low-images is a list of low-pass images (the zeroth one on the list is the original)
;;; steerables is a list of quadrature-steerable-basis objects
(DEFCLASS steerable-pyramid (viewable)
  (low-filter 
   end-filter
   even-filters 
   odd-filters
   low-images
   steerables)
  (:accessor-prefix ))

;;; note: no display type for these guys for now, look at the sub-objects to look at them.
;;; if :level is non-nil, initially builds pyramid to that level
(defun make-steerable-pyramid
    (image &key 
	   (level nil)
	   (low-filter *default-low-steerable-pyramid-filter*)
	   (end-filter *default-end-steerable-pyramid-filter*)
	   (even-filters *default-even-steerable-pyramid-filters*)
	   (odd-filters *default-odd-steerable-pyramid-filters*)
	   (display-type nil)
	   ((:-> name)))
  (when (viewable-p name) (error "Can not pass exisiting viewable to make-<vbl> functions"))
  (let* ((pyramid (make-instance 'steerable-pyramid
				 :display-type display-type
				 :low-filter low-filter
				 :end-filter end-filter
				 :even-filters even-filters
				 :odd-filters odd-filters
				 :low-images (list image)
				 :steerables nil)))
    (push pyramid (superiors-of image))
    (set-history pyramid 'make-steerable-pyramid image 
		 :low-filter low-filter
		 :end-filter end-filter
		 :even-filters even-filters 
		 :odd-filters odd-filters)
    (set-name pyramid name)
    (if level (build pyramid level))
    pyramid))

;;; if you try to destroy any sub-objeject,
;;; continuable error allows you to destroy the whole pyramid.
(defmethod notify-of-inferior-destruction ((pyramid steerable-pyramid) thing)
  (cerror "Destroy  both ~A and ~A."
	  "You are attempting to destroy ~A which is contained in ~A."
	  thing pyramid)
  (destroy pyramid))

(defmethod inferiors-of ((pyr steerable-pyramid))
  (append (low-images pyr) (remove nil (steerables pyr))))

(defmethod access-low ((pyr steerable-pyramid) level
		       &key ((:-> res)))
  (when (< (1- (length (low-images pyr))) level)
    (loop for i from (length (low-images pyr)) to level 
	  for new-low-image = (apply-filter (low-filter pyr) (car (last (low-images pyr))))
	  do
	  (setf (low-images pyr) (append (low-images pyr) (list new-low-image)))
	  (push pyr (superiors-of new-low-image))))
  (let ((image (nth level (low-images pyr))))
    (cond ((null res) image)
	  ((image-p res) (copy image :-> res))
	  (t (set-name image res) image))))

(defmethod access-qsb ((pyr steerable-pyramid) level &key ((:-> res)))
  (when (< (1- (length (steerables pyr))) level)
    (setf (steerables pyr) 
	  (append (steerables pyr)
		  (list-of-length (- level (1- (length (steerables pyr)))) nil))))
  (when (null (nth level (steerables pyr)))
	(let* ((low-image (access-low pyr level))
	       (new-qsb (make-quadrature-steerable-basis low-image
							 :even-filters (even-filters pyr)
							 :odd-filters (odd-filters pyr))))
	  (push pyr (superiors-of new-qsb))
	  (setf (nth level (steerables pyr)) new-qsb)))
  (let ((qsb (nth level (steerables pyr))))
    (cond ((null res) qsb)
	  (t (set-name qsb res) qsb))))

(defmethod build ((pyr steerable-pyramid) level)
  (access-low pyr level)
  (loop for i from 0 to level do
	(access-qsb pyr i)))

(defmethod even-collapse ((pyr steerable-pyramid) level 
			  &key ->)
  (with-result-image ((result ->) (dimensions (car (low-images pyr)))
		      'collapse pyr level)
    (with-local-images ((new-low (apply-filter (end-filter pyr) (access-low pyr (1+ level))))
			(next-image (similar (access-low pyr level))))
      (even-recursive-collapse-sp pyr level new-low next-image result))))
    
;;; result is twice bigger than low-image
;;; low-image is expanded in to result
;;; band-pass images from level 'level' are filtered and added into result
;;; this result becomes the low-image at the next level
(defun even-recursive-collapse-sp (pyr level low-image next-image result)
  (expand-filter (mul (low-filter pyr) 4.0) low-image :zero nil :-> next-image)
  (loop for i from 0 below (length (even-filters pyr)) do
	(expand-filter (nth i (even-filters pyr))
		       (nth i (image-list (even-steerable-basis 
					   (nth level (steerables pyr)))))
		       :zero nil
		       :-> next-image))
  (if (zerop level) 
      (copy next-image :-> result)
      (with-local-images ((new-next-image (similar (nth (1- level) (low-images pyr)))))
	(even-recursive-collapse-sp pyr (1- level) next-image new-next-image result))))

;;; exactly like code for even-collapse, except the word "even" replaced
;;; by "odd".
(defmethod odd-collapse ((pyr steerable-pyramid) level 
			  &key ->)
  (with-result-image ((result ->) (dimensions (car (low-images pyr)))
		      'collapse pyr level)
    (with-local-images ((new-low (apply-filter (end-filter pyr) (access-low pyr (1+ level))))
			(next-image (similar (access-low pyr level))))
      (odd-recursive-collapse-sp pyr level new-low next-image result))))

    
;;; result is twice bigger than low-image
;;; low-image is expanded in to result
;;; band-pass images from level 'level' are filtered and added into result
;;; this result becomes the low-image at the next level
(defun odd-recursive-collapse-sp (pyr level low-image next-image result)
  (expand-filter (mul (low-filter pyr) 4.0) low-image :zero nil :-> next-image)
  (loop for i from 0 below (length (odd-filters pyr)) do
	(expand-filter (nth i (odd-filters pyr))
		       (nth i (image-list (odd-steerable-basis 
					   (nth level (steerables pyr)))))
		       :zero nil
		       :-> next-image))
  (if (zerop level) 
      (copy next-image :-> result)
      (with-local-images ((new-next-image (similar (nth (1- level) (low-images pyr)))))
	(odd-recursive-collapse-sp pyr (1- level) next-image new-next-image result))))


(defmethod steer-even ((pyr steerable-pyramid) level angle &key ((:-> res)))
  (steer-even (access-qsb pyr level) angle :-> res))

(defmethod steer-odd ((pyr steerable-pyramid) level angle &key ((:-> res)))
  (steer-odd (access-qsb pyr level) angle :-> res))

(defmethod steer ((pyr steerable-pyramid) level angle &key ((:-> res)))
  (steer (access-qsb pyr level) angle :-> res))

(defmethod sum-even ((pyr steerable-pyramid) level &key ((:-> res)))
  (sum-even (access-qsb pyr level) :-> res))

(defmethod sum-odd ((pyr steerable-pyramid) level &key ((:-> res)))
  (sum-odd (access-qsb pyr level) :-> res))

(defmethod magnitude ((pyr steerable-pyramid) level angle &key ((:-> res)))
  (magnitude (access-qsb pyr level) angle :-> res))

(defmethod square-magnitude ((pyr steerable-pyramid) level angle &key ((:-> res)))
  (square-magnitude (access-qsb pyr level) angle :-> res))

(defmethod complex-phase ((pyr steerable-pyramid) level angle &key ((:-> res)))
  (complex-phase (access-qsb pyr level) angle :-> res))

(defmethod dimensions ((pyr steerable-pyramid))
  (dimensions (access-low pyr 0)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; DEFAULT STEERABLE PYRAMID FILTERS
;;; Designed using ~freeman/work/pyramid.lisp
;;; Steerable, tileable.  odd order steers like 3rd order polynomial;
;;; even order fit to its hilbert transform steers like 4th order polynomial

;;; 15-tap  Hilbert transforms of default-odd-steerable-pyramid-filters
;;;  frequency domain design.  There are 5 of these filters to the basis set.

(defun make-default-even-steerable-pyramid-filters ()
  (list (make-filter (obvius::data (load-image "~freeman/lisp/steer/bpevenb0"))
		     :edge-handler nil)
	(make-filter (obvius::data (load-image "~freeman/lisp/steer/bpevenb1"))
		     :edge-handler nil)
	(make-filter (obvius::data (load-image "~freeman/lisp/steer/bpevenb2"))
		     :edge-handler nil)
	(make-filter (obvius::data (load-image "~freeman/lisp/steer/bpevenb3"))
		     :edge-handler nil)
	(make-filter (obvius::data (load-image "~freeman/lisp/steer/bpevenb4"))
		     :edge-handler nil)))


;;; 15-tap. steers like 3rd order polynomials
;;; 4 filters per steerable basis set.
(defun make-default-odd-steerable-pyramid-filters ()
  (list 
   (make-filter (obvius::data (load-image "~freeman/lisp/steer/bpoddc0"))
		:edge-handler nil)
   (make-filter (obvius::data (load-image "~freeman/lisp/steer/bpoddc1"))
		:edge-handler nil)
   (make-filter (obvius::data (load-image "~freeman/lisp/steer/bpoddc2"))
		:edge-handler nil)
   (make-filter (obvius::data (load-image "~freeman/lisp/steer/bpoddc3"))
		:edge-handler nil)))

;;; 7-tap
(defun make-default-low-steerable-pyramid-filter ()
  (make-filter (obvius::data (load-image "~freeman/lisp/steer/lpfb")) 
	       :step-vector '(2 2)
	       :edge-handler nil))

;;; 13-tap recursion ending low-pass filter
(defun make-default-end-steerable-pyramid-filter ()
  (make-filter (obvius::data (load-image "~freeman/lisp/steer/lprimeb")) 
	       :edge-handler nil))


;;; 3rd order polynomial steering
(setq *default-odd-steerable-pyramid-filters* 
  (make-default-odd-steerable-pyramid-filters))

;;; 4th order polynomial fit to Hilbert transform of the above
(setq *default-even-steerable-pyramid-filters* 
  (make-default-even-steerable-pyramid-filters))

;;; low-pass filter
(setq *default-low-steerable-pyramid-filter* 
      (make-default-low-steerable-pyramid-filter))

;;; end low-pass filter
(setq *default-end-steerable-pyramid-filter* 
      (make-default-end-steerable-pyramid-filter))


;;; june 20, 1990 so these separable filters are now out of date.

;;; These values were derived from from ~freeman/lisp/steer/pyramid.lisp
;;; I cropped these to 15x1 so that they could operate on 16x16 images and so
;;; it would be a little faster than the prior 21x1 versions.
(defun make-odd-separable-steerable-pyramid-filters ()  
  (list 
   (make-separable-filter 
    (make-filter '(
		   -0.0012241235  -0.0023764507  -0.003607996  -0.0031322648  0.0067423284  
		   0.0737513  0.24757873  0.36081195  0.24757871  0.0737513  
		   0.0067423284  -0.0031322634  -0.0036079946  -0.002376452  -0.0012241263  ))
    (make-filter '(
		   -0.0059641814  -0.0014696772  0.02129536  0.028223535  0.09698914  
		   0.021359185  -0.36081195  -1.2511874E-9  0.36081195  -0.021359181  
		   -0.09698914  -0.028223544  -0.021295357  0.0014696819  0.005964178 ))
    :edge-handler "reflect1")
   (make-separable-filter 
    (make-filter '(
		   -0.004833335  -0.0075006904  -0.009438899  -0.003410729  0.027244598  
		   0.15351956  0.23880503  1.6398182E-9  -0.23880504  -0.15351956  
		   -0.0272446  0.0034107282  0.009438901  0.0075006895  0.004833334  ))
    (make-filter '(
		   -0.0030795108  -0.0027119636  5.4833235E-4  0.009854368  0.042126063  
		   0.11699543  -0.041454937  -0.23880504  -0.04145493  0.11699543  
		   0.042126063  0.009854364  5.483329E-4  -0.0027119622  -0.0030795103  ))
    :edge-handler "reflect1")
   (make-separable-filter 
    (make-filter '(
		   0.0030795098  0.0027119669  -5.483289E-4  -0.009854366  -0.042126063  
		   -0.11699543  0.041454926  0.23880506  0.041454934  -0.11699543  
		   -0.042126067  -0.009854366  -5.4832944E-4  0.002711969  0.0030795105  ))
    (make-filter '(
		   0.0048333313  0.007500688  0.0094388975  0.003410728  -0.027244594  
		   -0.15351956  -0.23880506  2.4400197E-9  0.23880506  0.15351956  
		   0.02724459  -0.0034107252  -0.0094388975  -0.007500691  -0.004833332 ))
    :edge-handler "reflect1")
   (make-separable-filter 
    (make-filter '(
		   0.0059641832  0.001469676  -0.02129536  -0.028223535  -0.09698914  
		   -0.021359183  0.36081195  1.0184825E-9  -0.36081195  0.021359183  
		   0.09698914  0.028223537  0.021295363  -0.0014696786  -0.0059641865  ))
    (make-filter '(
		   0.0012241219  0.0023764481  0.0036079912  0.003132259  -0.0067423307  
		   -0.0737513  -0.24757871  -0.36081195  -0.24757873  -0.0737513  
		   -0.0067423293  0.003132259  0.0036079902  0.0023764474  0.0012241211))
    :edge-handler "reflect1")))


;;; 3rd order separable approx to the bandpass filters
(setq *odd-separable-steerable-pyramid-filters*   
      (make-odd-separable-steerable-pyramid-filters))

;;;
;;; 4th order polynomials
;;; from ~freeman/lisp/steer/pyramid.lisp
(defun make-even-separable-steerable-pyramid-filters ()  
  (list 
   (make-separable-filter 
    (make-filter '(
		   -0.0022221229  -0.0040625622  -0.00626532  -0.0069051078  0.0028975303  
		   0.0787226  0.2657738  0.38673005  0.2657738  0.0787226  
		   0.002897532  -0.006905106  -0.0062653166  -0.0040625613  -0.0022221226  ))
    (make-filter '(
		   0.0086881695  0.015859136  0.019083789  0.0071989205  -0.008788917  
		   -0.21675833  -0.024348792  0.38673005  -0.024348794  -0.21675833  
		   -0.0087889135  0.007198924  0.019083783  0.015859134  0.008688175  ))
    :edge-handler "reflect1")
   (make-separable-filter 
    (make-filter '(
		   -0.0024234327  -0.002265249  -0.00293626  0.004872925  0.026676085  
		   0.12256687  0.19029094  4.8942033E-9  -0.19029094  -0.12256687  
		   -0.026676085  -0.0048729246  0.002936262  0.0022652468  0.0024234296  ))
    (make-filter '(
		   0.0017225681  0.0022979171  0.009654019  0.010583142  0.027341839  
		   -0.05383734  -0.19029094  -0.0  0.19029094  0.053837363  
		   -0.027341839  -0.010583142  -0.009654015  -0.002297919  -0.001722569 ))
    :edge-handler "reflect1")
   (make-separable-filter 
    (make-filter '(
		   0.0033666322  0.0028232834  -0.0012785012  -0.012164454  -0.037755236  
		   -0.074575834  0.06459016  0.22295803  0.06459016  -0.074575834  
		   -0.037755236  -0.012164451  -0.0012785009  0.0028232825  0.0033666312  ))
    (make-filter '(
		   0.0033666296  0.0028232802  -0.0012785028  -0.01216445  -0.037755236  
		   -0.07457586  0.064590186  0.22295803  0.064590186  -0.07457586  
		   -0.037755236  -0.012164451  -0.0012784986  0.0028232802  0.0033666291  ))
    :edge-handler "reflect1")
   (make-separable-filter 
    (make-filter '(
		   -0.0017225695  -0.002297915  -0.009654021  -0.010583136  -0.027341804  
		   0.053837534  0.19029127  0.0  -0.19029127  -0.053837534  
		   0.027341804  0.010583139  0.009654023  0.0022979127  0.0017225663  ))
    (make-filter '(
		   0.0024234313  0.002265251  0.0029362738  -0.0048729084  -0.026676044  
		   -0.122566976  -0.19029127  2.4470974E-9  0.19029127  0.122566976  
		   0.026676042  0.004872907  -0.0029362752  -0.0022652505  -0.0024234294 ))
    :edge-handler "reflect1")
   (make-separable-filter 
    (make-filter '(
		   0.008688172  0.01585915  0.019083802  0.0071989303  -0.008788907  
		   -0.21675843  -0.024348918  0.38673007  -0.024348924  -0.21675843  
		   -0.008788907  0.007198926  0.019083802  0.015859153  0.008688173  ))
    (make-filter '(
		   -0.0022221217  -0.0040625692  -0.0062653297  -0.00690511  0.0028975462  
		   0.07872275  0.2657739  0.38673007  0.2657739  0.078722745  
		   0.0028975483  -0.0069051078  -0.006265329  -0.0040625692  -0.0022221217))
    :edge-handler "reflect1")))


;;; 4th order separable approx to the bandpass filters
(setq *even-separable-steerable-pyramid-filters*   
      (make-even-separable-steerable-pyramid-filters))



#|

;;; 4rd order separable approx to the bandpass filters.
(setq *even-separable-steerable-pyramid-filters* 
  (make-even-separable-steerable-pyramid-filters))


(setq imp64 (make-impulse '(64 64)))
(setq imp128 (make-impulse '(128 128)))
(setq tst-filter (car *default-odd-steerable-pyramid-filters*))
(setq lo-filter *default-low-steerable-pyramid-filter*)
(setq bnd1 (expand-filter lo-filter (expand-filter tst-filter imp64)))
(setq bnd0 (expand-filter tst-filter imp128))
(setq resp0 (circular-shift (square-magnitude (fft bnd0)) 64 64))
(setq resp1 (circular-shift (square-magnitude (fft bnd1)) 64 64))
(setq checktile (add resp0 resp1))

(setq rand (make-uniform-random-image '(64 64)))
(setq zone (make-cos-zone-plate '(128 128)))

(setq pyr (make-steerable-pyramid zone :level 2))
(setq sm0-pi/4 (square-magnitude pyr 0 (* pi (/ 1.0 4.0))))
(setq sm0-pi/2 (square-magnitude pyr 0 (* pi (/ 1.0 2.0))))
(setq sm0-3pi/4 (square-magnitude pyr 0 (* pi (/ 3.0 4.0))))
(setq sm0-pi (square-magnitude pyr 0 (* pi (/ 1.0 1.0))))
(setq sm0 (+. (+. sm0-pi/4 sm0-pi/2) (+. sm0-3pi/4 sm0-pi)))
(setq sm1-pi/4 (square-magnitude pyr 1 (* pi (/ 1.0 4.0))))
(setq sm1-pi/2 (square-magnitude pyr 1 (* pi (/ 1.0 2.0))))
(setq sm1-3pi/4 (square-magnitude pyr 1 (* pi (/ 3.0 4.0))))
(setq sm1-pi (square-magnitude pyr 1 (* pi (/ 1.0 1.0))))
(setq sm1 (+. (+. sm1-pi/4 sm1-pi/2) (+. sm1-3pi/4 sm1-pi)))
(setq sm1-exp (expand-filter sm1 (mul (low-filter pyr) 4.0)))
(setq sm2-pi/4 (square-magnitude pyr 2 (* pi (/ 1.0 4.0))))
(setq sm2-pi/2 (square-magnitude pyr 2 (* pi (/ 1.0 2.0))))
(setq sm2-3pi/4 (square-magnitude pyr 2 (* pi (/ 3.0 4.0))))
(setq sm2-pi (square-magnitude pyr 2 (* pi (/ 1.0 1.0))))
(setq sm2 (+. (+. sm2-pi/4 sm2-pi/2) (+. sm2-3pi/4 sm2-pi)))

(setq pyr1 (make-steerable-pyramid 
	    zone
	    :low-filter (make-separable-filter 
			 '(0.125 0.5 0.75 0.5 0.125)
			 '(0.125 0.5 0.75 0.5 0.125)
			 :step-vector '(2 2)
			 :edge-handler "reflect1")
	    :even-filters *default-even-steerable-filters*
	    :odd-filters *default-odd-steerable-filters*))
|#
  
