;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: pyramid.lisp
;;;  Author: David Heeger
;;;  Description: general pyramids and filter pyramids
;;;  Creation Date: summer '88
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '(filter-pyramid qmf-pyramid one-d-qmf-pyramid
	  separable-qmf-pyramid hex-qmf-pyramid
	  pyramid-p filter-pyramid-p
	  forward-filters inverse-filters
	  build collapse
	  make-one-d-qmf-pyramid
	  make-separable-qmf-pyramid access-diag access-hor access-vert
	  make-hex-qmf-pyramid access-rdiag access-ldiag
	  greater-than copy
	  add mul sub div square square-root abs-value
	  apply-filter blur gauss-in gauss-out))


(error "this code has not yet been ported to OBVIUS version2.1")


;;;--------------------- Pyramid Classes -------------------------

;;; general image pyramids contain only an image tree

(def-simple-class pyramid (viewable)
  ((image-tree :initform nil))
  (:default-initargs :display-type 'pasteup))

(defmethod inferiors-of ((pyr pyramid))
  (collapse-tree (image-tree pyr)))

;;; Filter-pyramids --- self-similar sub-band image transforms including
;;; Laplacian Pyramids and QMF Pyramids.
;;; Filter-pyramid class contains forward-filters and inverse-filters, each of 
;;; which is a list of filters or nil.  The length of the forward-filters 
;;; list is the branching factor of the image tree

(def-simple-class filter-pyramid (pyramid)
  ((forward-filters :initform nil)
   (inverse-filters :initform nil)))

(defmacro pyramid-p (obj)
  `(typep ,obj 'pyramid))

(defmacro filter-pyramid-p (obj)
  `(typep ,obj 'filter-pyramid))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; UTILITIES

;;; The routine SET-RESULT is used by all functions which return 
;;; pyramids as results.  It is slightly different from SET-RESULT for images
;;;  1) If the result parameter is nil, create a new, blank pyramid of the
;;;     same type.
;;;  2) If the result parameter is a pyramid, check the tree structure.
;;;  3) If the result is a string or symbol, create a new pyramid with this name.
;;;  4) If the result is a symbol, create blank pyramid with this name, and 
;;;     bind the symbol to it, warning the user if the symbol is already bound.
;;;  The parameter pyr must be a pyramid.

(defmethod set-result ((model pyramid) (res pyramid))
  (check-size model res)
  res)

(defmethod set-result ((model pyramid) (name t))
  (check-type name viewable-name)
  (let* ((result (make-instance 
		  (class-of model)
		  :display-type (display-type model)
		  :image-tree (mapcar-tree #'similar (image-tree model)))))
    (mapcar-tree #'(lambda (im) (pushnew result (superiors-of im)))
		 (image-tree result))
    (when name (set-name result name))
    result))

(defmethod set-result ((model filter-pyramid) (name t))
  (let ((result (call-next-method)))
    (setf (forward-filters result) (forward-filters model))
    (setf (inverse-filters result) (inverse-filters model)))
  result)

(defmethod set-result ((model-plist list) (res pyramid))
  (apply 'reinitialize-instance res model-plist)
  res)

(defmethod complete-result ((res pyramid) plist)
  (declare (ignore plist))
  (mapcar-tree #'(lambda (im) (pushnew res (superiors-of im)))
	       (image-tree res))
  res)

;;; **** version 1.2 -> 2.0 transition function:
(defmacro with-result-pyramid (&rest stuff)
  (declare (ignore stuff))
  (error "This macro is not provided in v2.0.  Use with-result instead!"))

;;; Check that the tree structures are the same -- leave the checking of the
;;; individual image dimensions to the functions which operate on the
;;; images themselves.
(defmethod check-size ((pyr pyramid) &rest pyr-list)
  (cond ((null pyr-list) pyr)
	((mapcar-tree
	  #'(lambda (im1 im2)
	      (if (or (and (image-p im1) (image-p im2))
		      (and (null im1) (null im2)))
		  nil
		  (error "Pyramid tree structures are different")))
	  (list (image-tree pyr) (image-tree (car pyr-list)))))
	(t (apply 'check-size (car pyr-list) (cdr pyr-list)))))

(defmethod notify-of-inferior-destruction ((pyr pyramid) inf-vbl)
  (cerror "Destroy both ~A and ~A"
	  "Trying to destroy ~A, which is contained in ~A."
	  inf-vbl sup-vbl)
  (destroy pyr))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; filter-pyramid methods
;;;   for accessing pyramid images
;;;   for building pyramids from images (forward transform)
;;;   and for collapsing pyramids into images (inverse transform)

(defmethod access-by-recipe ((pyramid filter-pyramid) recipe &key ((:-> res)))
  (let ((image (recursive-access-pyramid-image 
		pyramid (image-tree pyramid) recipe)))
    (cond ((null res) image)
	  ((image-p res) (copy image :-> res))
	  (t (set-name image res) image))))

(defun recursive-access-pyramid-image (pyramid sub-tree recipe)
  (cond ((not recipe)				
	 (car sub-tree))
	((nth (car recipe) (cdr sub-tree)) ; if sub-tree exists, use it
	 (recursive-access-pyramid-image
	  pyramid (nth (car recipe) (cdr sub-tree)) (cdr recipe)))
	(t				; sub-tree doesn't exist, compute it
	 (let ((image (apply-filter
		       (nth (car recipe) (forward-filters pyramid))
		       (car sub-tree))))
	   (pushnew pyramid (superiors-of image))
	   (setf (nth (car recipe) (cdr sub-tree))
		 (cons image (list-of-length (length (forward-filters pyramid)) nil)))
	   (set-not-current pyramid))
	 (recursive-access-pyramid-image
	  pyramid (nth (car recipe) (cdr sub-tree)) (cdr recipe)))
	))

;;; build the pyramid to level "level" at each level, decompose the
;;; image specified by "which-image" into its sub-images
(defmethod build ((pyramid filter-pyramid) level &key (which-image 0))
  (loop for i from 0 below level do
	(loop for j from 0 below (length (forward-filters pyramid)) do
	      (access-by-recipe pyramid (append (list-of-length i which-image) (list j)))))
  pyramid)

;;; returns a sub-tree from a pyramid's image-tree
(defmethod sub-tree ((pyramid filter-pyramid) recipe)
  (access-by-recipe pyramid recipe)	; make sure the image itself is computed
  (recursive-access-sub-tree (image-tree pyramid) recipe)) ; access the sub-tree

(defun recursive-access-sub-tree (sub-tree recipe)
  (cond ((not recipe)
	 sub-tree)
	(t
	 (recursive-access-sub-tree (nth (car recipe) (cdr sub-tree)) (cdr recipe)))))

;;; COLLAPSE
;;; method for collapsing a pyramid representation, i.e., for inverting the
;;; pyramid transform.
;;; "from-level" specifies the deepest level to expand from
;;;       from-level = t means to expand from the existing leaves of the tree
;;; "from-level-only" = t means to expand only from images in the level specified
;;;       by from-level
;;; "sub-tree-recipe" specifies where to expand to
;;; "which-images" is a list specifying which images at each level should be expanded
;;; examples:
;;; default values: 
;;;       expand the leaves all the way to reconstruct the root image.
;;; (from-level 3) (from-level-only t) (which-images '(t nil nil nil)):
;;;       expand the first image at level 3 all the way to the root level
;;; (from-level 3) (from-level-only t) (which-images '(t t t t)):
;;;       expand and combine all the images at level 3 all the way to the root level
;;; (from-level t) (which-images '(t nil nil nil)):
;;;       expand and combine the first image from each level all the way to the root level
(defmethod collapse ((pyramid filter-pyramid)
		     &key (from-level t)
		     (from-level-only nil)
		     (sub-tree-recipe '())
		     (which-images (list-of-length (length (inverse-filters pyramid)) t))
		     ->)
  (with-result ((result ->) (access-by-recipe pyramid sub-tree-recipe)
		'collapse pyramid
		:from-level from-level :from-level-only from-level-only
		:sub-tree-recipe sub-tree-recipe :which-images which-images)
    (recursive-collapse-pyramid pyramid
				from-level
				from-level-only
				which-images
				sub-tree-recipe
				(sub-tree pyramid sub-tree-recipe)
				(zero! result))
    result))

(defun recursive-collapse-pyramid (pyramid
				   from-level from-level-only
				   which-images recipe
				   sub-tree
				   result)
;  (format t "~% expanding ~d" recipe)
  (let ((leaf-p (leaf-p pyramid recipe sub-tree))
	(in-which-images-p (if (not recipe)
			       t
			       (nth (car (last recipe)) which-images)))
	(at-from-level-p (eq (length recipe) from-level)))
    (cond ((and from-level-only at-from-level-p in-which-images-p)	; return the image
	   (copy (car sub-tree) :-> result))
	  ((and from-level-only leaf-p in-which-images-p)	; return 0 image
	   result)				
	  ((and at-from-level-p in-which-images-p)	; return the image
	   (copy (car sub-tree) :-> result))
	  ((and leaf-p in-which-images-p)	; return the image
	   (copy (car sub-tree) :-> result))
	  ((or at-from-level-p leaf-p)		; return 0 image
	   result)
	  (t
	   (loop for i from 0 below (length (cdr sub-tree)) do
	     (with-local-viewables ((next-result (similar (car (nth i (cdr sub-tree))))))
	       (recursive-collapse-pyramid pyramid
					   from-level
					   from-level-only
					   which-images
					   (append recipe (list i))
					   (nth i (cdr sub-tree))
					   next-result)
	       	; expand next-result and add it to result
	       (expand-filter (nth i (inverse-filters pyramid)) 
			      next-result :zero nil :-> result)))
	   result))))

;;; if (cdr sub-tree) is all nils then it is a leaf
;;; otherwise access each of the images to make sure they exist
(defmethod leaf-p ((pyramid filter-pyramid) recipe sub-tree)
  (cond ((every 'null (cdr sub-tree))
	 t)
	(t
	 (loop for i from 0 below (length (cdr sub-tree)) do
	   (if (not (nth i (cdr sub-tree)))
	       (access-by-recipe pyramid (append recipe (list i)))))
	 nil)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; pyramid operations that do imops

(defmethod unary-image-operation ((pyramid pyramid) (result pyramid) func &rest args)
  (if (null (image-tree result))
      (setf (image-tree result)
	    (mapcar-tree func
			 (list (image-tree pyramid))
			 :args args))
      (mapcar-tree func
		   (list (image-tree pyramid))
		   :args args
		   :result-tree (image-tree result))))

(defmethod binary-image-operation ((pyramid1 pyramid) (pyramid2 pyramid) (result pyramid) 
				   func &rest args)
  (if (null (image-tree result))
      (setf (image-tree result)
	    (mapcar-tree func
			 (list (image-tree pyramid1) (image-tree pyramid2))
			 :args args))
      (mapcar-tree func
		   (list (image-tree pyramid1) (image-tree pyramid2))
		   :args args
		   :result-tree (image-tree result))))

(defmethod copy ((pyr pyramid) &key ->)
  (with-result ((result ->) pyr 'copy pyr)
    (unary-image-operation pyr result 'copy)
    result))

(defmethod add ((pyramid1 pyramid) (pyramid2 pyramid) &key ->)
  (with-result ((result ->) pyramid1 'add pyramid1 pyramid2)
    (binary-image-operation pyramid1 pyramid2 result 'add)
    result))

(defmethod add ((pyramid pyramid) (const number) &key ->)
  (with-result ((result ->) pyramid 'add pyramid const)
    (unary-image-operation pyramid result 'add const)
    result))

(defmethod mul ((pyramid1 pyramid) (pyramid2 pyramid) &key ->)
  (with-result ((result ->) pyramid1 'mul pyramid1 pyramid2)
    (binary-image-operation pyramid1 pyramid2 result 'mul)
    result))

(defmethod mul ((pyramid pyramid) (const number) &key ->)
  (with-result ((result ->) pyramid 'mul pyramid const)
    (unary-image-operation pyramid result 'mul const)
    result))

(defmethod sub ((pyramid1 pyramid) (pyramid2 pyramid) &key ->)
  (with-result ((result ->) pyramid1 'sub pyramid1 pyramid2)
    (binary-image-operation pyramid1 pyramid2 result 'sub)
    result))

(defmethod sub ((pyramid pyramid) (const number) &key ->)
  (with-result ((result ->) pyramid 'sub pyramid const)
    (unary-image-operation pyramid result 'sub const)
    result))

(defmethod div ((pyramid1 pyramid) (pyramid2 pyramid)
		&key (zero-val *div-by-zero-result*) suppress-warning ->)
  (with-result ((result ->) pyramid1 'div pyramid1 pyramid2)
    (binary-image-operation pyramid1 pyramid2 result 'div :zero-val zero-val
			    :suppress-warning suppress-warning)
    result))

(defmethod div ((pyramid pyramid) (const number)
		&key (zero-val *div-by-zero-result*) suppress-warning ->)
  (with-result ((result ->) pyramid 'div pyramid const)
    (unary-image-operation pyramid result 'div const :zero-val zero-val
			   :suppress-warning suppress-warning)
    result))

;;; ***What about the number-pyramid method?

(defmethod square ((pyramid pyramid) &key ->)
  (with-result ((result ->) pyramid 'square pyramid)
    (unary-image-operation pyramid result 'square)
    result))

(defmethod square-root ((pyramid pyramid) &key ->)
  (with-result ((result ->) pyramid 'square-root pyramid)
    (unary-image-operation pyramid result 'square-root)
    result))

(defmethod abs-value ((pyramid pyramid) &key ->)
  (with-result ((result ->) pyramid 'abs-value pyramid)
    (unary-image-operation pyramid result 'abs-value)
    result))

(defmethod apply-filter ((filter filter) (pyramid pyramid) &key ->)
  (with-result ((result ->) pyramid 'apply-filter pyramid filter)
    (unary-image-operation pyramid result 'apply-filter filter)
    result))

;;; Should allow :zero keyword
(defmethod expand-filter ((filter filter) (pyramid pyramid) &key (zero t) ->)
  (with-result ((result ->) pyramid 'expand-filter pyramid filter)
    (unary-image-operation pyramid result 'expand-filter filter :zero zero)
    result))

(defmethod blur ((pyramid pyramid) &key (edge-handler :reflect1) (level 1) ->)
  (with-result ((result ->) pyramid
		'blur pyramid :level level :edge-handler edge-handler)
    (unary-image-operation pyramid result 'blur :level level :edge-handler edge-handler)
    result))

(defmethod gauss-in ((pyramid pyramid) &key (edge-handler :reflect1) ->)
  (with-result ((result ->) pyramid 'gauss-in pyramid :edge-handler edge-handler)
    (unary-image-operation pyramid result 'gauss-in :edge-handler edge-handler)
    result))

(defmethod gauss-out ((pyramid pyramid) &key (edge-handler :reflect1) ->)
  (with-result ((result ->) pyramid 'gauss-out pyramid :edge-handler edge-handler)
    (unary-image-operation pyramid result 'gauss-out :edge-handler edge-handler)
    result))

;;; threshold operations on pyramids
;;; result-type must be float for now until bit-pyramids are implemented
;;; *** need to write the rest of these ***

(defmethod greater-than ((pyramid1 pyramid) (pyramid2 pyramid)
			 &key (result-type 'single-float) ->)
  (with-result ((result ->) pyramid1 
		'greater-than pyramid1 pyramid2 :result-type result-type)
    (binary-image-operation pyramid1 pyramid2 result 'greater-than
			    :result-type result-type)
    result))

(defmethod greater-than ((pyramid pyramid) (const number)
			 &key (result-type 'single-float) ->)
  (with-result ((result ->) pyramid 
		'greater-than pyramid const :result-type result-type)
    (unary-image-operation pyramid result 'greater-than const
			   :result-type result-type)
    result))

;;; write other pyramid ops: ?

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; 1d-qmf-pyramid

(def-simple-class one-d-qmf-pyramid (filter-pyramid)
  ())

;;; *** This is pretty ugly!  Can we make this a method that does something
;;; a little different for 1-d-images?
;;; lo-filt may be a list or a one-d-filter.
;;; image must be a 1d image
(defun make-one-d-qmf-pyramid (image 
			       &key 
			       (lo-filt '(-0.07610252183564592  
					  0.3535533905932738  
					  0.8593118248578394 
					  0.3535533905932738  
					  -0.07610252183564592))
			       (level 0) (edge-handler :reflect1)
			       name ->
			       (display-type nil))
  (when (or (not (one-d-image-p image)) (> (y-dim image) (x-dim image)))
    (error "Image must be a 1xN one-d-image"))
  (setq lo-filt (if (filter-p lo-filt) 
		    lo-filt
		    (make-filter lo-filt)))
  (let* ((hi-filt (shift-by-pi lo-filt)))
    (setf (edge-handler lo-filt) edge-handler)
    (setf (edge-handler hi-filt) edge-handler)
    (setf (start-vector lo-filt) '(0))
    (setf (start-vector hi-filt) '(1))
    (setf (step-vector lo-filt) '(2))
    (setf (step-vector hi-filt) '(2))
    (let* ((forward-filters (list lo-filt hi-filt))
	   (inverse-filters forward-filters)) ; do they need to be rescaled?
      (with-result ((pyr ->)
		    (list :class 'one-d-qmf-pyramid
			  :display-type display-type :name name
			  :forward-filters forward-filters :inverse-filters inverse-filters
			  :image-tree (cons image (list nil nil)))
		    'make-one-d-qmf-pyramid image :lo-filt lo-filt :level level
		    :edge-handler edge-handler :display-type display-type)
	(pushnew pyr (superiors-of image))
	(build pyr level)
	pyr))))

(defmethod access-low ((pyr one-d-qmf-pyramid) level &key ((:-> res)))
  (access-by-recipe pyr (list-of-length level 0) :-> res))

(defmethod access-band ((pyr one-d-qmf-pyramid) level &key ((:-> res)))
  (access-by-recipe pyr (append (list-of-length (1- level) 0) (list 1)) :-> res))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; QMF pyramids

(def-simple-class qmf-pyramid (filter-pyramid)
  ())

(def-simple-class separable-qmf-pyramid (qmf-pyramid) ())
(def-simple-class hex-qmf-pyramid (qmf-pyramid) ())

;;; *** Make this a method on image and one-d-image ...
;;; lo-filt may be a list or a one-d-filter.
(defun make-separable-qmf-pyramid (image &key
					 (display-type nil display-supplied-p)
					 (lo-filt '(-0.07610252183564592  
						    0.3535533905932738  
						    0.8593118248578394 
						    0.3535533905932738  
						    -0.07610252183564592))
					 (level 0)
					 name ->
					 (edge-handler (if (filter-p lo-filt)
							   (edge-handler lo-filt)
							   :reflect1)))
  (when (one-d-image-p image) (error "Use make-one-d-qmf-pyramid for one-d-images"))
  (setq lo-filt (if (filter-p lo-filt) lo-filt (make-filter lo-filt)))
  (let* ((hi-filt (shift-by-pi lo-filt))
	 (low-filt (make-separable-filter lo-filt lo-filt
					  :edge-handler edge-handler
					  :start-vector '(0 0)
					  :step-vector '(2 2)))
	 (diag-filt (make-separable-filter hi-filt hi-filt
					   :edge-handler edge-handler
					   :start-vector '(1 1)
					   :step-vector '(2 2)))
	 (hor-filt (make-separable-filter hi-filt lo-filt
					  :edge-handler edge-handler
					  :start-vector '(1 0)
					  :step-vector '(2 2)))
	 (vert-filt (make-separable-filter lo-filt hi-filt
					   :edge-handler edge-handler
					   :start-vector '(0 1)
					   :step-vector '(2 2)))
	 (forward-filters (list low-filt diag-filt hor-filt vert-filt))
	 (inverse-filters forward-filters))   ; do they need to be rescaled?
    (with-result ((result ->)
		  `(:class separable-qmf-pyramid
		    ,@(when display-supplied-p (list :display-type display-type))
		    :name ,name
		    :forward-filters ,forward-filters
		    :inverse-filters ,inverse-filters
		    :image-tree ,(cons image (list nil nil nil nil)))
		  'make-separable-qmf-pyramid image :lo-filt lo-filt :edge-handler edge-handler)
      (build result level)
      result)))

;;; *** Put newer filter coeffs in here!
(defun default-hex-qmf-filter-coeffs ()
  (let (( a  0.6935061 )
	( b  0.2905438 )
	( c -0.0323747 )
	( e -0.0027388 )
	( d -0.0319443 )
	( f -0.0028679 )
	( z  0.0       ))
    (list  (list z z z e z f z f z e z z z )
	   (list z z f z c z d z c z f z z )
	   (list z f z d z b z b z d z f z )
	   (list e z c z b z a z b z c z e )
	   (list z f z d z b z b z d z f z )
	   (list z z f z c z d z c z f z z )
	   (list z z z e z f z f z e z z z ))))

(defun make-hex-qmf-pyramid (image &key 
				   (lo-filt (default-hex-qmf-filter-coeffs))
				   (level 0) 
				   name ->
				   (edge-handler (if (filter-p lo-filt)
						     (edge-handler lo-filt)
						     nil))
				   (display-type nil display-supplied-p))
  (when (one-d-image-p image) (error "Can not make hex pyramids on one-d-images"))
  (setq lo-filt (if (filter-p lo-filt) 
		    lo-filt
		    (make-hex-filter lo-filt)))
  (let* ((low (copy lo-filt))
	 (ldiag (hex-modulate (copy lo-filt) '(1 1) 4))
	 (rdiag (hex-modulate (copy lo-filt) '(1 -1) 4))
	 (vert  (hex-modulate (copy lo-filt) '(1 0) 2))
	 (forward-filters (list low ldiag rdiag vert))
	 (inverse-filters forward-filters))
    (setf (edge-handler low) edge-handler
	  (start-vector low) '(0 0)
	  (step-vector low) '(2 2)
	  (hex-start low) 0)
    (setf (edge-handler rdiag) edge-handler
	  (start-vector rdiag) '(0 0)
	  (step-vector rdiag) '(2 2)
	  (hex-start rdiag) 1)
    (setf (edge-handler ldiag) edge-handler
	  (start-vector ldiag) '(1 1)
	  (step-vector ldiag) '(2 2)
	  (hex-start ldiag) 0)
    (setf (edge-handler vert) edge-handler
	  (start-vector vert) '(1 1)
	  (step-vector vert) '(2 2)
	  (hex-start vert) 1)
    (with-result ((result ->)
		  `(:class hex-qmf-pyramid
		    ,@(when display-supplied-p (list :display-type display-type))
		    :forward-filters ,forward-filters
		    :inverse-filters ,inverse-filters
		    :image-tree ,(cons image (list nil nil nil nil)))
		  'make-hex-qmf-pyramid image lo-filt)
      (build result level)
      result)))

(defmethod access-low ((pyr qmf-pyramid) level &key ((:-> res)))
  (access-by-recipe pyr (list-of-length level 0) :-> res))

(defmethod access-vert ((pyr qmf-pyramid) level &key ((:-> res)))
  (access-by-recipe pyr (nconc (list-of-length (1- level) 0) (list 3)) :-> res))

(defmethod access-diag ((pyr separable-qmf-pyramid) level &key ((:-> res)))
  (access-by-recipe pyr (nconc (list-of-length (1- level) 0) (list 1)) :-> res))

(defmethod access-hor ((pyr separable-qmf-pyramid) level &key ((:-> res)))
  (access-by-recipe pyr (nconc (list-of-length (1- level) 0) (list 2)) :-> res))

(defmethod access-ldiag ((pyr hex-qmf-pyramid) level &key ((:-> res)))
  (access-by-recipe pyr (nconc (list-of-length (1- level) 0) (list 1)) :-> res))

(defmethod access-rdiag ((pyr hex-qmf-pyramid) level &key ((:-> res)))
  (access-by-recipe pyr (nconc (list-of-length (1- level) 0) (list 2)) :-> res))

#|
(make-separable-qmf-pyramid einstein 
			    '(-0.07610252183564592  0.3535533905932738  
			      0.8593118248578394 0.3535533905932738  
			      -0.07610252183564592) 
			    :-> 'pyr :level 1)
(build pyr 5)
(collapse pyr :-> 'reconstruct)
|#


;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
