;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: parameterized-filter.lisp
;;;  Author: Heeger/Simoncelli
;;;  Description: Parameterized filters for more efficient convolution
;;;  Creation Date: March, 1989
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '(parameterized-filter))

;;; Coefficients is a vector of floats.  Index-array is an array of indices
;;; into the coefficients, where 1 corresponds to the first element.  The 
;;; zeroeth element is ignored.
;;; NOTE: zeroes in the index-array ALWAYS represent a zero in the kernel. They 
;;; are not used to index into the parameter array.
(def-simple-class parameterized-filter (filter)
  (index-array
   coefficients))

(defmacro parameterized-filter-p (obj)
  `(typep ,obj 'parameterized-filter))

(defun make-parameterized-filter (index-array 
				  coefficients
				  &key 
				  name ->
				  (display-type nil)
				  (edge-handler nil) 
				  (start-vector '(0 0))
				  (step-vector  '(1 1))
				  (hex-start     nil))
  (unless (typep index-array '(array (signed-byte 32) (* *)))
    (error "Index array is not a 2D fixnum array: ~A~%" index-array))
  (unless (typep coefficients '(array single-float (*)))
    (error "Coefficient array is not a 1D float array: ~A~%" coefficients))
  ;; check that index-array is consistent with size of coefficients vector.
  (when (or (> (maximum index-array) (1- (length coefficients)))
	    (< (minimum index-array) 0))
    (error "Index array ~a contains indices out-of-range [0,~A)~%" 
	   index-array (length coefficients)))
  (let ((kernel (make-array (dimensions index-array) :element-type 'single-float
			    :initial-element 0.0)))
    (setf (aref coefficients 0) 0.0)
    (loop for j from 0 below (y-dim kernel) do
	  (loop for i from 0 below (x-dim kernel) do
		(setf (aref kernel j i) (aref coefficients (aref index-array j i)))))
    (let* ((grid (make-instance 'grid
				:start-vector (mapcar #'mod start-vector step-vector)
				:step-vector step-vector
				:hex-start hex-start)))
      (with-result ((filter nil)
		    `(:class 'parameterized-filter
		      :index-array ,index-array
		      :coefficients ,coefficients
		      :display-type ,display-type
		      :kernel ,kernel
		      :name (or name ->)
		      :filter-grid ,grid
		      :edge-handler ,edge-handler)
		    'make-parameterized-filter index-array coefficients
		    :name name :display-type display-type :edge-handler edge-handler
		    :start-vector start-vector :step-vector step-vector :hex-start hex-start)
	filter))))

(defmethod apply-filter ((filter parameterized-filter) (im image) &key ->)
  (when (or (> (y-dim filter) (y-dim im)) (> (x-dim filter) (x-dim im)))
    (error "Filter dimensions ~A are greater than image dimensions ~A." 
	   (dimensions filter) (dimensions im)))
  (when (or (> (y-start filter) (y-dim im)) (> (x-start filter) (x-dim im)))
    (error "Filter grid start ~A is greater than image dimensions ~A."
	   (start-vector (filter-grid filter)) (dimensions im)))
  (with-slots (index-array coefficients kernel filter-grid edge-handler) filter
    (with-result ((result ->)
		  (list :class (class-of im)
			:dimensions (list 
				     (ceiling (- (y-dim im) (y-start filter)) 
					      (y-step filter))
				     (ceiling (- (x-dim im) (x-start filter))
					      (x-step filter))))
		  'apply-filter filter im)
      (when (eq im result) 
	(error "Destructive modification impossible for convolutions."))
      (let ((stagger (if (null (hex-start filter-grid)) 0
			 (1+ (hex-start filter-grid)))))
	(if edge-handler
	    (internal-p-filter (data im) (x-dim im) (y-dim im)
			       index-array coefficients (length coefficients)
			       kernel (similar kernel) ;temp space
			       (x-dim filter) (y-dim filter)
			       (x-start filter) (x-step filter) 
			       (y-start filter) (y-step filter) 
			       (data result)
			       edge-handler
			       stagger)
	    (internal-wrap-p-filter (data im) (x-dim im) (y-dim im)
				    index-array coefficients (length coefficients)
				    (x-dim filter) (y-dim filter)
				    (x-start filter) (x-step filter) 
				    (y-start filter) (y-step filter) 
				    (data result)
				    stagger)))
      result)))
  
(defmethod expand-filter ((filter parameterized-filter) (im image) &key (zero t) ((:-> res)))
  (cond ((not (image-p res)) 
	 (setq res (make-image (list (* (y-dim im) (y-step filter))
				     (* (x-dim im) (x-step filter)))
			       :-> res)))
	((eq im res) (error "Destructive modification impossible for convolutions."))
	((or (/= (ceiling (- (y-dim res) (y-start filter)) (y-step filter))
		 (y-dim im))
	     (/= (ceiling (- (x-dim res) (x-start filter)) (x-step filter))
		 (x-dim im)))
	 (error "Result argument size is incompatible.")))
  (when (or (> (y-dim filter) (y-dim res)) (> (x-dim filter) (x-dim res)))
    (error "Filter dimensions ~A are greater than image dimensions ~A." 
	   (dimensions filter) (dimensions res)))
  (when (or (> (y-start filter) (y-dim im)) (> (x-start filter) (x-dim im)))
    (error "Filter grid start ~A is greater than image dimensions ~A."
	   (start-vector (filter-grid filter)) (dimensions im)))
  (with-slots (index-array coefficients kernel edge-handler filter-grid) filter
    (with-result ((result res) res 'expand-filter filter im)
      (when zero (zero! result))
      (let ((stagger (if (null (hex-start filter-grid)) 0
			 (1+ (hex-start filter-grid)))))
        (if edge-handler
	    (internal-p-expand (data im) 
			       index-array coefficients (length coefficients)
			       kernel (similar kernel)
			       (x-dim filter) (y-dim filter)
			       (x-start filter) (x-step filter) 
			       (y-start filter) (y-step filter) 
			       (data result) (x-dim result) (y-dim result)
			       edge-handler
			       stagger)
	    (internal-wrap-p-expand (data im)
				    index-array coefficients (length coefficients)
				    (x-dim filter) (y-dim filter)
				    (x-start filter) (x-step filter) 
				    (y-start filter) (y-step filter) 
				    (data result) (x-dim result) (y-dim result)
				    stagger)))
      result)))

#|
;;;; Try it out.
(make-filter '((0.0  0.15  0.0)
	       (0.15 0.4  0.15)
	       (0.0  0.15  0.0))
	     :edge-handler :reflect1
	     :-> 'filt)
(make-parameterized-filter (make-array '(3 3) :element-type 'fixnum
				       :initial-contents '((0 1 0) (1 2 1) (0 1 0)))
			   (make-array 3 :element-type 'float 
				       :initial-contents '(0.0 0.15 0.4))
			   :edge-handler :reflect1
			   :-> 'p-filt)
(mean-square-error (apply-filter p-filt im :-> 'res2) (apply-filter filt im :-> 'res1))
|#





;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
