;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: gaussian-pyramid.lisp
;;;  Author: David Heeger
;;;  Description: gaussian and laplacian pyramids
;;;  Creation Date: revised summer '90
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)

(export '(gaussian-pyramid make-gaussian-pyramid gaussian-pyramid-p
	  laplacian-pyramid make-laplacian-pyramid laplacian-pyramid-p
	  build access collapse
	  mul sub div add square-error abs-error linear-xform clip
	  abs-value square square-root copy zero! point-operation 
	  apply-filter expand-filter blur
	  greater-than greater-than-or-equal-to less-than less-than-or-equal-to
	  minimum maximum range))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; gaussian pyramid class

(def-simple-class gaussian-pyramid (viewable)
  ((filter :initform (make-separable-filter 
		      '(0.0625 0.25 0.375 0.25 0.0625)
		      '(0.0625 0.25 0.375 0.25 0.0625)
		      :step-vector '(2 2)
		      :edge-handler nil))
   (image-list :initform nil))
  (:default-initargs :display-type 'pasteup))

(def-simple-class laplacian-pyramid (gaussian-pyramid)
  (low-pass-image))

(defmacro gaussian-pyramid-p (obj)
  `(typep ,obj 'gaussian-pyramid))

(defmacro laplacian-pyramid-p (obj)
  `(typep ,obj 'laplacian-pyramid))

(defmethod print-object ((pyr gaussian-pyramid) stream)
  (format stream "#<~A " (object-class-name pyr))  
  (format stream " ~S>" (name pyr)))

(defmethod inferiors-of ((pyr gaussian-pyramid))
  (image-list pyr))

(defmethod inferiors-of ((pyr laplacian-pyramid))
  (if (low-pass-image pyr)
      (cons (low-pass-image pyr) (image-list pyr))
      (image-list pyr)))

(defmethod notify-of-inferior-destruction ((sup-vbl gaussian-pyramid) inf-vbl)
  (cerror "Destroy both ~A and ~A"
	  "Trying to destroy ~A, which is contained in ~A."
	  inf-vbl sup-vbl)
  (destroy sup-vbl))
   
;;; *** Shouldn't we force the step-vector of a passed-in filter to be '(2 2)???
(defun make-gaussian-pyramid (image &rest initargs &key level name display-type filter ->)
  (declare (ignore name display-type filter))
  (when -> (setf (getf initargs :name) ->))
  (remf initargs :level)
  (with-result ((result nil)
		`(:class gaussian-pyramid :image-list ,(list image) ,@initargs)
		'apply 'make-gaussian-pyramid image initargs)
    (when level (build result level))
    result))

(defun make-laplacian-pyramid (image &rest initargs &key level name display-type filter ->)
  (declare (ignore name display-type filter))
  (when -> (setf (getf initargs :name) ->))
  (remf initargs :level)
  (with-result ((result nil)
		`(:class laplacian-pyramid :low-pass-image ,image ,@initargs)
		'apply 'make-laplacian-pyramid image initargs)
    (when level (build result level))
    result))
   
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Set-Result and Check-Size

(defmethod set-result ((name t) (model gaussian-pyramid))
  (check-type name viewable-name)
  (make-instance 
   (class-of model)
   :name name
   :display-type (display-type model)
   :filter (filter model)
   :image-list (mapcar #'similar (image-list model))))

;;; *** This should do some more error checking ***
(defmethod set-result ((res gaussian-pyramid) (model-plist list))
  (unless (typep res (getf model-plist :class))
    (error "Result ~a is incompatible with argument type ~a"
	   res (getf model-plist :class)))
  res)

(defmethod initialize-instance :after
     ((pyr gaussian-pyramid) &key &allow-other-keys)
  (dolist (im (image-list pyr)) (pushnew pyr (superiors-of im))))

;;; Check that the image lists have the same structure (same length)
;;; -- leave the checking of the individual image dimensions to the
;;; functions which operate on the images themselves.
(defmethod check-size ((pyr gaussian-pyramid) &rest pyr-list)
  (cond ((null pyr-list) pyr)
	((not (= (length (image-list pyr)) (length (image-list (car pyr-list)))))
	 (error "Pyramid structures are different"))
	(t (apply 'check-size (car pyr-list) (cdr pyr-list)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Build and access methods

;;; must set not current when modify the pyramid structure

(defmethod access ((pyr gaussian-pyramid) level
		   &key ((:-> res)))
  (when (< (1- (length (image-list pyr))) level)
    (set-not-current pyr)
    (loop for i from (length (image-list pyr)) to level 
	  for new-low-image = (apply-filter (filter pyr) (car (last (image-list pyr))))
	  do
	  (setf (image-list pyr) (append (image-list pyr) (list new-low-image)))
	  (pushnew pyr (superiors-of new-low-image))))
  (let ((image (nth level (image-list pyr))))
    (cond ((null res) image)
	  ((image-p res) (copy image :-> res))
	  (t (set-name image res) image))))

(defmethod access ((pyr laplacian-pyramid) level
		   &key ((:-> res)))
  (when (< (1- (length (image-list pyr))) level)
    (set-not-current pyr)
    (loop for i from (length (image-list pyr)) to level
	  for new-low-image = (apply-filter (filter pyr) (low-pass-image pyr))
	  for new-band-image = (expand-filter (filter pyr) new-low-image
					      :-> (similar (low-pass-image pyr)))
	  do
	  (mul new-band-image (* (x-step (filter pyr)) (y-step (filter pyr)))
	       :-> new-band-image)
	  (sub (low-pass-image pyr) new-band-image :-> new-band-image)
	  (setf (image-list pyr) (append (image-list pyr) (list new-band-image)))
	  (pushnew pyr (superiors-of new-band-image))
	  (delete pyr (superiors-of (low-pass-image pyr)))
	  (setf (low-pass-image pyr) new-low-image)
	  (pushnew pyr (superiors-of new-low-image))))
  (let ((image (nth level (image-list pyr))))
    (cond ((null res) image)
	  ((image-p res) (copy image :-> res))
	  (t (set-name image res) image))))

(defmethod build ((pyr gaussian-pyramid) level)
  (access pyr level)
  pyr)

;;; *** Conses extra images that are not destroyed ***
(defmethod collapse ((pyr laplacian-pyramid))
  (let ((collapse-img (low-pass-image pyr)))
    (loop for i from (1- (length (image-list pyr))) downto 0
	  for new-image = (similar (access pyr i))
	  do
	  (expand-filter (filter pyr) collapse-img :-> new-image)
	  ;;(destroy collapse-img)
	  (mul new-image (* (x-step (filter pyr)) (y-step (filter pyr)))
	       :-> new-image)
	  (+. (access pyr i) new-image :-> new-image)
	  (setq collapse-img new-image))
    collapse-img))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Laplacian pyramid operations

(defmacro unary-gaussian-pyramid-op (function pyr res
					       &rest args)
  (let ((arg-im (gensym))
	(result-pyr (gensym))
	(res-im (gensym)))
    `(with-result ((,result-pyr ,res) ,pyr ,function ,pyr ,@args)
      (loop for ,arg-im in (image-list ,pyr)
	    for ,res-im in (image-list ,result-pyr)
	    do (funcall ,function ,arg-im ,@args :-> ,res-im))
      ,result-pyr)))

(defmacro binary-gaussian-pyramid-op (function pyr1 pyr2 res
				       &rest args)
  (let ((arg1 (gensym))
	(arg2 (gensym))
	(result-pyr (gensym))
	(res-im (gensym)))
    `(with-result ((,result-pyr ,res) (check-size ,pyr1 ,pyr2) ,function ,pyr1 ,pyr2 ,@args)
      (loop for ,arg1 in (image-list ,pyr1)
	    for ,arg2 in (image-list ,pyr2)
	    for ,res-im in (image-list ,result-pyr)
	    do (funcall ,function ,arg1 ,arg2 ,@args :-> ,res-im))
      ,result-pyr)))

(defmethod add ((pyr1 gaussian-pyramid) (pyr2 gaussian-pyramid) &key ->)
  (binary-gaussian-pyramid-op 'add pyr1 pyr2 ->))

(defmethod mul ((pyr1 gaussian-pyramid) (pyr2 gaussian-pyramid) &key ->)
  (binary-gaussian-pyramid-op 'mul pyr1 pyr2 ->))

(defmethod sub ((pyr1 gaussian-pyramid) (pyr2 gaussian-pyramid) &key ->)
  (binary-gaussian-pyramid-op 'sub pyr1 pyr2 ->))

(defmethod div ((pyr1 gaussian-pyramid) (pyr2 gaussian-pyramid)
		&key (zero-val *div-by-zero-result*) suppress-warning ->)
  (binary-gaussian-pyramid-op 'div pyr1 pyr2 -> :zero-val zero-val
			      :suppress-warning suppress-warning))

(defmethod square-error ((pyr1 gaussian-pyramid) (pyr2 gaussian-pyramid) &key ->)
  (binary-gaussian-pyramid-op 'square-error pyr1 pyr2 ->))

(defmethod abs-error ((pyr1 gaussian-pyramid) (pyr2 gaussian-pyramid) &key ->)
  (binary-gaussian-pyramid-op 'abs-error pyr1 pyr2 ->))

(defmethod add ((pyr gaussian-pyramid) (const number) &key ->)
  (unary-gaussian-pyramid-op 'add pyr -> const))

(defmethod add ((const number) (pyr gaussian-pyramid) &key ->)
  (unary-gaussian-pyramid-op 'add pyr -> const))

(defmethod sub ((pyr gaussian-pyramid) (const number) &key ->)
  (unary-gaussian-pyramid-op 'sub pyr -> const))

(defmethod sub ((const number) (pyr gaussian-pyramid) &key ->)
  (with-result ((result ->) pyr 'sub const pyr)
    (loop for im in (image-list pyr)
	  for res in (image-list result)
	  do
	  (sub const im :-> res))
    result))

(defmethod mul ((pyr gaussian-pyramid) (const number) &key ->)
  (unary-gaussian-pyramid-op 'mul pyr -> const))

(defmethod mul ((const number) (pyr gaussian-pyramid) &key ->)
  (unary-gaussian-pyramid-op 'mul pyr -> const))

(defmethod div ((pyr gaussian-pyramid) (const number)
		&key (zero-val *div-by-zero-result*) suppress-warning ->)
  (unary-gaussian-pyramid-op 'div pyr -> const :zero-val zero-val
			     :suppress-warning suppress-warning))

(defmethod div ((const number) (pyr gaussian-pyramid) &key ->)
  (with-result ((result ->) pyr 'div const pyr)
    (loop for im in (image-list pyr)
	  for res in (image-list result)
	  do
	  (div const im :-> res))
    result))

(defmethod linear-xform ((pyr gaussian-pyramid) scale offset &key ->)
  (unary-gaussian-pyramid-op 'linear-xform pyr -> scale offset))

;;; *** need to define circular-shift, shift by an amount proportional
;;; to image size.

(defmethod clip ((pyr gaussian-pyramid) below above &key ->)
  (unary-gaussian-pyramid-op 'clip pyr -> below above))

(defmethod abs-value ((pyr gaussian-pyramid) &key ->)
  (unary-gaussian-pyramid-op 'abs-value pyr ->))

(defmethod square ((pyr gaussian-pyramid) &key ->)
  (unary-gaussian-pyramid-op 'square pyr ->))

(defmethod square-root ((pyr gaussian-pyramid) &key ->)
  (unary-gaussian-pyramid-op 'square-root pyr ->))

(defmethod copy ((pyr gaussian-pyramid) &key ->)
  (unary-gaussian-pyramid-op 'copy pyr ->))

(defmethod zero! ((pyr gaussian-pyramid))
  (loop for i from 0 below (length (image-list pyr)) do
	(zero! (nth i (image-list pyr))))
  pyr)

(defmethod zero! ((pyr laplacian-pyramid))
  (loop for i from 0 below (length (image-list pyr)) do
	(zero! (nth i (image-list pyr))))
  (zero! (low-pass-image pyr))
  pyr)
  
(defmethod point-operation ((pyr gaussian-pyramid) (func t) &key binsize ->)
  (unary-gaussian-pyramid-op 'point-operation pyr -> func :binsize binsize))

;;; *** need to define cut and paste, offsets and sizes proportional
;;; to image size.

(defmethod apply-filter ((filter filter) (pyr gaussian-pyramid) &key ->)
  (unary-gaussian-pyramid-op 'apply-filter pyr -> filter))

(defmethod apply-filter ((pyr gaussian-pyramid) (filter filter) &key ->)
  (unary-gaussian-pyramid-op 'apply-filter pyr -> filter))

(defmethod expand-filter ((filter filter) (pyr gaussian-pyramid) &key ->)
  (unary-gaussian-pyramid-op 'expand-filter pyr -> filter))

(defmethod expand-filter ((pyr gaussian-pyramid) (filter filter) &key ->)
  (unary-gaussian-pyramid-op 'expand-filter pyr -> filter))

(defmethod blur ((pyr gaussian-pyramid)
		 &key 
		 (level 1)
		 (kernel '(0.125 0.50 0.750 0.50 0.125))
		 (edge-handler nil) 
		 ->)
  (unary-gaussian-pyramid-op 'blur pyr ->
			      :level level
			      :kernel kernel
			      :edge-handler edge-handler))


;;; *** Need to generify pasteups to deal with bitmap sub-pictures.
;;; Until then, these will crap out when they try to display.

(defmethod greater-than ((pyr1 gaussian-pyramid) (pyr2 gaussian-pyramid)
			 &key ->)
  (binary-gaussian-pyramid-op 'greater-than pyr1 pyr2 ->))

(defmethod less-than ((pyr1 gaussian-pyramid) (pyr2 gaussian-pyramid)
		      &key ->)
  (binary-gaussian-pyramid-op 'less-than pyr1 pyr2 ->))

(defmethod greater-than-or-equal-to ((pyr1 gaussian-pyramid) (pyr2 gaussian-pyramid)
				     &key ->)
  (binary-gaussian-pyramid-op 'greater-than-of-equal-to pyr1 pyr2 ->))

(defmethod less-than-or-equal-to ((pyr1 gaussian-pyramid) (pyr2 gaussian-pyramid)
				  &key ->)
  (binary-gaussian-pyramid-op 'less-than-or-equal-to pyr1 pyr2 ->))

;;; *** greater-than, less-than, etc not defined for comparing with a
;;; constant.  Would need to rescale the number appropriately at the
;;; different scales.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; laplacian pyramid operations that return scalars

(defmethod minimum ((pyr gaussian-pyramid))
  (loop for im in (image-list pyr)
	minimize (minimum im)))

(defmethod maximum ((pyr gaussian-pyramid))
  (loop for im in (image-list pyr)
	maximize (maximum im)))

(defmethod range ((pyr gaussian-pyramid))
  (- (maximum pyr) (minimum pyr)))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Gaussian and laplacian pyramid display.  This should eventually be
;;; rewritten to use pasteups.

;;; Allow vertical or horizontal pyramid display.  Default is horizontal.
;;; *** Should also do square!
(defmethod pasteup-dimensions ((pyr gaussian-pyramid) &key pasteup-format border)
  (let ((x-dim (x-dim (car (image-list pyr))))
	(y-dim (y-dim (car (image-list pyr)))))
    (if (or (null pasteup-format) (eq pasteup-format :horizontal))
	(list y-dim (+ (* 2 x-dim) (* (1- (length (image-list pyr))) border)))
	(list (+ (* 2 y-dim) (* (1- (length (image-list pyr))) border)) x-dim))))

(defmethod pasteup-format-to-layout ((pyr gaussian-pyramid) format)
  (when (eq format :square)
    (warn "Pasteup format must be horizontal or vertical for pyramids")))

;;; *** Behavior when independent-parameters is non-nil is wrong.  It
;;; should be the same as for grays.  How can we abstract this?
(defmethod compute-picture ((pic pasteup) (pyr gaussian-pyramid))
  (setf (system-dependent-frob pic)
	(make-bltable (screen-of (pane-of pic)) (dimensions pic)
		      :bltable (system-dependent-frob pic)))		      
  (clear (system-dependent-frob pic) :color (background (pane-of pic)))
  (loop with f-sum = (volume (filter pyr))
	with format = (if (not (eq (pasteup-format pic) :vertical)) :horizontal :vertical)
	with x-offset = 0
	with y-offset = 0
	for i from 0 
	for im in (image-list pyr)
	for factor = (expt f-sum i)
	for ped = (if (independent-parameters pic) (minimum im) (* (pedestal pic) factor))
	for scale = (if (independent-parameters pic) (range im) (* (scale pic) factor))
	do
	(draw-float-array (system-dependent-frob pic) (data im)
			  ped scale (zoom pic) y-offset x-offset)
	(if (eq format :horizontal)
	    (setq x-offset (+ x-offset (border pic) (x-dim im)))
	    (setq y-offset (+ y-offset (border pic) (y-dim im))))))

(defmethod compute-picture ((pic pasteup) (pyr laplacian-pyramid))
  (setf (system-dependent-frob pic)
	(make-bltable (screen-of (pane-of pic)) (dimensions pic)
		      :bltable (system-dependent-frob pic)))
  (clear (system-dependent-frob pic) :color (background (pane-of pic)))
  (loop with f-sum = (volume (filter pyr))
	with format = (if (not (eq (pasteup-format pic) :vertical)) :horizontal :vertical)
	with x-offset = 0
	with y-offset = 0
	for i from 0 
	for im in (image-list pyr)
	for factor = (expt f-sum i)
	for ped = (if (independent-parameters pic) (minimum im) (/ (pedestal pic) factor))
	for scale = (if (independent-parameters pic) (range im) (* (scale pic) factor))
	do
	(draw-float-array (system-dependent-frob pic) (data im)
			  ped scale (zoom pic) y-offset x-offset)
	(if (eq format :horizontal)
	    (setq x-offset (+ x-offset (border pic) (x-dim im)))
	    (setq y-offset (+ y-offset (border pic) (y-dim im))))))


;;; Local Variables:
;;; buffer-read-only: t 
;;; End:

