;;;
;;; world TROUBLE-WITH-TRIBBLES.
;;;
;;; Basically, the grain, chicken, and fox problem, but patterned after
;;; a Star Trek episode.
;;;
;;; Problem: On planet Rigel, there is some grain, a tribble who eats grain,
;;; and a tribble eater who eats tribbles, when left alone in the same
;;; container.  You are a starship with room for only one thing at a time,
;;; which must be held in your arm during the voyage.
;;; You are to get all three objects to planet Regulus.





;;; MOTION extension provides a process that can execute actions
;;;  when motion is detected.

(load-extension "motion")

;;; GRAIN is eaten by tribbles

(defobject grain (thingoid))

;;; PREDATOR is a template for specific types of predators

(defobject predator (thingoid)
  (hungry nil)
  (hunt-process)			; The process that detects motion
  (diet))

;;; Tribbles eat grain

(defobject tribble (predator)
  (diet '(grain)))

;;; A tribble-eater eats tribbles (duh)

(defobject tribble-eater (predator)
  (diet '(tribble)))

;;; constructing a predator with make-tame-predator creates the predator
;;; object, and a process that performs SUPERVISED-EAT whenever
;;; the predator, or something around the predator moves.

(defun make-tame-predator (predator-type)
  (let ((pred (make-sim-object predator-type)))
    (setp pred 'hunt-process 
	  (start-motion-watching-process pred
					 #'my-container
					 #'supervised-eat))
    pred))

;;;
;;; When a predator is destroyed, the process that watches for prey
;;; must stop too.
;;;

(defmethod destroy-object :before ((self predator))
  (stop-process (query self 'hunt-process)))

;;; a predator will eat all of it's prey if there is no truck around
;;; Also, a predator will not eat if it has not seen a truck since it's
;;; creation.

(defun supervised-eat (pred)
  (let* ((stuff (my-container-neighbors pred))
	 (truck (find-if #'(lambda (x) (typep x 'truck)) stuff))
	 (prey (if (or truck (not (query pred 'hungry)))
		   nil
		 (remove-if-not #'(lambda (x) 
				    (some #'(lambda (y) (typep x y))
					  (query pred 'diet)))
				stuff))))
    ;; (format *terminal-io* "Pred: ~S Stuff: ~S Truck: ~S  Prey ~S Hungry: ~S~%"
    ;;	    pred stuff truck prey (query pred 'hungry))
    (setp pred 'hungry (or (query pred 'hungry) truck))
    (mapc #'destroy-object prey)))

					      
					 

(defun create-world ()
  (let* ((wmap (make-map 
		:nodes `((rigel :display-info ,(->rectangle 25 25 100 100))
			 (regulus :display-info ,(->rectangle 300 25 100 100)))
		:links '((trading-route rigel regulus e :length 20))
		:sectors '((sector-1 (rigel regulus trading-route)))))
	 (events '())
	 (world (make-world wmap events)))
    
    (populate-world world (make-sim-object 'grain) 'rigel)

    ;; Small hack: the order of predator creation will affect
    ;;  whom eats whom first, so let's randomize the order.

    (cond
     ((= 0 (random 2))
      (populate-world world (make-tame-predator 'tribble) 'rigel)
      (populate-world world (make-tame-predator 'tribble-eater) 'rigel))
     (t
      (populate-world world (make-tame-predator 'tribble-eater) 'rigel)
      (populate-world world (make-tame-predator 'tribble) 'rigel)))
     
    world))

;;; Here's the truck you should use for this world:
;;;  it (effectively) only has one arm (arm-1), and no cargo bays, so only
;;;  one thing can be moved from rigel to regulus at a time.
;;; If you try putting predator and prey together in the same arm,
;;; predator will eat prey.

(defun make-starship ()
  (make-truck :truck-id NCC1701
	      :arm2 (arm-2 :capacity 0)
	      :bay1 (bay-1 :capacity 0)
	      :bay2 (bay-2 :capacity 0)))
