;;;;
;;;; Graphical user interface for manual truck control
;;;;
;;;; by Mike Williamson, 5/4/93
;;;;
;;;; Note: Be sure your DISPLAY environment variable is set
;;;; appropriately before starting the Lisp process.
;;;
;;;; To use this, you must first make a connection to the server, and
;;;; set the global variable *client-truck-channel* to the channel
;;;; connected to the sever.
;;;;
;;;; Start the truckdriver with (START-TRUCKDRIVER).
;;;;

(require 'clim)


;;;
;;; The truckdriver consists of a collection of button objects.  Each
;;; button has display-text, an X and Y position on the display, and
;;; an associated function which is called when that button is pushed.
;;; The display-text of a button may change depending on its
;;; current-status.  The current-status is determined by evaluating
;;; the buttons status-form.  Whenever the current-status changes, the
;;; display-form will be evaluated to determine the new display-text.
;;; Note that when the display-form is evaluated, the special variable
;;; *button-current-status* will be bound to the current status of the
;;; button.
;;;

;;; A list of all the buttons
;;;
(defvar *button-list*)
(setf *button-list* '())

;;; A list of the buttons which have changable status
;;;
(defvar *status-form-buttons*)
(setf *status-form-buttons* '())

;;; Bound to the current-status of a button when its
;;; display-form is evaluated
;;;
(defvar *button-current-status*)

(defclass button ()
    (
     (display-form :initarg :display-form)
     (x :initarg :x)
     (y :initarg :y)
     (function :initarg :function)
     (status-form :initform nil :initarg :status-form)
     (display-text)
     (current-status :initform :uninitialized)
     ))


(defun make-button (display-form row col function &key (status-form nil))
  "Make an instance of a button and set its display text appropriately."
  (let ((button 
	 (make-instance 'button 
	   :display-form display-form
	   :x col :y row
	   :function function
	   :status-form status-form)))
    (set-display-text button)
    (if status-form (push button *status-form-buttons*))
    (push button *button-list*)
    (values button)))


(defun set-display-text (button)
  "Set the button's current-status by evaluating its status-form.
   If the current-status changes, then set its display-text by
   evaluating its display-form.  The special variable 
   *button-current-status* is set to the current-status of the
   button while evaluating the display-form."
  (with-slots (current-status status-form display-text display-form) button
    (let ((*button-current-status* (eval status-form)))
      (unless (eql current-status *button-current-status*)
	(setf current-status *button-current-status*)
	(setf display-text (eval display-form))))))


(defun update-status-form-buttons ()
  "Update the current-status and display-text for all buttons that
   have a non-nil status form."
  (dolist (b *status-form-buttons*)
    (set-display-text b)))

	

;;;
;;;
;;; The CLIM application
;;;

(defvar *root-window* (clim:open-root-window :clx))
(defvar *truckdriver-frame* nil)
(defvar *x-border* 10)
(defvar *y-border* 10)

(clim:define-presentation-type button ())

(clim:define-application-frame truckdriver ()
  ()
  (:panes ((button-region :application
			  :display-function 'display-buttons
			  :incremental-redisplay t
			  :scroll-bars nil
			  :default-size 1/2)
	   (output-region :application
			  :display-after-commands nil)
	   (menu :command-menu))))

(define-truckdriver-command push-button ((b 'button :gesture :select))
  "Evaluate the function associated with the given button.
   If the button has an associated status-form, then update the 
   display-text of all buttons possessing status-forms."
  (display-messages (report-stream))
  (with-slots (function status-form) b
    (eval function)
    (when status-form
      (update-status-form-buttons))))


(define-truckdriver-command (exit-truckdriver :menu "Exit Truckdriver") ()
  (clim:frame-exit clim:*application-frame*))


(defmethod display-buttons ((frame truckdriver) stream)
  "Incrementally redisplay all buttons."
  (display-messages (report-stream))
  (dolist (b *button-list*)
    (with-slots (display-text x y current-status) b
      (clim:updating-output (stream 
			     :unique-id b
			     :cache-value current-status)
	(clim:with-output-as-presentation (:type 'button
					   :stream stream
					   :object b)
	  (clim:surrounding-output-with-border 
	    (stream)
	    (clim:stream-set-cursor-position* stream (x-pos x stream) (y-pos y stream))
	    (format stream display-text)))))))

;;;
;;; These functions determine the translation of X and Y coordinates
;;; from "character" positions to pixel positions.
;;;

(defun x-pos (x stream)
  (+ *x-border* (* x (clim:stream-string-width stream "m"))))

(defun y-pos (y stream)
  (+ *y-border* (* y (+ 8 (clim:stream-line-height stream)))))

;;; Output to this stream will go into the output region.
;;;
(defun report-stream ()
  (clim:get-frame-pane clim:*application-frame* 'output-region))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Send commands to the truck and receive output from truck
;;;
;;; Assumes that the truck is connected to the server by a channel
;;; stored in the global *client-truck-channel*.
;;;

;;;(print "Set *testing* to T if not using real truck...")
(defvar *testing* nil)

;;; Automatically send GO control messages?
(defvar *auto-go* nil)


(defun send-command-to-truck (c)
  (format (report-stream) "com: ~a~%" c)
  (unless *testing*
    (send-command *client-truck-channel* c))
  (when *auto-go* 
    ;; kludge to make server correctly order command and control messages
    (sleep 1)
    ;; end kludge
    (send-control-to-truck '(go)))
  )


(defun send-control-to-truck (c)
  (format (report-stream) "con: ~a~%" c)
  (unless *testing*
    (send-control *client-truck-channel* c))
  )


(defun display-messages (stream)
  "Read command and control messages from *client-truck-channel* and
   print them nicely on the given stream."
  (unless *testing*
    (let ((msg nil))
      (loop while (control-message-available? *client-truck-channel*) do
	    (format stream "CON: ~S~%" 
		    (setf msg (read-control *client-truck-channel*)))
	    (when (equal (first msg) 'TIME)
	      (setf *current-time* (second msg)))
	    (wait-delay 10))
      (loop while (command-message-available? *client-truck-channel*) do
	    (format stream "CMD: ~S~%" 
		    (read-command *client-truck-channel*))
	    (wait-delay 10)))))


;;;
;;; Pop-up menus used by the buttons
;;;

;;; A menu to allow choosing a position, 0..15.
;;; The number is returned, or NIL if no item is
;;; selected.
;;;
(defconstant *position-menu* '(0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15))
(defun choose-position ()
  (clim:menu-choose *position-menu*
		    :printer #'(lambda (item stream)
				 (format stream "~2d" item))
		    :cache t
		    :unique-id 1
		    :n-rows 4
		    :n-columns 4
		    :label "Arm position"))

;;; A menu to allow choosing a position, 0..14, or the
;;; value NIL which actually returns :nil to the caller.
;;; (Since NIL is returned if no menu item is selected.)
;;;
(defconstant *position-nil-menu* '((NIL :value :nil) 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14))
(defun choose-position-nil ()
  (clim:menu-choose *position-nil-menu*
		    :printer #'(lambda (item stream)
				 (if (listp item)
				     (format stream "~3A" (car item))
				   (format stream "~3d" item)))
		    :cache t
		    :unique-id 1
		    :n-rows 4
		    :n-columns 4
		    :label "Arm position"))


;;; A menu to allow choosing a speed.  
;;; Returns the syombol SLOW, MEDIUM, or FAST.
;;;
(defconstant *speed-menu* '(("Slow" :value slow)
			    ("Medium" :value medium)
			    ("Fast" :value fast)))
(defun choose-speed ()
  (clim:menu-choose *speed-menu*
		    :cache t
		    :unique-id 1
		    :label "Vehicle speed"))


;;; A menu to allow choosing a truck sensor to set or read.
;;; Returns a symbol naming the sensor.
;;;
(defconstant *truck-thing-menu* '(("Truck sensor" :value truck-sensor)
				  ("Arm 1 sensor" :value arm-1-sensor)
				  ("Arm 2 sensor" :value arm-2-sensor)
				  ("Bay 1 sensor" :value bay-1-sensor)
				  ("Bay 2 sensor" :value bay-2-sensor)
				  ("Tire bay sensor" :value tire-bay-sensor)
				  ("Weapon bay sensor" :value weapon-bay-sensor)
				  ("Fuel tank sensor" :value fuel-tank-sensor)
				  ("Odometer sensor"  :value odometer-sensor)))

(defun choose-truck-thing ()
  (clim:menu-choose *truck-thing-menu*
		    :cache t
		    :unique-id 1))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Some state that is associated with the agent.
;;;

(defvar *current-time* 0)
(defvar *current-speed* 'medium)
(defvar *current-arm* 'arm-1)


;;;
;;; Now, the actual buttons...
;;;

(make-button "Go" 0 0 '(send-control-to-truck '(go)))
(make-button "Go +1" 0 4 '(send-control-to-truck (list 'go (+ 1 *current-time*))))
(make-button '(if *button-current-status*
	       "Go automatically [On] "
	       "Go automatically [Off]")
	     0 11
	     '(setf *auto-go* (not *auto-go*))
	     :status-form '*auto-go*)
(make-button "Check Output" 0 35 '())

(make-button "NW" 2 0 '(send-command-to-truck (list 'change-heading 'NW)))
(make-button "N " 2 3 '(send-command-to-truck (list 'change-heading 'N)))
(make-button "NE" 2 6 '(send-command-to-truck (list 'change-heading 'NE)))
(make-button "W " 3 0 '(send-command-to-truck (list 'change-heading 'W)))
(make-button "E " 3 6 '(send-command-to-truck (list 'change-heading 'E)))
(make-button "SW" 4 0 '(send-command-to-truck (list 'change-heading 'SW)))
(make-button "S " 4 3 '(send-command-to-truck (list 'change-heading 'S)))
(make-button "SE" 4 6 '(send-command-to-truck (list 'change-heading 'SE)))

(make-button '(case *button-current-status*
	       (slow   " Slow ")
	       (medium "Medium")
	       (fast   " Fast "))
	     6 0
	     '(let ((new-speed (choose-speed)))
	       (when new-speed
		 (setf *current-speed* new-speed)
		 (send-command-to-truck (list 'change-speed *current-speed*))))
	     :status-form '*current-speed*)

(make-button '(case *button-current-status*
	       (arm-1 "Current arm: Arm-1")
	       (arm-2 "Current arm: Arm-2"))
	     2 11
	     '(setf *current-arm* (if (eq *current-arm* 'arm-1)
				     'arm-2
				   'arm-1))
	     :status-form '*current-arm*)

(make-button "Arm-move Folded" 3 11 
	    '(send-command-to-truck (list 'arm-move *current-arm* 'folded)))
(make-button "Arm-move Outside" 4 11 
	    '(send-command-to-truck (list 'arm-move *current-arm* 'outside)))
(make-button "Arm-move to position" 5 11
	    '(let ((pos (choose-position)))
	      (when pos
		(send-command-to-truck (list 'arm-move *current-arm* pos)))))
(make-button "Arm-move Inside" 6 11 
	    '(send-command-to-truck (list 'arm-move *current-arm* 'inside)))
(make-button "Arm-move Bay-1" 7 11
	    '(send-command-to-truck (list 'arm-move *current-arm* 'bay-1)))
(make-button "Arm-move Bay-2" 8 11
	    '(send-command-to-truck (list 'arm-move *current-arm* 'bay-2)))
(make-button "Arm-move Tire-bay" 9 11
	    '(send-command-to-truck (list 'arm-move *current-arm* 'tire-bay)))
(make-button "Arm-move Weapon-bay" 10 11
	    '(send-command-to-truck (list 'arm-move *current-arm* 'weapon-bay)))
(make-button "Arm-move Fuel-tank" 11 11
	     '(send-command-to-truck (list 'arm-move *current-arm* 'fuel-tank)))
(make-button "Truck-move" 13 11
	     '(send-command-to-truck (list 'truck-move)))

(make-button "Arm Grasp"
	     2 35
	     '(send-command-to-truck (list 'arm-grasp *current-arm*)))

(make-button "Arm Ungrasp"
	     3 35
	     '(let ((pos (choose-position)))
	       (when pos
		 (send-command-to-truck (list 'arm-ungrasp *current-arm* pos)))))

(make-button "Arm Pour" 
	     4 35
	     '(let ((pos (choose-position)))
	       (when pos
		 (send-command-to-truck (list 'arm-pour *current-arm* pos)))))

(make-button "Arm Set"
	     6 35
	     '(let ((pos (choose-position-nil)))
	       (when pos
		 (when (eq pos :nil) (setf pos nil))
		 (send-command-to-truck (list 'arm-set *current-arm* pos)))))

(make-button "Arm Read"
	     7 35
	     '(let ((pos (choose-position-nil)))
	       (when pos
		 (when (eq pos :nil) (setf pos nil))
		 (send-command-to-truck (list 'arm-read *current-arm* pos)))))

(make-button "Truck Set"
	     9 35
	     '(let ((thing (choose-truck-thing)))
	       (when thing
		 (send-command-to-truck (list 'truck-set thing)))))

(make-button "Truck Read"
	     10 35
	     '(let ((thing (choose-truck-thing)))
	       (when thing
		 (send-command-to-truck (list 'truck-read thing)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Top-level control
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun start-truckdriver ()
  (when (not *truckdriver-frame*)
    (setf *truckdriver-frame* 
      (clim:make-application-frame 'truckdriver 
				   :pretty-name "Truckdriver"
				   :parent *root-window*
				   :height 800 :width 500)))
  (clim:run-frame-top-level *truckdriver-frame*))

(defun fresh-truckdriver ()
  (setf *truckdriver-frame* nil)
  (start-truckdriver))
