;;;
;;; $LOG$
;;;

(defobject truck (container)
  (material 'truck-stuff)
  (truck-id (make-unique-id 'TRUCK))
  (:slot time-scale :accessor time-scale)
  (:slot arms :accessor arms)
  (:slot bays :accessor bays)
  (:slot fuel-tank :accessor fuel-tank)
  (:slot tire-bay :accessor tire-bay)
  (:slot weapon-bay :accessor weapon-bay)
  (:slot odometer-gauge :accessor odometer-gauge)
  (:slot speed-gauge  :accessor speed-gauge)
  (:slot heading-gauge  :accessor heading-gauge)
  (:slot speed-setting   :accessor speed-setting)
  (:slot heading-setting :accessor heading-setting)
  (:slot status-gauge  :accessor status-gauge)
  (:slot truck-sensor  :accessor truck-sensor)
  (:slot clock-sensor :accessor clock-sensor)
  (:slot base-mpg :accessor base-mpg)
  (:slot base-mph :accessor base-mph)
  (:slot travel-mpg :accessor travel-mpg)
  (:slot travel-mph :accessor travel-mph)
  (:slot built-in-sensors :accessor built-in-sensors)
  (:slot advance-clock :accessor advance-clock)
  (:slot channel :accessor channel)
  (:slot outstanding-commands :accessor outstanding-commands
	 :initform 0))

(defvar *the-truck-table* '())

;;;;;;
;;;
;;; TRUCK PARTS
;;;



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; TRUCK PARTS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;
;;; CARGO BAYS
;;;

(defobject cargo-bay (container)
  (:slot moveable? :initform nil)
  (:slot sensor :accessor sensor))

(defun make-bay (bay-name &key (bigness *default-bay-bigness*)
			       (capacity *default-bay-capacity*)
			       (holdings '())
			       (sensing-method *default-sensing-method*)
			       (sensed-properties *default-sensed-properties*)
			       (sensing-duration *default-sensing-duration*)
			       (sensing-duration-noise *default-sensing-duration-noise*))
  (let* ((new-bay (make-sim-object 'cargo-bay
				   'id bay-name
				   'bigness bigness
				   'capacity capacity
				   'max-positions *max-bay-displayer-size*)))
    (mapcar #'(lambda (obj-form) (put-in new-bay (eval obj-form) t)) holdings)
    (setf (sensor new-bay) (make-bay-sensor new-bay
					    sensing-method
					    sensed-properties
					    sensing-duration
					    sensing-duration-noise))
    new-bay))

(defun make-bay-sensor (bay method properties time time-dur)
  (let ((new-sensor
	 (make-sensor :sensing-scope      #'(lambda (sensor) (holdings bay))
		      :sensing-method     method
		      :sensed-properties  properties
		      :sensing-duration       time
		      :sensing-duration-noise time-dur)))
    (setp new-sensor 'id (intern (format nil "~A-SENSOR" (query bay 'id))))
    new-sensor))

;;;;;;;;;;;;;;;;;;;
;;;
;;; TIRE-BAY
;;;

(defobject tire-bay (container)
  (:slot moveable? :initform nil)
  (:slot sensor :accessor sensor)
  (:slot membership-test 
	 :initform #'(lambda (bay tire) (and (null (holdings bay)) 
					     (typep tire 'tires)))))

(defun make-tire-bay (bay-name &key (capacity *default-bay-capacity*)
				    (holdings '((make-tire)))
				    (sensing-method *default-sensing-method*)
				    (sensed-properties *default-sensed-properties*)
				    (sensing-duration *default-sensing-duration*)
				    (sensing-duration-noise *default-sensing-duration-noise*))
  (let* ((new-bay (make-sim-object 'tire-bay
				   'id bay-name
				   'capacity capacity
				   'max-positions *max-tire-bay-displayer-size*)))
    (mapcar #'(lambda (obj-form) (put-in new-bay (eval obj-form) t)) holdings)
    (setf (sensor new-bay) (make-bay-sensor new-bay
					    sensing-method
					    sensed-properties
					    sensing-duration
					    sensing-duration-noise))
    new-bay))


;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; WEAPON-BAY
;;;

(defobject weapon-bay (container)
  (:slot moveable? :initform nil)
  (:slot sensor :accessor sensor)
  (:slot membership-test 
	 :initform #'(lambda (bay gun) (and (null (holdings bay))
					    (typep gun 'weapon)))))

(defun make-weapon-bay (bay-name &key (capacity *default-bay-capacity*)
				      (holdings '())
				      (sensing-method *default-sensing-method*)
				      (sensed-properties '(kind amount-held))
				      (sensing-duration *default-sensing-duration*)
				      (sensing-duration-noise *default-sensing-duration-noise*))
  (let* ((new-bay (make-sim-object 'weapon-bay
				   'id bay-name
				   'capacity capacity
				   'max-positions *max-weapon-bay-displayer-size*)))
    (mapcar #'(lambda (obj-form) (put-in new-bay (eval obj-form) t)) holdings)
    (setf (sensor new-bay) (make-bay-sensor new-bay
					    sensing-method
					    sensed-properties
					    sensing-duration
					    sensing-duration-noise))
    new-bay))

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; FUEL-TANK
;;;

(defobject fuel-tank (vessel)
  (:slot moveable? :initform nil)
  (:slot sensor :accessor sensor))


(defun make-fuel-tank (tank-name &key (capacity *default-fuel-tank-capacity*)
				      (amount-held 0)
				      (sensing-method *default-sensing-method*)
				      (sensed-properties '(composition 
							   amount-held))
				      (sensing-duration *default-sensing-duration*)
				      (sensing-duration-noise *default-sensing-duration-noise*))
  (let* ((new-tank (make-sim-object 'fuel-tank
				    'amount-held amount-held
				    'composition (if (> amount-held 0)
						     '(gasoline)
						   nil)
				    'capacity capacity
				    'id tank-name)))

    (setf (sensor new-tank) (make-fuel-sensor new-tank
					      sensing-method
					      sensed-properties
					      sensing-duration
					      sensing-duration-noise))
    new-tank))

(defun make-fuel-sensor (tank method properties time time-noise)
  (let ((new-sensor
	 (make-sensor :sensing-scope      #'(lambda (sensor) (list tank))
		      :sensing-method     method
		      :sensed-properties  properties
		      :sensing-duration       time
		      :sensing-duration-noise time-noise)))
    (setp new-sensor 'id (intern (format nil "~A-SENSOR" (query tank 'id))))
    new-sensor))

;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; GAUGES
;;;

(defobject truck-gauge (controller)
  (:slot moveable? :initform nil)
  (:slot sensor :accessor sensor))

;;; GENERIC gauge/sensor constructor

(defun make-gauge (name value method time time-noise)
  (let ((new-gauge (make-sim-object 'truck-gauge
				    'id name
				    'value value)))
    (setf (sensor new-gauge)
      (make-sensor :sensing-scope #'(lambda (sensor) (list new-gauge))
		   :sensing-method method
		   :sensed-properties '(value)
		   :sensing-duration time
		   :sensing-duration-noise time-noise))
    (setp (sensor new-gauge) 'id 
	  (intern (format nil "~A-SENSOR" (query new-gauge 'id))))
    new-gauge))


;;; SPEED GAUGE

(defun make-speed-cont (name &key (sensing-method *default-sensing-method*)
				  (sensing-duration *default-sensing-duration*)
				  (sensing-duration-noise *default-sensing-duration-noise*))
  (make-gauge name 'STOP sensing-method sensing-duration
	      sensing-duration-noise))

;;; HEADING GAUGE

(defun make-heading-cont (name &key (sensing-method *default-sensing-method*)
				    (sensing-duration *default-sensing-duration*)
				    (sensing-duration-noise *default-sensing-duration-noise*))
  (make-gauge name 'N sensing-method sensing-duration
	      sensing-duration-noise))

;;; STATUS GAUGE

(defun make-status-cont (name &key (status 'HAPPY)
				   (sensing-method *default-sensing-method*)
				   (sensing-duration *default-sensing-duration*)
				   (sensing-duration-noise *default-sensing-duration-noise*))
  (make-gauge name status sensing-method sensing-duration
	      sensing-duration-noise))

;;; ODOMETER GAUGE

(defun make-odometer-cont (name &key (mileage 0)
				     (sensing-method *default-sensing-method*)
				     (sensing-duration *default-sensing-duration*)
				     (sensing-duration-noise *default-sensing-duration-noise*))
  (make-gauge name mileage sensing-method sensing-duration
	      sensing-duration-noise))


;;;;;;;;;;;;;;
;;;
;;; TRUCK SENSOR
;;;

(defun make-truck-sensor (name &key (sensing-method *default-sensing-method*)
				    (sensed-properties *default-sensed-properties*)
				    (sensing-duration *default-sensing-duration*)
				    (sensing-duration-noise *default-sensing-duration-noise*))
  (let ((new-sensor (make-sensor :sensing-scope #'my-node-contents
				 :sensing-method sensing-method
				 :sensed-properties sensed-properties
				 :sensing-duration sensing-duration
				 :sensing-duration-noise sensing-duration-noise)))
    (setp new-sensor 'id name)
    new-sensor))

;;;
;;; Truck clock
;;;

(defun clock-of-world (x)
  (declare (ignore x))
  (clock *the-world*))

(defun make-truck-clock (name &key (sensing-duration *default-sensing-duration*)
				   (sensing-duration-noise *default-sensing-duration-noise*)
				   (sensed-properties '(value)))
  (let ((new-sensor (make-sensor :sensing-scope #'clock-of-world
				 :sensing-method t
				 :sensed-properties sensed-properties
				 :sensing-duration sensing-duration
				 :sensing-duration-noise sensing-duration-noise)))
    (setp new-sensor 'id name)
    (setp new-sensor 'sensor-id name)
    (setf (moveable? new-sensor) nil)
    new-sensor))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; TRUCK CONSTRUCTORS
;;;

;;;
;;; MAKE-TRUCK
;;;
;;; The generic truck maker
;;; It is a macro, because it is generally used as a top-level form
;;; constructed by the client, and EVAL'ed by the server to create
;;; a truck.  The macro allows the truck part arguments to remain unquoted,
;;; as per the documentation.
;;;

(defmacro make-truck (&rest args)
  `(make-truck-fn
    ,@(mapcar #'(lambda (arg)
		  (cond
		   ((keywordp arg)  arg)
		   (t 		    `(quote ,arg))))
	      args)))


(defun make-truck-fn (&key (truck-id     (make-unique-id 'TRUCK)) 
			   (time-scale   *default-time-scale*)
			   (mpg          *default-truck-mpg*)
			   (mph          *default-truck-mph*)
			   (bigness      *default-truck-bigness*)
			   (arm1         '(arm-1))
			   (arm2         '(arm-2))
			   (bay1         '(bay-1))
			   (bay2         '(bay-2))
			   (fuel-tank    '(fuel-tank))
			   (tire-bay     '(tire-bay))
			   (weapon-bay   '(weapon-bay))
			   (odometer     '(odometer))
			   (speed        '(speed))
			   (heading      '(heading))
			   (status       '(status))
			   (clock        '(clock))
			   (truck-sensor '(truck-sensor)))

  
  (let* ((new-truck (make-sim-object 'truck
				     'id truck-id
				     'truck-id truck-id
				     'time-scale time-scale
				    'bigness bigness
				    'base-mpg mpg
				    'base-mph mph
				    'arms (list 
					   (apply #'make-arm arm1)
					   (apply #'make-arm arm2))
				    'bays (list
					   (apply #'make-bay bay1)
					   (apply #'make-bay bay2))
				    'fuel-tank (apply #'make-fuel-tank 
						      fuel-tank)
				    'tire-bay (apply #'make-tire-bay 
						     tire-bay)
				    'weapon-bay (apply #'make-weapon-bay 
						       weapon-bay)
				    'odometer-gauge (apply #'make-odometer-cont
							   odometer)
				    'speed-gauge (apply #'make-speed-cont
							speed)
				    'heading-gauge (apply #'make-heading-cont
							  heading)
				    'speed-setting (make-sim-object
						    'controller
						    'id 'speed-setting
						    'value 'STOP)
				    'heading-setting (make-sim-object
						      'controller
						      'id 'speed-setting
						      'value 'N)
				    'status-gauge (apply #'make-status-cont
							 status)
				    'clock-sensor (apply #'make-truck-clock
							 clock)
				    'truck-sensor (apply #'make-truck-sensor
							 truck-sensor))))
    ;; Physically put the truck parts in the truck.
    ;; Only the bays, the fuel-tank, and the arms are going in,
    ;; because they are the only physically accessable things.
    ;; Except for the sensor: It depends on containment within the
    ;; truck to do it's sensing.
    
    (put-in new-truck (tire-bay new-truck) t)
    (put-in new-truck (weapon-bay new-truck) t)
    (put-in new-truck (fuel-tank new-truck) t)
    (mapcar #'(lambda (x) (put-in new-truck x t)) (bays new-truck))
    (mapcar #'(lambda (x) (install-arm x new-truck)) (arms new-truck))
    (put-in new-truck (truck-sensor new-truck) t)
    
    ;; Cache the sensors immediately accessable to the truck for
    ;; lookup later
    
    (setf (built-in-sensors new-truck)
      (nconc (list (truck-sensor new-truck) (clock-sensor new-truck))
	     (mapcar #'sensor (append (arms new-truck) 
				      (bays new-truck)
				      (list (weapon-bay new-truck)
					    (tire-bay new-truck)
					    (fuel-tank new-truck)
					    (speed-gauge new-truck)
					    (status-gauge new-truck)
					    (odometer-gauge new-truck)
					    (heading-gauge new-truck))))))
    new-truck))


;;;;;;;;;;;;;;;;;
;;;
;;; Here are some convenient truck creators
;;;

(defun create-standard-truck ()
  (make-truck :fuel-tank (fuel-tank :amount-held 10)))

(defun create-mutant-truck ()
  (make-truck
   :arm1 (arm-1 :capacity 100)
   :arm2 (arm-2 :capacity 1000)
   :bay1 (bay-1 :capacity 1000)
   :bay2 (bay-2 :capacity 10000)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; This is the function used to create a truck for a client communicating
;;; over a given channel
;;;

(defvar *new-truck*)

(defun create-client-truck (channel)
  (setf *new-truck* nil)
  (handler-case (try-to-create-client-truck channel)
    (error (condition)
      (when *new-truck*
	(take-out (query *new-truck* 'container) *new-truck*))
      (format *terminal-io* "Couldn't create client truck:~%~A~%" condition)
      (send-control channel '(ERROR handshake-error))
      (close-channel channel)
      nil)))

(defun try-to-create-client-truck (channel)
  (let* ((truck-def (or (read-control channel) '(make-truck)))
	 (start-name (read-control channel))
	 (disp-name (read-control channel))
	 (new-truck (eval truck-def))
	 (start-node (find-if #'(lambda (x) (eq start-name (query x 'id)))
			      (nodes (world-map *the-world*)))))
    
    (setf (channel new-truck) channel)
    (if start-node
	(put-in start-node new-truck)
      (put-in (first (nodes (world-map *the-world*))) new-truck))
    (setf *new-truck* new-truck)
    (when disp-name
      (make-displayer new-truck *the-world* (clock *the-world*)
		      disp-name))
    (send-control new-truck (list 'time (actual-time *the-world*)))
    (startup-truck new-truck)
    new-truck))

;;
;; STARTUP-TRUCK
;;
;;

(defun startup-truck (the-truck)
  (setf (advance-clock the-truck) nil))

;;
;; SHUTDOWN-TRUCK
;;
;; Removes truck from the table of usable trucks
;;

(defun shutdown-truck (the-truck)
  (setf *the-truck-table* (delete the-truck *the-truck-table*))
  (format *terminal-io* "~S Leaving simulation~%" the-truck)
  (send-control the-truck '(BYE))
  (terminate-displayer-for-truck the-truck)
  (take-out (query the-truck 'container) the-truck)
  (close-channel (channel the-truck)))


;;;
;;; DESTROY-OBJECT
;;;
;;; The trucks are more durable objects.  When one gets "destroyed", it
;;; simply goes dead.
;;;
;;; This is :AROUND because we don't want CLOS to call the container
;;; method (which would destroy all items in the truck)
;;;

(defmethod destroy-object :around ((self truck))
  (setf (truck-status self) 'dead))

  
;;; OPERATORS ;;;

;;
;; TRUCK CONTROLLER OPERATIONS
;;
;;  The values of various truck gauges can be queried and setf
;; without accessing the controllers themselves.
;;


(defun truck-heading  (truck)
  (value (heading-gauge  truck)))

(defun truck-speed (truck)
  (value (speed-gauge  truck)))

(defun truck-heading-setting  (truck)
  (value (heading-setting  truck)))

(defun truck-speed-setting (truck)
  (value (speed-setting  truck)))

(defun truck-status (truck)
  (value (status-gauge  truck)))

(defun truck-odometer (truck)        
  (value (odometer-gauge  truck)))

(defun truck-fuel (truck)
  (space-full (fuel-tank truck)))

(defun truck-tires (self)
  (car (holdings (tire-bay self))))


;;;******************

(defun (setf truck-heading)  (direction truck)
  (enforce #'map-direction? direction t)
  (setp (heading-gauge  truck) 'value direction))

(defun (setf truck-speed) (value truck)
  (enforce #'speed? value t)
  (setp (speed-gauge  truck) 'value value)
  (update-speed-parameters truck))

(defun (setf truck-heading-setting)  (direction truck)
  (enforce #'map-direction? direction t)
  (setp (heading-setting truck) 'value direction))

(defun (setf truck-speed-setting) (value truck)
  (enforce #'speed? value t)
  (setp (speed-setting truck) 'value value))


(defun (setf truck-status) (value truck)
  (enforce #'status? value t)
  (setp (status-gauge  truck) 'value value))


(defun (setf truck-odometer) (value truck)
  (setp (odometer-gauge  truck) 'value value))

(defun update-speed-parameters (self)
  (case (truck-speed self)
	((stop)
	 (setf (travel-mpg self) 0)
	 (setf (travel-mph self) 0))

	((slow)
	 (setf (travel-mpg self) 
	       (percent-adjust *default-slow-mpg-adjust* (base-mpg self)))
	 (setf (travel-mph self)
	       (percent-adjust *default-slow-mph-adjust* (base-mph self))))
	
	((medium)
	 (setf (travel-mpg self) (base-mpg self))
	 (setf (travel-mph self) (base-mph self)))
	
	((fast)
	 (setf (travel-mpg self) 
	   (percent-adjust *default-fast-mpg-adjust* (base-mpg self)))
	 (setf (travel-mph self) 
	   (percent-adjust *default-fast-mph-adjust* (base-mph self)))))
  
  (when (truck-tires self)
    (setf (travel-mph self) 
      (percent-adjust (mph-adjust (truck-tires self)) (travel-mph self)))
    (setf (travel-mpg self) 
      (percent-adjust (mpg-adjust (truck-tires self)) (travel-mpg self)))))


;;;
;;; CHEAT FUNCTIONS
;;;

(defun cheat-fill-fuel-tank ()
  (mapc #'(lambda (truck) (pour-in (fuel-tank truck) 'gasoline
				   (query (fuel-tank truck) 'capacity)))
	*the-truck-table*))

(defun cheat-reset-status ()
  (mapc #'(lambda (truck) (setf (truck-status truck) 'happy))
	*the-truck-table*))


