

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; MATERIAL/SENSING system
;;;

(defvar *transparency-table* nil)

;;;
;;; DEF-SENSING-METHOD
;;;
;;; Defines which materials a sensing method may or may not see through.
;;;
;;; :TRANSPARENT  is a list of materials that the method may see through.
;;;                 All other materials are opaque to that sensing method
;;; :OPAQUE       is a list of materials that the sensor may not see through.
;;;                 All other materials are transparent to  method.
;;;
;;; AT most one of :TRANSPARENT or :OPAQUE may be specified.
;;; If neither are specified, it is assumed that the sensing method
;;;  cannot see through any material.
;;;
;;; Examples:
;;;
;;; (def-sensing-method 'x-ray :opaque '(lead))
;;;    an X-RAY sensor can see through anything but lead
;;;
;;; (def-sensing-method 'visible-light 
;;;                     :transparent '(glass plexiglass plastic-wrap))
;;;    a Visible-light sensor (like a video-camera) can see through
;;;    glass, plexiglass, and plastic-wrap, but nothing else.
;;;
;;; (def-sensing-method 'miracle-method :opaque '())
;;;    The Miracle-method can see through anything.
;;;
;;; (def-sensing-method 'really-bad-method :transparent '())
;;;    The really-bad-method cannot see through anything
;;;



(defun def-sensing-method (method &key (transparent nil t-spec)
				       (opaque      nil o-spec))
  (cond
   ((and t-spec o-spec)
    (error "Only one of :transparent or :opaque may be specified"))
   
   ((and t-spec (listp transparent))
    (push (cons method (cons nil transparent)) *transparency-table*))
   
   (t-spec
    (error ":TRANSPARENCY argument must be a true list"))
   
   ((and o-spec (listp opaque))
    (push (cons method (cons t opaque)) *transparency-table*))
    
   (o-spec
    (error ":OPAQUE argument must be a true list"))
   
   (t
    (push (cons method (cons nil '())) *transparency-table*))))


;;; TRANSPARENT-TO?
;;;
;;; Determines if a container is transparent to a given method(s) of sensing
;;;

(defun transparent-to? (obj method)
  (cond
   ((not (typep obj 'container)) nil)

   (t
    (material-transparent-to-method? (query obj 'material) method))))


(defun material-transparent-to-method? (material method)
  (cond
   ((typep method 'list)
    (some #'(lambda (meth) (material-transparent-to-method? material meth)) 
	  method))

   ;; the sensing method T makes everything transparent
   ((eq method t)
    t)

   ;; The method NIL penetrates nothing
   ((null method)
    nil)
   
   (t
    (let ((trans (cdr (assoc method *transparency-table*))))
      (or
       ;; Container transparent to everything except...
       (and (car trans) (not (member material (cdr trans))))
       ;; Container transparent to nothing except...
       (and (null (car trans)) (member material (cdr trans))))))))

  
