
(defobject sim-object ()
  (id        (make-unique-id 'obj))		; Every object should have unique id
  (kind      'sim-object		; KIND = CLASS of object
	     ;;	     (:set-properties #'myself (id (function-value make-kind-id 
	     ;;						   ((arg 0) kind))))
	     )
  (container nil)			; All sim objects can be inside container
  (position  nil)			;  Position inside an object
  (movement  nil)			; gets SETPed when anything higher up
					;  in the containment hierarchy gets
					;  moved.
  (set       nil			; When SETPed, SET-OBJECT is invoked
					; (except at initialization time)
	     (:effect (let ((init-set t))
			#'(lambda (obj old new)
			    (if init-set
				(setf init-set nil)
			      (apply #'set-object obj nil (insure-list new)))))))
  (:slot moveable?			; Can the object be picked up?
	 :accessor moveable?
	 :initform t)
  (:slot set-duration 
	 :accessor set-duration
	 :initform 0)
  (:slot set-duration-noise 
	 :accessor set-duration-noise
	 :initform 0))

					   
(defun make-kind-id (kind)
  (make-unique-id kind))

;;; MAKE-SIM-OBJECT
;;;
;;; Creates a sim-object of the specified type, and assigns a KIND and ID to
;;; it.  These can be overridden by specifying them as initprops to
;;; MAKE-SIM-OBJECT.
;;; 

(defun make-sim-object (class &rest initprops)
  (let ((obj   (apply #'make-object class 
		      'kind class
		      initprops)))
    obj))


(defun id (self)
  (query self 'id))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; GENERIC OBJECT OPERATORS
;;;

;;;
;;; DESTROY-OBJECT
;;;
;;; Responsible for removing objects from the simulation.
;;; The default action is to simply take the object out of it's container,
;;;   and throw it away.
;;; Other actions may be specified for more specific classes by
;;;  adding :before, :after, :around or new primary methods.

(defgeneric destroy-object (self))

(defmethod destroy-object ((self t))
  t)

;;;
;;; SET-OBJECT
;;;
;;; All sim-objects may be SET by another object.
;;; The object being set, and the object doing the setting, and any
;;; other arguments specific to the SET are passed.
;;; The semantics/arguments of SET are particular to each object class.
;;;

;;;
;;; Whenver an arm wishes to set an object, it calls INIT-SET-OBJECT.
;;; It spawns a process that lasts as long as the parameters SET-DURATION
;;; and SET-DURATION-NOISE specify.  At the end of the process,
;;; the set operation is actually done via SET-OBJECT.
;;;

(defun init-set-object (object setter &rest set-values)
  (when (typep object 'sim-object)
    (let* ((start-time (actual-time))
	   (end-time (add-noise (add-times start-time (set-duration object))
				(set-duration-noise object)
				:lo start-time)))
      
      (start-process
       nil
       #'(lambda (tok time why)
	   (when (compare-times time '= start-time)
	     (stop-process tok :at end-time))
	   (when (compare-times time '>= end-time)
	     (apply #'set-object object setter set-values)))))))
   



(defgeneric set-object (object setter &rest set-values))

(defmethod set-object ((object t) setter &rest set-values)
  t)


;;;
;;; READ-OBJECT
;;;
;;; All sim-objects may be READ.  When read, the object returns some
;;; information whose interpretation is specific to the class of the object.
;;; It is the intent of Truckworld that the only objects which return
;;; any information are sensor objects.
;;;

(defgeneric read-object (object))

(defmethod read-object ((object t))
  nil)

;;;
;;; BITMAP-PATHNAME
;;;
;;; This provides support for the displayer:
;;; Each object should be able to specify a pathname for it's icon bitmap.
;;; This way, objects can decide to display themselves differently
;;; depending on their state.
;;; The default way to select an icon is just to use the class name
;;; as the bitmap filename.
;;;

(defmethod bitmap-pathname ((object t))
  (make-bitmap-pathname (query object 'kind)))

;;;
;;; MAKE-BITMAP-PATHNAME
;;;
;;; A utility function to make bitmap pathnames out of atoms:
;;; Given either a string or symbol X, the pathname returned will default
;;; to file "X.xbm" in the *truckworld-bitmap-directory*.
;;; A different desired pathname may be specified as part of the NAME argument.
;;; If the pathname given as part of the NAME is a relative pathname,
;;; it will be relative to *truckworld-bitmap-directory*.
;;;

(defun make-bitmap-pathname (name)
  (let ((string-name (if (stringp name) 
			 name
		       (string-downcase (princ-to-string name)))))
    (make-pathname :type "xbm"
		   :defaults 
		   (merge-pathnames string-name
				    *truckworld-bitmap-directory*))))


(defmethod print-object ((self sim-object) stream)
  (format stream "#{~a:~a}" (query self 'kind) (id self)))

(defun class? (self class-name)
  (typep self class-name))


