;;; SENSOR.LISP
;;;

;;; SENSOR
;;;  A sensor may be parameterized by 3 values:
;;;    VISIBLES : a function, when passed the sensor object, returns
;;;               the objects which may be sensed by the sensor.
;;;    METHOD   : A symbol(s) that describes the method used to sense
;;;               (e.g. visible, x-ray, sonar, radiation). Used for
;;;               determining if a sensor may see inside a container.
;;;               If NIL, all containers are opaque to sensor (useful for
;;;                 a scanner that scans only for top-level objects)
;;;               If T, all containers are transparent to sensor.
;;;               METHOD may be a list of such symbols.  If an object is
;;;                transparent to any of the methods in the list, then it is
;;;                transparent to the sensor.
;;;    SENSE-INFO : A list of properties which the sensor will sense
;;;                 about each object.

(defobject sensor (thingoid)
  (sensor-id (make-sensor-id))
  (:slot set-duration :initform *default-sensing-duration*)
  (:slot set-duration-noise :initform *default-sensing-duration-noise*)
  (:slot sense-method
	 :accessor sense-method 
	 :initform *default-sensing-method*)
  (:slot sense-info
	 :accessor sense-info
	 :initform *default-sensed-properties*)
  (:slot visibles    :accessor visibles    :initform *default-sensing-scope*)
  (:slot descriptors :accessor descriptors :initform nil))


;;;
;;; DEFSENSOR
;;;
;;; Allows the user to create new types of sensors.  The new sensor type
;;; may be customized through the keyword arguments.  If a particular
;;; keyword argument is not supplied, then that value is inherited
;;; from class SENSOR.
;;;

(defmacro defsensor (type 
		     &key extra-properties
			  (parent-sensor 'sensor)
			  (sensing-method nil smp)
			  (sensed-properties nil spp)
			  (sensing-scope nil ssp)
			  (sensing-duration nil sdp)
			  (sensing-duration-noise nil sdnp))
  `(defobject ,type (,parent-sensor)
     ,@extra-properties
     ,@(create-initforms '(sense-method
			   sense-info
			   visibles
			   set-duration
			   set-duration-noise)
			 (list sensing-method sensed-properties sensing-scope
			       sensing-duration sensing-duration-noise)
			 (list smp spp ssp sdp sdnp))))

(defun create-initforms (slot-names initforms supplied)
  (delete nil
	  (mapcar #'(lambda (n i s)
		      (when s
			`(:slot ,n :initform ,i)))
		  slot-names 
		  initforms 
		  supplied)))

(defun make-sensor-id (&optional (prefix 'sensor))
  (make-unique-id prefix))
  
;;; MAKE-SENSOR
;;;


(defun make-sensor (&key (sensor-id (make-sensor-id))
			 (sensing-scope *default-sensing-scope*)
			 (sensing-method *default-sensing-method*)
			 (sensed-properties *default-sensed-properties*)
			 (sensing-duration *default-sensing-duration*)
			 (sensing-duration-noise *default-sensing-duration-noise*))
  (let ((new-sensor (make-sim-object 'sensor
				     'sensor-id sensor-id
				     'sense-method sensing-method
				     'sense-info   sensed-properties
				     'set-duration sensing-duration
				     'set-duration-noise sensing-duration-noise
				     'visibles  sensing-scope)))
    new-sensor))


  

;;; READ-OBJECT
;;;
;;;   Returns the list of descriptors accumulated since the last READ-OBJECT
;;;

(defmethod read-object ((sensor sensor))
  (prog1
      (descriptors sensor)
    (setf (descriptors sensor) nil)))

;;;
;;; SET-OBJECT
;;;
;;; Here is how a sensor reacts to being set:
;;;   SENSE is called to actually perform the sensing.
;;;

(defmethod set-object ((object sensor) setter &rest set-values)
  (sense object))


;;;
;;; SENSE
;;;
;;; Collect all objects visible to sensor, and build descriptors for each
;;;

(defun sense (sensor)
  (let* ((top     (funcall (visibles sensor) sensor))
	 (visible (gather-visibles top (sense-method sensor))))
    (setf (descriptors sensor)
	  (nconc (descriptors sensor)
		 (build-descriptors visible (sense-info sensor))))))

;;; GATHER-VISIBLES
;;;
;;; Collects all objects visible to sensor (starting with those in obj-list)
;;; using the given sensing method.  Objects are collected into a tree
;;; structure such that:
;;;   car of a list = root of subtree = an object
;;;   cadr of list  = children of root = list of trees of visible objects
;;;     contained within root object
;;; Therefore, the returned tree mirrors the containment hierarchy of
;;; all visible objects.
;;;
;;; GATHER-VISIBLES returns a list of trees made out of each object in obj-list
;;;

(defun gather-visibles (obj-list method)
  (let ((obj-list (insure-list obj-list)))
    (mapcar #'(lambda (obj)
		(list obj (if (transparent-to? obj method)
			      (gather-visibles (holdings obj) method)
			    nil)))
	    obj-list)))

	
      
;;; BUILD-DESCRIPTORS
;;;
;;; Given the list returned from GATHER-VISIBLES, and a list of properties
;;; to sense, Builds a descriptor for each item in the tree which reports
;;; the given properties of each item.
;;;
;;; The property POSITION is treated specially: 
;;; If an object is at the
;;;   root of one of the trees in the list, its own position is reported.
;;; But if an object is some non-root node within a tree, the position
;;;   of the root object is reported.
;;;
;;; This means that all objects contained in object X are sensed as
;;; being at X's position.
;;;

(defun build-descriptors (visible-tree props)
  (build-descriptor-from-list visible-tree props))

(defun build-descriptor-from-tree (visible-tree props &optional pos)
  (let ((top (build-descriptor (car visible-tree) props pos))
	(lower (build-descriptor-from-list (cadr visible-tree) props 
					   (append pos 
						   (list (query (car visible-tree)
								'position))))))
    (if top
	(cons top lower)
      lower)))
    
(defun build-descriptor-from-list (visible-tree-list props
				   &optional pos)
  (apply #'append
	 (mapcar #'(lambda (tree)
		     (build-descriptor-from-tree tree props pos))
		 visible-tree-list)))


(defun build-descriptor (obj props &optional pos)
  (remove nil
	  (mapcar
	   #'(lambda (prop)
	       (build-single-descriptor-property obj prop pos))
	   props)))

(defun build-single-descriptor-property (obj prop &optional pos)
  (let ((prop-name (if (symbolp prop) prop (first prop))))
    (multiple-value-bind (prop-val exists)
	(query obj prop-name)
      
      (cond
       ((eq prop-name 'position)
	(append (list prop-name) pos (list prop-val)))
      
       (exists
	(list prop-name
	      (if (symbolp prop)
		  (distort-property prop-name prop-val)
		(distort-property prop-name prop-val :noise (second prop)))))
       (t nil)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; PROPERTY DISTORTION SYSTEM
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defparameter *distortion-table* (make-hash-table))

;;;
;;; DEFPROPERTY <property-name> &key copy enumeration range (distortion 0)
;;;

(defun defproperty (name &key copy enumeration range (distortion 0))
  (let (entry)
    (cond
     (copy
      (setf entry (gethash copy *distortion-table*))
      (if entry
	  (setf (gethash name *distortion-table*) entry)
	(cerror "Ignore" "Distortion entry for ~S not defined" copy)))
     
     (enumeration
      (setf (gethash name *distortion-table*) 
	(list :enumeration (length enumeration) enumeration distortion)))
     
     (range
      (setf (gethash name *distortion-table*) 
	(list :range (1+ (- (second range) (first range))) range distortion)))
     
     (t
      (cerror "Ignore and continue" "Not a recognized property qualifier")))))


;;; CHOOSE-RANDOM-PROPERTY-VALUE
;;;
;;; This just chooses a random property value, uniformily distributed,
;;; from among those specified by DEFPROPERTY.
;;;

(defun choose-random-property-value (pname)
  (let* ((entry (gethash pname *distortion-table*))
	 (range (second entry))
	 pos)
    
    (cond

     ;; Choose an enumeration value
     ((eq (first entry) :enumeration)
      (setf pos (random range))
      (nth pos (third entry)))
     
     ;; Distort a numeric range
     
     ((eq (first entry) :range)
      (setf pos (+ (first (third entry))
		   (random range)))
      ;; If quantization is not 0, round to nearest quantum
      (if (not (eq (third entry) 0))
	  (setf pos (* (third (third entry))
		       (round (/ pos (third (third entry)))))))
      pos)
     
     (t nil))))


;;;
;;; DISTORT-PROPERTY
;;;
;;; If noise is to be added to a sensed property, here's how to do it.
;;;
;;; DISTORT-PROPERTY takes a property-name, a property-value, and an optional
;;; prob-cond for noise.  If no noise parameter is given, the default
;;; noise as defined by DEFPROPERTY is used.  The noise level is added
;;; to the value to produce the sensed value.
;;;


(defun distort-property (pname pval &key noise)
  (let* ((entry (gethash pname *distortion-table*))
	 (range (second entry))
	 (distortion (if noise noise (fourth entry)))
	 pos)
    
    (cond
     ;; Distort an enumeration value

     ((null entry)
      pval)
     ((eq (first entry) :enumeration)
      (setf pos (position pval (third entry)))
      (cond
       ((not (integerp pos))
	(values pval))
       (t
	(setf pos (round (add-noise pos distortion :lo 0 :hi (1- range))))
	(nth pos (third entry)))))
     
     ;; Distort a numeric range
     
     ((and (eq (first entry) :range) (numberp pval))
      (setf pos (add-noise pval distortion 
			   :lo (first (third entry))
			   :hi (second (third entry))))
      ;; If quantization is not 0, round to nearest quantum
      (if (not (eq (third entry) 0))
	  (setf pos (* (third (third entry))
		       (round (/ pos (third (third entry)))))))
      pos)
     
     (t pval))))

