
;;;
;;; A recording sensor is one that senses and stores the result until
;;; the next READ on the sensor.  It's sensing is triggered by changes
;;; in world state
;;;

;;; 
;;;  Recording sensors need (in addition to normal sensor info):
;;;
;;;  - Function that describes the object(s) whose state change will
;;;    trigger the sensor
;;;    NOTE: Right now, only 1 such object is allowed.
;;;  - The property(ies) in above object that triggers the sensor
;;;
;;;  Each activated recording sensor uses a process to watch the objects
;;;

(defobject recording-sensor (sensor)
  (power 'off)
  (:slot watched-objects     :accessor watched-objects)
  (:slot current-watched     :accessor current-watched)
  (:slot watched-properties  :accessor watched-properties)
  (:slot watching-process    :accessor watching-process :initform nil)
  (:slot set-duration	     :initform 0) ; SET just turns the sensor off/on
  (:slot set-duration-noise  :initform 0)) ; No reason to drag that out.
	


;;;
;;; DEFRSENSOR
;;;
;;; Allows the user to create new types of recording sensors.  
;;; The new sensor type
;;; may be customized through the keyword arguments.  If a particular
;;; keyword argument is not supplied, then that value is inherited
;;; from class RECORDING-SENSOR.
;;;

(defmacro defrsensor (type &key extra-properties
				(parent-sensor 'recording-sensor)
				(sensing-method nil smp)
				(sensed-properties nil spp)
				(sensing-scope nil ssp)
				(watched-objects nil wop)
				(watched-properties nil wpp)
				(sensing-duration nil sdp)
				(sensing-duration-noise nil sdnp))
  `(defobject ,type (,parent-sensor)
     ,@extra-properties
     ,@(create-initforms '(sense-method
			   sense-info
			   visibles
			   watched-objects
			   watched-properties
			   set-duration
			   set-duration-noise)
			 (list sensing-method sensed-properties sensing-scope
			       watched-objects watched-properties
			       sensing-duration sensing-duration-noise)
			 (list smp spp ssp wop wpp sdp sdnp))))


;;;
;;; MAKE-RECORDING-SENSOR
;;;

(defun make-recording-sensor (&key (sensor-id (make-sensor-id))
				(sensing-scope *default-sensing-scope*)
				(sensing-method *default-sensing-method*)
				(sensed-properties *default-sensed-properties*)
				(sensing-duration 0)
				(sensing-duration-noise 0)
				(watched-objects *default-watched-objects*)
				(watched-properties *default-watched-properties*))
  (let ((new-sensor (make-sim-object 'recording-sensor
				     'sensor-id sensor-id
				     'sense-method sensing-method
				     'sense-info   sensed-properties
				     'watched-objects watched-objects
				     'watched-properties watched-properties
				     'set-duration sensing-duration
				     'set-duration-noise sensing-duration-noise
				     'visibles  sensing-scope)))
    new-sensor))




;;;
;;; DESTROY-OBJECT
;;;
;;; When an recording sensor is destroyed, it must stop generating reports
;;;

(defmethod destroy-object :before ((sensor recording-sensor))
  (stop-process (watching-process sensor)))
  
;;;
;;; When an recording sensor is SET on, or toggled to on,
;;; it begins sensing.
;;;

(defmethod set-object ((sensor recording-sensor) (setter t) &rest args)
  (cond
   ;; If sensor is off, and set to ON, or no args, turn sensor on
   ((and (not (watching-process sensor)) 
	 (or (eq (first args) 'ON)
	     (null (first args))))
    (setp sensor 'power 'on)
    (setf (watching-process sensor) (start-monitoring-sensor-process sensor)))
   
   ;; If sensor is on, and set to OFF, or no args, turn sensor off
   ((and (watching-process sensor)
	 (or (eq (first args) 'off)
	     (null (first args))))
    (setp sensor 'power 'off)
    (stop-process (watching-process sensor))
    (setf (watching-process sensor) nil))))


(defun start-monitoring-sensor-process (sensor)
  (let ((start (actual-time)))
    (start-process
     nil
     #'(lambda (tok time why)
	 (case why
	   (ADVANCE
	    (cond
	     
	     ;; On startup, link to all conditionals
	     ((compare-times start '= time)
	      ;; Make a condition that gets tripped when the sensor is moved
	      (add-condition tok sensor 'movement 'movement-condition)
	      ;; Nullify the focus of the sensor
	      (setf (current-watched sensor) nil)
	      ;; Make all conditions that get tripped when the watched
	      ;; object changes state: note: because we set current-watched
	      ;; to nil, make-all... will automatically set up necessary
	      ;; maintainance conditions.
	      (make-all-monitored-sensor-conditions tok sensor))))
	   
	   ;; Something about the world has changed: respond to it
	   
	   (CONDITION
	    (respond-to-monitor-sensor-condition tok sensor (name why))))))))

(defun make-all-monitored-sensor-conditions (tok sensor)
  (let ((watchedobj (take-first (funcall (watched-objects sensor) sensor))))
    
    ;; If the object the sensor is watching doesn't match
    ;; the object it should be watching, then remove all conditionals
    ;; to the former object, and link conditionals to the latter.
    
    (when (and (typep watchedobj 'sim-object)
	       (not (eq watchedobj (current-watched sensor))))
      (setf (current-watched sensor) watchedobj)
      (mapc #'(lambda (prop)
		(delete-condition tok prop)
		(add-condition tok watchedobj prop prop))
	    (watched-properties sensor)))))

(defun respond-to-monitor-sensor-condition (tok sensor name)
  (cond

   ;; The sensor has moved: make sure the object it should be watching
   ;; is still the object it is watching.

   ((eq name 'MOVEMENT-CONDITION)
    (make-all-monitored-sensor-conditions tok sensor))
   
   ;; If condition is for one of the sensor-triggering properties, make a
   ;; sensor report
   
   ((member name (watched-properties sensor))
    (sense sensor)
    (perform-sensor-specific-action sensor))
   
   (t
    (format t "WARNING: Active sensor process got an unknown signal~%"))))

;;; A recording sensor does nothing after it senses something: it simply
;;; stores the resulting descriptors.

(defmethod perform-sensor-specific-action ((sensor recording-sensor))
  nil)

