
;;; Randomize the seed when this file is loaded

(make-random-state t)

;;;  prob-cond = (PROB-COND
;;;	       (<cond> <prob-cond>)
;;;	       (<cond> <prob-cond>)
;;;	       .
;;;	       .
;;;	       .
;;;	       ) | <prob-dist> | <mean-variance> | <val>
;;;
;;;  cond = (<comparator> (<object> <property>) <value>) | T
;;;
;;;  object = <arg-i> | (<function> <arg-i>)
;;;
;;;  prob-dist = (PROB-DIST
;;;	       (<prob> <val>)
;;;	       (<prob> <val>)
;;;	       .
;;;	       .
;;;	       .
;;;	       )
;;;
;;;  mean-variance = (DIST-MEAN-VAR <mean> <var>)


;;;
;;; probability = (PROB-COND <cond-prob-body> | 
;;;               (PROB-DIST <prob-dist-body> |
;;;               <val>
;;;

(defun process-probability (form &rest args)
  (cond
   
   ((not (listp form))	 (process-val form args))
   
   ((eq (car form) 'PROB-COND)
    (process-conditional-probability (cdr form) args))
   
   ((eq (car form) 'PROB-DIST)
    (process-probability-distribution (cdr form) args))
   
   (t
    (process-val form args))))

;;;
;;; prob-dist-body = (<prob> <probability>)
;;;	             (<prob> <probability>)
;;;	             .
;;;	             .
;;;	             .
;;;	             
;;; prob = a number between 0 and 1
;;;

(defun process-probability-distribution (form args)
  (do* ((possible form (cdr possible))
	(cum-prob (first (first possible)) 
		  (+ cum-prob (if (first (first possible))
				  (first (first possible))
				0)))
	(r (/ (random 10000) 10000.0)))
      ((or (null possible) (< r cum-prob))
       (if possible
	   (process-probability (second (first possible)) args)
	 (progn
	   (cerror "Return NIL" "Probabilities in ~S don't sum to 1" form)
	   nil)))))


;;;
;;;  cond-prob-body = (<cond> <probability>)
;;;	              (<cond> <probability>)
;;;	              .
;;;	              .
;;;	              .
;;;
;;;

(defun process-conditional-probability (form args)
  (do ((rest-body form (cdr rest-body)))
      ((or (null rest-body) 
	   (process-condition (first (first rest-body)) args))
       (when rest-body
	 (apply #'process-probability (second (first rest-body)) args)))))

;;;
;;;  cond = (<comparator> (<object> <property>) <value>) | T
;;;

(defun process-condition (form args)
  (cond
   ((eq form t) t)
   
   (t
    (let ((comparator (eval (first form)))
	  (object (process-object (first (second form)) args))
	  (property (second (second form)))
	  (value (third form)))
      (funcall comparator (query object property) value)))))

;;;
;;;  object = <arg> | (<function> <object>)
;;;

(defun process-object (form args)
  (cond
   ((eq (first form) 'ARG)
    (nth (second form) args))
   
   (t
    (funcall (symbol-function (first form)) 
	     (process-object (second form) args)))))

;;;
;;; mean-variance-body = mean variance
;;;

(let ((second-normal nil))

  (defun process-normal-distribution (form args)
    
    (if second-normal
	(prog1
	    (+ (first form) (* second-normal (sqrt (second form))))
	  (setf second-normal nil))
      
      (do* ((mean (first form))
	    (deviation (sqrt (second form)))
	    (fac)
	    (u1 (- (random 2.0) 1.0) (- (random 2.0) 1.0) )
	    (u2 (- (random 2.0) 1.0) (- (random 2.0) 1.0))
	    (r (+ (* u1 u1) (* u2 u2)) (+ (* u1 u1) (* u2 u2))))
	  ((and (< r 1.0) (> r 0.0))
	   
	   (setf fac (sqrt (* -2.0 (/ (log r) r))))
	   
	   (setf second-normal (* u1 fac))
	   (+ mean (* u2 fac deviation))))))
	
      
  )


;;; val = (PROB-VALUES <valu>*) | <valu>
;;;
;;; valu = <value> | 
;;;        (DIST-MEAN-VAR <mean-variance-body> |
;;;        (FUNCTION-VALUE <func> | NIL (<object> <property)) |
;;;        (RANDOM-PROP-VALUE <property>)

(defun process-val (form args)
  (cond
   ((not (listp form)) (process-valu form args))

   ((eq (first form) 'PROB-VALUES)
    (mapcar #'(lambda (x) (process-valu x args)) (cdr form)))
   (t (process-valu form args))))

(defun process-valu (form args)
  (cond
   ((not (listp form)) form)
    
   ((eq (car form) 'DIST-MEAN-VAR)
    (process-normal-distribution (cdr form) args))
   
   ((eq (car form) 'FUNCTION-VALUE)
    (process-function-value-body (cdr form) args))
   
   ((eq (car form) 'RANDOM-PROP-VALUE)
    (process-random-prop-value (cdr form) args))
   
   (t form)))

(defun process-function-value-body (form args)
  (if (first form)
      (funcall (symbol-function (first form))
	       (query (process-object (first (second form)) args)
		      (second (second form))))
    (query (process-object (first (second form)) args)
	   (second (second form)))))

(defun process-random-prop-value (form args)
  (choose-random-property-value (first form)))
	     
;;;
;;; ADD-NOISE
;;;

(defun add-noise (signal noise &key lo hi args)
  (let ((output (+ signal (apply #'process-probability noise args))))
    (when lo (setf output (max lo output)))
    (when hi (setf output (min hi output)))
    output))

;;;
;;; PERCENT-NOISE
;;;
;;; Interprets the NOISE parameter as a prob-cond, when evaluated, gives
;;; a percentage adjustment to make to the signal.  100 = no noise
;;;

(defun percent-noise (signal noise &key lo hi args)
  (let ((output (/ (* signal (apply #'process-probability noise args)) 100)))
    (when lo (setf output (max lo output)))
    (when hi (setf output (min hi output)))
    output))
