
;;;
;;; BROADCAST-MESSAGE
;;;
;;; If an object wishes to send a message (make a noise, send out
;;; a burst of radiation, etc), it specifies the medium by which
;;; the message will travel (visibile-light, sound, radiation, etc)
;;; and the message to be sent.
;;;

(defgeneric broadcast-message (originator medium prop val))

(defmethod broadcast-message ((originator t) medium prop val)
  nil)

(defmethod broadcast-message ((originator sim-object) medium prop val)
  (propagate-message-across (my-container-contents originator) medium prop val)
  (propagate-message-outside (my-container originator) medium prop val))

;;;
;;; RECEIVE-MESSAGE
;;;
;;; When a message reaches an object, RECEIVE-MESSAGE gets invoked
;;; for the receiving object.  All it does is set the object's property
;;; to the given val.
;;;

(defun receive-message (destination medium prop val)
  (setp destination prop val))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Private functions
;;;

;;; Here's how messages propagate from object to object:
;;; If a message is going INSIDE a container, it gets sent ACROSS to
;;; all of the container's objects.  Likewise, if a message is
;;; going OUTSIDE an object, it gets sent ACROSS to the object's neighbors.
;;; When a message gets sent ACROSS to an object, it gets sent
;;; INSIDE the object.
;;; When the message is broadcast from the orignating object,
;;; it gets sent OUTSIDE the originating object's container, 
;;; and ACROSS its neighbors.
;;; Messages are received by an object when the message tries to go INSIDE
;;; or OUTSIDE the object.

;;; This scheme makes sure no object in the containment hierarchy hears
;;; the same message more than once.
;;; 

;;; 
;;;
;;; PROPAGATE-MESSAGE-ACROSS
;;;
;;; Given a list of objects (which should be all neighbors of each other),
;;; will propagate the message into each one of them.
;;;

(defun propagate-message-across (dests medium prop val)
  (mapc #'(lambda (x) (propagate-message-inside x medium prop val))
	dests))

;;;
;;; PROPAGATE-MESSAGE-INSIDE
;;;
;;; Will deliver the message to an object, and if the object is a container,
;;; and container is transparent to medium of message,
;;; will propagate the message across all container's contents.
;;;

(defun propagate-message-inside (dest medium prop val)
  (when (typep dest 'sim-object)
    (receive-message dest medium prop val)
    (when (and (typep dest 'container)
	       (transparent-to? dest medium))
      (propagate-message-across (holdings dest) medium prop val))))

;;;
;;; PROPAGATE-MESSAGE-OUTSIDE
;;;
;;; Will deliver the message to an object,
;;; and container is transparent to medium of message,
;;; will propagate the message across all container's neighbors,
;;; and to the container of the container.
;;;

(defun propagate-message-outside (dest medium prop val)
  (when (typep dest 'sim-object)
    (receive-message dest medium prop val)
    (when (transparent-to? dest medium)
      (propagate-message-across (my-container-neighbors dest) medium prop val)
      (propagate-message-outside (my-container dest) medium prop val))))




