(defobject map-link (container)
  (fuel-drag    *default-link-fuel-drag*
		(:set-properties #'road-signs
				 (fuel-drag 
				  (function-value nil ((arg 0) fuel-drag)))))
  (speed-drag   *default-link-speed-drag*
		(:set-properties #'road-signs
				 (speed-drag 
				  (function-value nil ((arg 0) speed-drag)))))
  (max-load    *default-link-max-load*
	       (:set-properties #'road-signs
				(max-load
				 (function-value nil ((arg 0) max-load)))))
  (pavement  'asphalt
	     (:set-properties #'road-signs
			      (pavement
			       (function-value nil ((arg 0) pavement)))))
  (condition 'good
	     (:set-properties #'road-signs
			      (condition
			       (function-value nil ((arg 0) condition)))))
  (bumpy     'nil
	     (:set-properties #'road-signs
			      (bumpy 
			       (function-value nil ((arg 0) bumpy)))))
  (length    *default-link-length*
	     (:set-properties #'road-signs
			      (length
			       (function-value nil ((arg 0) length)))))
  (:slot trucks-on-road :accessor trucks-on-road 
	 :initform nil)
  (:slot direction :accessor direction
	 :initform nil)
  (:slot start-node :accessor start-node
	 :initform nil)
  (:slot start-node-direction :accessor start-node-direction
	 :initform nil)
  (:slot finish-node :accessor finish-node
	 :initform nil)
  (:slot finish-node-direction :accessor finish-node-direction
	 :initform nil)
  (:slot sub-nodes :accessor sub-nodes
	 :initform nil)
  (:slot road-signs :accessor road-signs
	 :initform nil)
  (:slot display-info :accessor display-info
	 :initform nil))

(defobject roadsign (thingoid)
  (bigness          10)	
  (fuel-drag        0)
  (speed-drag       0)
  (max-load         nil)
  (pavement     'asphalt)
  (condition    'good)
  (bumpy        'nil)
  (length       0)
  (direction    'n)
  (:slot moveable? :initform nil))	; No-one can pick it up!


(defmethod destroy-object ((self map-link))
  nil)

;;; CONSTRUCTORS ***************************************************8


;;; MAKE-ROADSIGN
;;;
;;; Given a road, and a direction to paint on the sign, creates a sign
;;; for the road.
;;;

(defun make-roadsign (road dir)
  (make-sim-object 'roadsign
		   'fuel-drag  (query road 'fuel-drag)
		   'speed-drag  (query road 'speed-drag)
		   'max-load   (query road 'max-load)
		   'pavement   (query road 'pavement)
		   'condition  (query road 'condition)
		   'bumpy      (query road 'bumpy)
		   'length     (query road 'length)
		   'direction  dir))


;;;
;;; MAKE-MAP-LINK
;;;

;;; When making a map link, enter it's name in the map symbol table,
;;; so that it can be referenced when creating it's sector

(defun make-map-link (link-name start-node end-node dir
		      &key (length *default-link-length*)
			   (pavement *default-link-pavement*)
			   (bumpy *default-link-bumpy*)
			   (fuel-drag *default-link-fuel-drag*)
			   (speed-drag *default-link-speed-drag*)
			   (max-load *default-link-max-load*))
  (let ((new-link (make-sim-object 'map-link
				   'id link-name
				   'direction dir
				   'length length
				   'pavement pavement
				   'bumpy bumpy
				   'fuel-drag fuel-drag
				   'speed-drag speed-drag
				   'max-load max-load))
	(start (gethash start-node *map-name-table*))
	(end   (gethash end-node   *map-name-table*)))
    (if (null start)
	(error "~S doesn't name a defined map node" start-node))
    (if (null end)
	(error "~S doesn't name a defined map node" end-node))

    (setf (sub-nodes new-link) (make-array (1+ length) :initial-element nil))
    (connect-road-to-node new-link start 0)
    (connect-road-to-node new-link end length)
    (setf (gethash link-name *map-name-table*) new-link)
    (recompute-road-condition nil new-link)
    new-link))


(defun connect-road-to-node (road node milepost)
  (cond
   ;; Connect the start node
   ((<=  milepost 0)
    (setf (start-node road) node)
    (setf (start-node-direction road) (direction road))
    (connect-road-end-to-node road node (direction road)))
   ;; Connect the finish node
   ((>= milepost (query road 'length))
    (setf (finish-node road) node)
    (setf (finish-node-direction road) 
      (map-direction-opposite (direction road)))
    (connect-road-end-to-node road node 
			      (map-direction-opposite (direction road))))
   ;; Connect a node in the middle
   (t
    (connect-road-end-to-node road node (direction road))
    (connect-road-end-to-node road node 
			      (map-direction-opposite (direction road)))
    ;; Set the position along the road (this assumes the node is a sub-node)
    (setf (milepost new-node) milepost)
    (setf (svref road-vector milepost) node)
    (put-in link new-node))))

(defun connect-road-end-to-node (road node dir)
  (let ((sign (make-roadsign road dir)))
    (put-in node sign)
    (push sign (road-signs road))
    (setf (svref (map-roads node) (map-direction-index dir)) (cons road dir))))


(defun recompute-road-condition (sector link)
  (let ((p-i (or (position (query link 'pavement) '(asphalt dirt gravel))
		 0))
	(w-i (or (position (query sector 'weather) 
			   '(sunny rainy stormy))
		 0)))
    (setp link 'condition (nth p-i (nth w-i *road-condition-table*)))))


(defun accident-probability (road truck)
  (let ((entry (cdr (assoc (query road 'condition)
			   (cdr (assoc (truck-speed truck)
				       *accident-probabilities*)))))
	(tire-adjust (if (truck-tires truck)
			 (accident-adjust (truck-tires truck))
		       '(100 100 100)))
	(night-adjust (if (query (my-sector road) 'daytime)
			  '(100 100 100)
			*default-night-accident-adjust*)))
			  
    (mapcar #'percent-adjust
	    (mapcar #'percent-adjust entry tire-adjust)
	    night-adjust)))
    

;;;
;;; TRUCK-IS-ON/OFF-ROAD
;;;
;;; Used to declare the fact that a particular truck is traveling down a road.
;;; Right now, this is being used to see whether the max-load is being
;;; exceeded among all the trucks.
;;;

(defun truck-is-on-road (road truck)
  (setf (trucks-on-road road) (nunion (trucks-on-road road) (list truck)))
  ;; if the total bigness of all the trucks on the road, is more than the
  ;; max load, then kill all the trucks.
  (when (<= (query road 'max-load)
	    (reduce #'+ (mapcar #'(lambda (truck) (query truck 'gross-bigness))
				(trucks-on-road road))
		    :initial-value 0))
    (mapc #'(lambda (truck) (setf (truck-status truck) 'dead))
	  (trucks-on-road road))))
  
(defun truck-is-off-road (road truck)
  (setf (trucks-on-road road) (delete truck (trucks-on-road road))))

;;; OPERATIONS **************************************************


(defun direction-choice (dir ch1 ch2)
  (cond ((= dir 1) ch1)
	((= dir -1) ch2)
	(t (error "Road Direction not -1 or 1"))))

