;;;
;;; MY-XXXX
;;;
;;; These functions look for the most immediately enclosing XXX of an object.
;;; Both the container Y, and the containment hierarchy in which the object
;;; resides are returned (for consumption by MY-XXX-NEIGHBORS)
;;; If Y directly contains the object, Y and the object are returned.
;;;
;;; Examples:  ( "->" = "contains" )
;;;   A -> B -> C   where A is a map node
;;;     (my-node C) => A,B
;;;     (my-node B) => A,B


(defun myself (obj)
  (values obj (list obj)))

(defun my-container (obj)
  (values (query obj 'container) (list obj (query obj 'container))))
 
;;; VICINITY means immediate container, except when being held by an arm,
;;; in which case it means the container that the arm's gripper is
;;; hovering inside.

(defun my-vicinity (obj)
  (let ((cont (query obj 'container)))
    (if (typep cont 'arm)
	(values (query cont 'claw-container) 
		(list obj cont (query cont 'claw-container)))
      (values cont (list obj cont)))))

(defun my-node (obj)
  (do* ((container obj (query container 'container))
	(self-container (list obj) (cons container self-container)))

      ((or (null container) (class? container 'map-node))
       (values container (nreverse self-container)))))

(defun my-sector (obj)
  (do* ((container obj  (query container 'container))
	(self-container (list obj) (cons container self-container)))
      ((or (null container) (class? container 'map-sector)) 
       (values container (nreverse self-container)))))

(defun my-truck (obj)
  (do* ((container obj (query container 'container))
	(self-container (list obj) (cons container self-container)))
      ((or (null container) (class? container 'truck) )
       (values container (nreverse self-container)))))

;;; This is a little hack.  Since a world is not actually a container,
;;; we can only go up the containment hierarchy to the map level.  So
;;; this function will return *the-world* and the containing map-sector
;;; The corresponding my-world-xxxx will have to work around this hack also.
;;; This assumes only 1 world in the simulation, *the-world*.

(defun my-world (obj)
  (do* ((container obj (query container 'container))
	(self-container (list obj) (cons container self-container)))
      ((or (null container) (class? container 'world-map))
       (values (if (null container)
		   nil
		 *the-world*)
	       (nreverse (cons *the-world* self-container))))))

;;;
;;; MY-XXX-NEIGHBORS
;;;
;;; Returns a list of all objects contained in the most immediately enclosing
;;; XXX, except for OBJ itself, or the container that encloses OBJ.
;;;
;;; Example:
;;;    A -> B         where A is a map node
;;;      -> C -> E
;;;      -> D
;;;  (my-node-neighbors E) => (B D)
;;;  (my-node-neighbors B) => (C D)
;;;

(defun my-container-neighbors (obj)
  (multiple-value-bind (cont self-cont)
      (my-container obj)
    (set-difference (holdings-of-any-nodes cont)
		    self-cont)))

(defun my-vicinity-neighbors (obj)
  (multiple-value-bind (cont self-cont)
      (my-vicinity obj)
    (set-difference (holdings-of-any-nodes cont)
		    self-cont)))

(defun my-node-neighbors (obj)
  (multiple-value-bind (cont self-cont)
      (my-node obj)
    (set-difference (holdings-of-any-nodes cont)
		    self-cont)))

;;; AREA is the enclosing map node, except when in a sub-node (i.e., on a
;;; road), in which case it means the road.  The area neighbors are
;;; all objects contained in sub-nodes along the road.

(defun my-area-neighbors (obj)
  (multiple-value-bind (cont self-cont)
      (my-node obj)
    (set-difference (my-area-contents obj)
		    self-cont)))
   
(defun my-sector-neighbors (obj)
  (multiple-value-bind (cont self-cont)
      (my-sector obj)
    (set-difference (holdings-of-any-nodes cont) self-cont)))

(defun my-truck-neighbors (obj)
  (multiple-value-bind (cont self-cont)
      (my-truck obj)
    (set-difference (holdings-of-any-nodes cont) self-cont)))

(defun my-world-neighbors (obj)
  (multiple-value-bind (cont self-cont)
      (my-world obj)
    ;; If my-world returns anything, it will be a world object: extract the
    ;; map of the world, and use it
    (set-difference (holdings-of-any-nodes cont) self-cont)))

;;;
;;; POINTED-AT-THING (obj)
;;; 
;;; If OBJ is being held by an arm, returns the thing underneath the
;;; arm's gripper.  This can be used to make sensors "point" at particular
;;; objects.
;;;

(defun pointed-at-thing (obj)
  (if (typep (query obj 'container) 'arm)
      (thing-at-arm (query obj 'container))
    nil))

;;;
;;; MY-XXX-CONTENTS
;;;
;;; Returns a list of all things immediately contained in the most 
;;; immediately enclosing XXX.
;;;
;;; Example:
;;;    A -> B         where A is a map node
;;;      -> C -> E
;;;      -> D
;;;  (my-node-contents E) => (B C D)
;;;  (my-node-contents B) => (B C D)
;;;

(defun my-container-contents (obj)
  (holdings (my-container obj)))

(defun my-vicinity-contents (obj)
  (holdings (my-vicinity obj)))

(defun my-node-contents (obj)
  (holdings (my-node obj)))

(defun my-area-contents (obj)
  (let* ((node (my-node obj))
	 (road (query node 'container)))
    (if (typep road 'map-link)
	(apply #'append (mapcar #'holdings (holdings road)))
      (holdings node))))
	 
(defun my-sector-contents (obj)
  (holdings-of-any-nodes (my-sector obj)))

(defun my-truck-contents (obj)
  (holdings (my-truck obj)))

(defun my-world-contents (obj)
  ;; If my-world returns anything, it will be a world object: extract the
  ;; map of the world, and use it
  (multiple-value-bind (cont self-cont)
      (my-world obj)
    (if cont
	(holdings-of-any-nodes (world-map cont))
      nil)))


;;;
;;; SECTOR/NODE/LINK
;;;
;;; Retrieves the named sector/node/link from the world, nil if none
;;;

(defun sector (name &optional (world *the-world*))
  (find-if #'(lambda (s) (eq name (query s 'id)))
	   (sectors (world-map world))))

(defun node (name &optional (world *the-world*))
  (find-if #'(lambda (s) (eq name (query s 'id)))
	   (nodes (world-map world))))

(defun link (name &optional (world *the-world*))
  (find-if #'(lambda (s) (eq name (query s 'id)))
	   (links (world-map world))))

;;;
;;; SECTOR/NODE/LINK-LIST
;;;
;;; Returns lists of the named sectors/nodes/links
;;;

(defun sector-list (names &optional (world *the-world*))
  (mapcar #'(lambda (n) (sector n world)) names))

(defun node-list (names &optional (world *the-world*))
  (mapcar #'(lambda (n) (node n world)) names))

(defun link-list (names &optional (world *the-world*))
  (mapcar #'(lambda (n) (link n world)) names))

;;;
;;; WORLD-SECTORS/NODES/LINKS
;;;
;;; Returns a list of all sectors/nodes/links in the world
;;;

(defun world-sectors (&optional (world *the-world*))
  (sectors (world-map world)))

(defun world-nodes (&optional (world *the-world*))
  (nodes (world-map world)))

(defun world-links (&optional (world *the-world*))
  (links (world-map world)))

;;;
;;; SECTOR-NODES/LINKS
;;;
;;; Returns a list of all nodes/links contained in a sector
;;;

(defun sector-links (sec)
  (remove-if-not #'(lambda (x) (typep x 'map-link)) (query sec 'contents)))

(defun sector-nodes (sec)
  (remove-if-not #'(lambda (x) (typep x 'map-node)) (query sec 'contents)))

;;;
;;; ADJACENCY functions
;;; Positions define what objects are ADJACENT to what other objects
;;; to the left is defined as having a smaller position,
;;; right is a larger position.
;;;

(defun my-adjacent-neighbors (obj)
  (let ((pos (query obj 'position))
	(cont (query obj 'container)))
    (remove nil (list (nth-contents cont (1- pos))
		      (nth-contents cont (1+ pos))))))

(defun my-left-neighbor (obj)
  (nth-contents (query 'obj 'container) (1- (query obj 'position))))

(defun my-right-neighbor (obj)
  (nth-contents (query 'obj 'container) (1+ (query obj 'position))))



;;;**********************************************

(defun holdings-of-any-nodes (self)
  (cond
   ((typep self 'world-map)
    (nconc (mapcan #'holdings (nodes self))
	   (mapcan #'holdings-of-any-nodes (links self))))
   ((typep self 'world)
    (holdings-of-any-nodes (world-map self)))
   ((typep self 'map-sector)
    (mapcan #'holdings-of-any-nodes (holdings self)))
   ((typep self 'map-link)
    (mapcan #'holdings-of-any-nodes (holdings self)))
   (t
    (holdings self))))
