;;;----------------------------------------------------------------------
;;;
;;; ENEMY-UNIT
;;;
;;; Enemy-units are things that can be created via a ENEMY-UNIT-MAKER.
;;; Once created, they watch to see if any trucks drive into their
;;; location.  When they do, each enemy unit tries to capture the truck.
;;; If it succeeds, the truck changes to CAPTURED.
;;; If it fails, the enemy unit will not try to capture the truck again.
;;;
;;; An enemy unit will remain for some time determined by the duration
;;; argument passed to the enemy-unit-maker that created it, unless
;;; a captured truck is in the area.  Enemy units in the same nodes
;;; with captured trucks will remain there until all the enemy units are
;;; somehow destroyed (like with a gun).
;;;
;;; A captured truck will remain captured until there are no more enemy
;;; units in the same node.
;;;

(defvar *default-enemy-unit-bigness* 500)
(defvar *default-enemy-unit-capture-prob* '(prob-dist
					    (0.3 t)
					    (0.7 nil)))

(defobject enemy-unit (thingoid)
  (bigness *default-enemy-unit-bigness*)
  (:slot capture-prob
	 :accessor capture-prob)
  (:slot watched-trucks
	 :accessor watched-trucks
	 :initform nil)
  (:slot truck-watching-process
	 :accessor truck-watching-process))

(defun make-enemy-unit (&key (capture-prob *default-enemy-unit-capture-prob*))
  (make-sim-object 'enemy-unit
		   'id (make-unique-id 'SCUM)
		   'capture-prob capture-prob))

(defmethod destroy-object :before ((self enemy-unit))
  (let* ((node-conts (query (query self 'container) 'contents))
	 (enemies (count-if #'(lambda (x) (typep x 'enemy-unit)) node-conts))
	 (trucks (remove-if-not
		  #'(lambda (x) (and (typep x 'truck)
				    (eq (truck-status x) 'captured)))
		  node-conts)))
    
    ;; If I am the only enemy unit left, then when I get destroyed,
    ;; all captured trucks become freed.
    (when (= enemies 1)
      (dolist (truck trucks)
	(setf (truck-status truck) 'happy)))
    
    (stop-process (truck-watching-process self))))
      

;;;
;;; ENEMY-UNIT-MAKER
;;;
;;; This thing is invoked by an exogenous event.  It randomly places 
;;; enemy units in the world locations given in its enemy-territory slot
;;; If the random location is a node, the enemy units go into the node.
;;; If the random location is a road, the enemy units will go into some
;;;   random sub-node along that road.
;;;

(defobject enemy-unit-maker (sim-object)
  (:slot enemy-territory :accessor enemy-territory))

;;;
;;; MAKE-ENEMY-UNIT-MAKER
;;;
;;; Creates an enemy-unit-maker which will deposit enemy-units into
;;; some location (either a road or a node) in the territory list.
;;;

(defun make-enemy-unit-maker (territory)
  (make-sim-object 'enemy-unit-maker
		   'enemy-territory territory))

;;;
;;; SET-OBJECT
;;;
;;; When set, the enemy-unit-maker deposits new enemy-units in a place
;;;
;;; The args to an enemy-unit maker are:
;;;    - The number of enemy units to create at some random position
;;;         chosen from among the enemy-territory list of the maker
;;;    - A prob-cond that specifies the length of time the enemy
;;;       units will remain in their spot, unless they capture a truck
;;;       (in which case they will remain there until someone destroys
;;;        them).
;;;

(defmethod set-object ((self enemy-unit-maker) setter &rest args)
  (let* ((place (nth (random (length (enemy-territory self)))
		     (enemy-territory self))))
    (cond
     ((not (integerp (first args)))
      (cerror "Ignore, creating no enemy units"
	      "Enemy count: ~S not an integer"
	      (first args)))
	      
     ((null place)
      (cerror "Ignore, creating no enemy units"
	      "~S names no link or node in *the-world* while creating enemy units"
	      place-name))
     
     ((typep place 'map-node)
      (place-enemy-units (first args) (second args) place))
     
     (t
      (place-enemy-units (first args) (second args)
			 (node-at place (random (query place 'length))))))))


(defun place-enemy-units (number dur node)
  (dotimes (i number)
    (place-enemy-unit dur node)))

(defun place-enemy-unit (dur node)
  (let* ((enemy (make-enemy-unit))
	 (start (actual-time))
	 (stop (round (add-noise start dur :lo start))))

    (setf (truck-watching-process enemy)
      (start-process
       nil

       #'(lambda (tok time why)
	   (case why
	     
	     (ADVANCE
	      (when (compare-times time '= start)
		(kick-process tok stop))

	      ;; At the end of our lifetime, try to capture any trucks
	      
	      (when (compare-times time '= stop)
		(try-to-capture-new-trucks enemy))

	      ;; When it is time to leave, we will leave if either:
	      ;;   - we have failed to capture any trucks while we were here
	      ;;   - or there is a captured truck, and at least one other
	      ;;     enemy unit to keep an eye on it.

	      (when (and (compare-times time '>= stop)
			 (or (= 0 (count-if #'(lambda (x)
						(and (typep x 'truck)
						     (eq (truck-status x)
							 'captured)))
					    (query (query enemy 'container)
						   'contents)))
			     (find-if #'(lambda (x) (typep x 'enemy-unit))
				      (my-node-neighbors enemy))))

		;; Normally, here we would call (stop-process tok), but
		;; destroy-object does this for us (since the existence
		;; of this process depends on the existence of the enemy
		;; unit).
		(destroy-object enemy)))))))
		

    (put-in node enemy)))

(defun try-to-capture-new-trucks (enemy)
  ;; Figure out which HAPPY trucks weren't here before
  (let* ((all-trucks (remove-if-not #'(lambda (truck)
					(and (typep truck 'truck)
					     (eq (truck-status truck)
						 'happy)))
				    (query (query enemy 'container) 
					   'contents)))
	 (new-trucks (set-difference all-trucks (watched-trucks enemy)))
	 ;; Pick one of the happy trucks
	 (truck (if new-trucks
		    (nth (random (length new-trucks)) new-trucks)
		  nil)))
    
;;; (format t "  ~S trying to capture ~S~%"  (query enemy 'id) 
;;;                                          (query truck 'id))
    ;; Try to capture that truck
    (if (and truck (process-probability (capture-prob enemy)))
	(setf (truck-status truck) 'captured))
    
    ;; The enemy unit will only make one attempt at capturing one of the
    ;; new trucks, so remember the trucks that we tried to capture.
    ;; Note that this also takes care of the case of a truck driving
    ;; away.  It is effectively removed from the watched-trucks list,
    ;; since it isn't in the all-trucks list.  So when it drives back,
    ;; it will be considered a new truck.
    
    (setf (watched-trucks enemy) all-trucks)))


	
	 
