(defun will-object-fit? (self thing)
  (and (typep thing 'thingoid)
       (or (null (query self 'capacity))
	   (<= (query thing 'gross-bigness) (space-empty self)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; CONTAINER
;;;

(defobject container (thingoid)
  (kind 'container)
  (capacity    nil)			; NIL = unbounded capacity
  (jolt        0 (:effect #'shuffle-container))
					; When set to a number N between
					; 0 and 100, will randomly
					; move N percent of the contents.
  (contents    '() (:effect #'recompute-own-gross-bigness))

  (:slot max-positions			; This is a limitation placed on
	 :accessor max-positions	; containers by the displayer.
	 :initform most-positive-fixnum) ; NOTHING may be placed in this
					; position or higher, or else
					; the displayer blows up.
					; The value must be set correctly
					; according to how many positions
					; the displayer can display.
  (:slot membership-test 
	 :accessor membership-test
	 :initform #'will-object-fit?))
					; If T, any object can be in container
					; If Nil, No object can be put-in
					;  (for fixed content containers)
					; Also can be a predicate which takes
					; the container
					; and the candidate object, and tests
					; whether the thing can be sucessfully
					; placed in the container.


;;; CONSTRUCTORS ;;;;;

(defun make-container (bigness capacity)
  (let ((new-cont (make-sim-object 'container
				   'bigness bigness
				   'capacity capacity)))
    new-cont))


;;;
;;; DESTROY-OBJECT
;;;
;;; When you destroy a container, all things in that container get destroyed
;;; too.
;;;

(defmethod destroy-object :before ((self container))
  (mapcar #'destroy-object (query self 'contents)))


;;; DERIVED ATTRIBUTES ;;;


;;;
;;; COMPUTE-GROSS-BIGNESS
;;;
;;; A container's gross bigness is its own bigness plus the gross bigness
;;; of all it's contents
;;;

(defmethod compute-gross-bigness ((self container))
  (+ (query self 'bigness) 
     (apply #'+
	    (remove-if-not #'numberp
			   (mapcar #'(lambda (o)
				       (query o 'gross-bigness))
				   (holdings self))))))

(defmethod space-full ((self container))
  (apply #'+ (mapcar #'(lambda (obj) (query obj 'gross-bigness))
		     (holdings self))))

(defmethod space-empty ((self container))
  (let ((cap (query self 'capacity)))
    (if cap
	(- cap (space-full self))
      nil)))

(defun holdings (self)
  (remove-if #'null (query self 'contents)))

(defun nth-contents (self n)
  (cond
   ((or (< n 0) (>= n (max-positions self)))
    nil)
   (t
    (nth n (query self 'contents)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; OPERATORS
;;;

;;;
;;; PUT-IN
;;;
;;; Attempts to place an object inside a container.  If this violates
;;; the constraints of the container, returns NIL.  Else, the object
;;; is placed at an arbitrary position within the container, and non-NIL
;;; is returned.
;;;
;;; If FORCE-IT is non-nil, then the object will be placed inside the
;;; container, regardless.  Use with caution!
;;;

(defgeneric put-in (container thing &optional force-it))

(defmethod put-in (anything thing &optional force-it)
  nil)

(defmethod put-in ((self container) thing &optional force-it)
  (let ((old-cont (query thing 'container)))
    (cond
     ((or force-it (will-hold? self thing))
      (setp thing 'container self)
      (maintain-container-invariant thing old-cont self) 
      (unless *update-often*
	(update-displayers old-cont)	;**UPDATE**
	(update-displayers self))	;**UPDATE**
      (send-movement-message thing)
      t)
     (t nil))))


;;;
;;; PUT-IN-AT-POSITION
;;;
;;; Attempts to place an object at a given position inside a container.
;;; If this violates the constraints of the container, or there is already.
;;; an object at the position, returns NIL.  Else, the object
;;; is placed at the position within the container, and non-NIL
;;; is returned.
;;;
;;; If FORCE-IT is non-nil, and there isn't an object at the position,
;;; then the object will be placed inside the container, regardless.
;;; Use with caution!
;;;

(defgeneric put-in-at-position (container position thing &optional force-it))

(defmethod put-in-at-position (anything position thing &optional force-it)
  nil)

(defmethod put-in-at-position ((self container) position thing
			       &optional force-it)
  (let ((old-cont (query thing 'container)))
    (cond

     ((<= (max-positions self) position)
      nil)
     
     ((and (or force-it (will-hold? self thing))
	   (null (nth-contents self position)))
      (setp thing 'container self)
      (maintain-container-position-invariant thing old-cont self position)
      (unless *update-often*
	(update-displayers old-cont)	;**UPDATE**
	(update-displayers self))	;**UPDATE**
      (send-movement-message thing)
      t)

     (t nil))))
					     
;;;
;;; TAKE-OUT
;;;
;;; Removes the given object from the given container.  If the container
;;; doesn't contain the object, or the object is not moveable,
;;; NIL is returned.  Else, the object is removed from the container, 
;;; and non-nil is returned.
;;;

(defgeneric take-out (container thing))

(defmethod take-out (anything thing)
  t)

(defmethod take-out ((self container) thing)
  (cond 
   ((and (moveable? thing) (eq (query thing 'container) self))
    (setp thing 'container nil)
    (maintain-container-invariant thing self nil)
    (unless *update-often*
      (update-displayers self))		;**UPDATE**
    (send-movement-message thing)
    thing)
   (t nil)))

;;;
;;; WILL-HOLD?
;;;
;;; Returns NIL if putting thing in the container violates the
;;; membership-test constraint, or the max-position constraint

(defun will-hold? (self thing)
  (if (typep self 'container)
      (let ((membership (membership-test self)))
	(and
	 (> (max-positions self) (count-if-not #'null (query self 'contents)))
	 (cond
	  ((functionp membership) (funcall membership self thing))
	  (membership   t)
	  (t           nil))))
    nil))


;;;******************************************************
;;;  Do I contain this object (or contain somebody who does)?
;;;  Bay displayers use this to see if they should display 
;;;  the gripper "inside" an object

(defmethod contains-object (self object)
  (declare (ignore object))
  NIL)

(defmethod contains-object ((self container) object)
  (let ((my-holdings (holdings self)))
    (or (member object my-holdings)
        (some #'(lambda (subobj) (contains-object subobj object)) my-holdings))))

;;; OPERATORS ;;;

;;; See messages.lisp for the message-passing mechanism
;;;  (propagate-message-inside isn't that kosher to use, It's kind of private
;;;   to the general broadcast-message mechanism)

(defun send-movement-message (object)
  (propagate-message-inside object t 'movement t))




(defun maintain-container-invariant (object old-container new-container)
  (remove-from-contents old-container object)
  (let ((position (insert-in-contents   new-container object)))
    (setp object 'position position)))

(defun maintain-container-position-invariant (object old-container
					      new-container new-position)
  (remove-from-contents old-container object)
  (let ((position (insert-in-contents-at new-container object new-position)))
    (setp object 'position position)))
   
(defun remove-from-contents (container object)
  (setp container 'contents (nsubstitute nil object 
					(query container 'contents))))

(defun insert-in-contents (container object)
  (cond
   ((null (query container 'contents))
    (setp container 'contents (list object))
    0)
   (t
  
    (do* ((conts   (query container 'contents))
	  (current conts (cdr current))
	  (position 0    (1+ position)))
	((or (null current) (null (car current)))
	 (cond ((null current)
		(setp container 'contents (nconc conts (list object)))
		position)
	       (t
		(setf (car current) object)
		(setp container 'contents conts)
		position)))))))

(defun insure-length (list size)
  (if (< (length list) (1+ size))
      (append list (make-list (- (1+ size) (length list))))
    list))

		 
(defun insert-in-contents-at (container object new-position)
  ;; Make sure contents list is long enough
  
  (let ((conts   (insure-length (query container 'contents) new-position)))
    (cond
     ((nth new-position conts)		     ; If already an object there,
      (insert-in-contents container object)) ; Arbitrarily place in contents
     (t
      (setf (nth new-position conts) object) ; else place at correct spot
      (setp container 'contents conts)	; Kind of bogus, since the setf is
					; destructive, and changes 'contents
					; anyway, but needed for side
					; effects of setp.
      new-position))))
  

