(defvar *stop-server* nil)

;;;
;;; SERVER-LOOP
;;;
;;; This is the driving loop of the simulator.
;;;
;;; It runs until the server crashes (cross your fingers), or until
;;; *stop-server* is non-nil.
;;;
;;; At each iteration, here's what happens:
;;;   New client requests are serviced
;;;   Input from the terminal (if any) is read, evaluated, and printed
;;;   command/control messages from each truck are read and executed
;;;   If the simulator is in real-time, or every truck has issued
;;;     a GO/STEP command, the simulator is advanced.
;;;

(defun server-loop ()
  (loop 
      with earliest
      for now-time      = (real-time)
      for all-advancing = *the-truck-table*
      until *stop-server* do
	
	(setf earliest now-time)
	
	(service-new-client-requests)
	
	(when (listen *terminal-io*)
	  (format *terminal-io* "~S~%" (eval (read *terminal-io*))))

	(loop for the-truck in *the-truck-table* do
	      
	      (loop while (and (null (advance-clock the-truck))
			       (command-message-available? the-truck))
		  do
		    (execute-command the-truck (read-command the-truck)))
	      
	      (loop while (and (null (advance-clock the-truck))
			       (control-message-available? the-truck)
			       ;; EXECUTE-CONTROL will return NIL if
			       ;; the truck leaves the simulation
			       (execute-control the-truck 
						(read-control the-truck))))
      
	      ;; If the truck's controller wants to advance the simulator,
	      ;;  check that time against the other times for the earliest
	      ;;  time to advance to.
	      ;; If not, flag the fact that we are waiting for somebody
	      ;;  to advance.
	      
	      (cond
	       ((null (advance-clock the-truck))
		(setf all-advancing nil))
	       ((typep (advance-clock the-truck) 'sim-time)
		(if (or (null earliest)
			(compare-times (advance-clock the-truck) '< earliest))
		    (setf earliest (advance-clock the-truck))))))
	
	(when (or (and all-advancing *the-truck-table*)
		  (and now-time (compare-times now-time '> (actual-time))))
	  (format *terminal-io* "Advancing simulator to ~S~%" earliest) ; *DEBUG*
	  (let ((boring-world? (advance-simulator earliest)))
	    
	    ;; After advancing the simulator, if any truck has had it's
	    ;; advance-counter reset, or a truck has advanced up to, or past
	    ;; it's advance-counter time, signal that truck with a (TIME)
	    ;; message on its control channel
	    ;; Alternatively, if advancing the simulator would result
	    ;; in nothing interesting, (all trucks say (GO), but there are no
	    ;; events in the queue to go to), then all trucks will be signaled
	    
	    (dolist (truck *the-truck-table*)
	      (let ((ac (advance-clock truck)))
		(when (or boring-world?
			  (null ac)
			  (and (typep ac 'sim-time)
			       (compare-times ac '<= (actual-time *the-world*))))
		  (setf (advance-clock truck) nil)
		  (send-control truck `(TIME ,(actual-time *the-world*))))))))))


(defun stop-server ()
  (setf *stop-server* t))


;;
;; EXECUTE-CONTROL
;;
;; Executes a command that controls the simulator
;;

(defun execute-control (truck cmd)
  (let ((still-here? T))
    (cond 
     ((not (consp cmd))
      (send-control truck '(ERROR BAD-CONTROL-COMMAND)))
       
     ;; When the server is advancing in real time, clients don't
     ;;   have the capability to advance the simulator.
     ((and (real-time-mode?)
	   (or (eq (car cmd) 'go)
	       (eq (car cmd) 'step)))
      (send-control truck '(ERROR GO-NOT-ALLOWED-IN-REAL-TIME)))
   
     ((and (eq (car cmd) 'go)
	   (consp (cdr cmd)))
      (setf (advance-clock truck) (second cmd)))
     
     ((eq (car cmd) 'go)
      (if (> (outstanding-commands truck) 0)
	  (setf (advance-clock truck) t)
	;; A truck cannot ask to advance the simulator (without specifying
	;; the time) unless
	;; the truck is in the process of executing a command, guaranteeing
	;; that the simulator will stop when the command ends.
	(send-control truck '(ERROR NOTHING-TO-DO))))

     ((and (eq (car cmd) 'step) 
	   (consp (cdr cmd)))      
      (setf (advance-clock truck)
	(add-times (actual-time *the-world*)
		   (second cmd))))

     ((eq (car cmd) 'time) 
	(send-control truck `(time ,(actual-time *the-world*))))

     ((eq (car cmd) 'bye)
      (shutdown-truck truck)
      (setf still-here? NIL))
     
     (t (send-control truck '(ERROR BAD-CONTROL-COMMAND))  t))
    still-here?))


(defun service-new-client-requests ()
  (let ((new-channel (accept-client-connection)))
    (when new-channel
      (format *terminal-io* "New truck entering... ")
      (let ((new-truck (create-client-truck new-channel)))
	(when new-truck
	  (push new-truck *the-truck-table*)
	  (format *terminal-io* "Name: ~A~%" (id (car *the-truck-table*))))))))
	

