;;***********************************************************************
;; Parsing and executing simulator commands.  Entry point is 
;;       (EXECUTE-COMMAND command)
;; where the argument is an s-expression, e.g. (TRUCK-TURN SW)
;; 


;;;
;;; *COMMAND-PROCESS-TABLE*
;;;
;;; This is a table that keeps track of which trucks are associated
;;; with processes implementing truck commands.  It is used by STOP-PROCESS.
;;;

(defvar *command-process-table* (make-hash-table))

;;
;; EXECUTE-COMMAND
;;
;; Checks the command syntax, and if OK, enqueues the event that
;; executes the command.
;;

(defun execute-command (truck user-command)
  (multiple-value-bind (error command)
      (standardize-command user-command truck)
    (format *terminal-io* "~A: Received command ~S -> " truck user-command)
    (cond
      (error
       (send-control truck `(error ,error))
       (format *terminal-io* "Error: ~A~%" error))
      
      (t
       (let ((new-event (make-event (actual-time *the-world*)
				    (generate-initiating-event command
							       truck))))
	 (insert-event new-event)
	 (setf (outstanding-commands truck) (1+ (outstanding-commands truck)))
	 (setf (gethash (token new-event) *command-process-table*) truck) 
	 (format *terminal-io* "~A~%" (token new-event))
	 (send-control truck `(PROCESS ,(token new-event))))))))
				 


	     

;;;
;;; GENERATE-INITIATING-EVENT
;;;
;;; From a standardized command, generate the event that will start
;;;  the process associated with that command.
;;;

(defun generate-initiating-event (command truck)
  #'(lambda (token time)
      ;; Get all information needed to manage the process
      (multiple-value-bind (preconds-violated conds update)
	  (generate-process-control command)
	
	(cond
	 (preconds-violated
	  ;; stop the process (which was never created, but has useful
	  ;; side effects, like notifying the truck that its command is
	  ;; complete.)
	  (stop-process token)
	  (format *terminal-io* 
		  "~A: Preconditions violated for command process ~A at ~A : ~S~%" 
		  truck token time preconds-violated))
	 (t
	  ;; Create the process with the given token, conditions, and update
	  ;;   function.
	  (start-process conds update :token token)
	  (format *terminal-io* "~A: Command process ~A started at ~A~%"
		  truck token time))))))

;;;
;;; STOP-PROCESS
;;;
;;; In this simulator, this is how we stop a process.  This enqueues
;;; An event which kills the process at the given time.
;;; Additionally, if this process was implementing a truck command,
;;; the truck will be notified of the command's completion with a
;;; (TERMINATED <token>) message over the control channel, where TOKEN
;;; is the token of the process (the same token sent to the client
;;; when the truck command was sent to the server).
;;;
;;; The token of the event which will stop the process is returned.

(defun stop-process (token &key (at (actual-time *the-world*)))
  (insert-event 
   (make-event at
	       #'(lambda (tok time)
		   (really-stop-process token)))))


(defun really-stop-process (token &optional (time (actual-time)))
  (let ((truck (gethash token *command-process-table*)))
    ;; First, kill the process
    (kill-process token)
    ;; Then, if there is a truck associated with process...
    (when truck
      (format *terminal-io*
	      "~A: Command process ~A terminated at ~A~%"
	      truck token time)
      ;; Decrease the counter of commands-in-progress
      (setf (outstanding-commands truck)
	(1- (outstanding-commands truck)))
      ;; And send the terminate message to the client
      (send-control truck `(terminated ,token))
      (remhash token *command-process-table*))))

;;
;; GENERATE-PROCESS-CONTROL
;;
;; From a standardized command, compute 3 values (in order):
;;   - Whether the preconditions are violated
;;   - The maintainance conditions for the command process
;;   - The update function for the command process
;;
;; To do this, G-P-C just calls the function at the head of the command,
;;  passing the command arguments
;;

(defun generate-process-control (command)
  (apply (first command) (rest command)))


