;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;; CHANGE-SPEED speed
;;;
;;; duration  : 0
;;; preconds  : speed must be valid
;;; effects   : The truck's speed setting is set to SPEED
;;;

(defun change-speed-command (truck speed)
  (cond
   
   ((speed? speed)
    (values
     nil
     nil
     #'(lambda (tok time why)
	 (setf (truck-speed-setting truck) speed)
	 (stop-process tok))))
   
   (t
    (values 'NOT-VALID-SPEED nil nil))))

(install-command 'CHANGE-SPEED '(1) #'speed?
		 #'change-speed-command)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; CHANGE-HEADING heading
;;;
;;; duration  : 0
;;; preconds  : heading must be valid
;;; effects   : The truck's heading setting is set to HEADING
;;;

(defun change-heading-command (truck heading)
  (cond
   
   ((map-direction? heading)
    (values
     nil
     nil
     #'(lambda (tok time why)
	 (setf (truck-heading-setting truck) heading)
	 (stop-process tok))))
   
   (t
    (values 'NOT-VALID-HEADING nil nil))))

(install-command 'CHANGE-HEADING '(1) #'map-direction?
		 #'change-heading-command)



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;; TRUCK-MOVE speed heading
;;;
;;; duration  : depends on many things
;;; preconds  : Truck must be stopped
;;; effects   : The truck's speed is set to SPEED, heading to HEADING,
;;;             and the truck will start moving.
;;;


(defun truck-move-process (truck)
  (let ((speed   (truck-speed-setting truck))
	(heading (truck-heading-setting truck)))
    (cond
     ((not (map-direction? heading))	; speed/heading must be valid
      (values 'heading nil nil nil))
     ((not (speed? speed))
      (values 'speed nil nil nil))
     ((not (eq (truck-speed truck) 'stop)) ; truck can't be moving already
      (values 'not-stopped nil nil nil))
     ((not (neq speed 'stop))		; must give gas to engine
      (values 'must-go-somewhere nil nil nil))
     ((not (equal (query (fuel-tank truck) 'composition) '(gasoline)))
      (values 'tank-not-gas nil nil nil))
     ((not (eq (truck-status truck) 'happy)) ; must be happy
      (values 'not-happy nil nil nil))
     ((not (truck-tires truck))		; must have tires
      (values 'no-tires nil nil nil))
     ((not (every #'folded?		; All arms must be folded
		  (arms truck)))
      (values 'arms-not-folded nil nil nil))
     (t
      (initiate-node-traverse (query truck 'container)
			      truck speed heading)))))
  
;; Given the node to move out of, find the position along the road
;; to begin moving, and the direction to move along the road.
;; If we are at a map-node, then the position will be either the beginning
;; or the end of the road.  If at a map-link-subnode, then we start
;; somewhere from the middle.

(defun initiate-node-traverse (self truck speed heading)
  (let ((chosen-link (svref (map-roads self)
			    (map-direction-index heading))))
    (cond
     ((null chosen-link)  ;; If no road there, the truck crashes
      (values nil nil
	      #'(lambda (tok time why)
		  (setf (truck-heading truck) heading)
		  (setf (truck-status truck) 'rolled)
		  (stop-process tok))))
     ((eq 'map-node (type-of self))
      (initiate-road-traverse (car chosen-link) truck speed
			      (if (eq (start-node (car chosen-link))
				      self)
				  0 
				(query (car chosen-link) 'length))
			      (if (eq (start-node (car chosen-link))
				      self)
				  (direction (car chosen-link))
				(map-direction-opposite
				 (direction (car chosen-link))))))
     (t
      (initiate-road-traverse (car chosen-link) truck speed
			      (milepost self) (cdr chosen-link))))))

;; From the road to travel, figure out the actual mpg, and the first
;; "interesting" location the truck will encounter (node, subnode, accident,
;; or running out of gas).

(defun initiate-road-traverse (road truck speed start compass-direction)
  (let* ((start-time (actual-time))
	 mpg
	 mph
	 (road-dir   (direction road))
	 (dir        (if (eq compass-direction road-dir) 1 -1))
	 (last-loc   start)
	 (last-time  (actual-time *the-world*))
	 (new-node   nil)
	 next-interest
	 next-location
	 next-time)
    
      
    (values
     ;; Preconditions
     nil
     
     ;; Maintainance conditions
     (condition-list 
      (speed-gauge truck)   'value 'TRUCK-SPEED
      (speed-setting truck) 'value 'TRUCK-SPEED-SETTING
      (status-gauge truck)  'value 'TRUCK-STATUS
      (heading-gauge truck) 'value 'TRUCK-HEADING
      (heading-setting truck) 'value 'TRUCK-HEADING-SETTING
      (my-sector road)        'daytime 'DAYTIME
      road 'condition              'ROAD-CONDITION)
     
     
     ;; UPDATE function
     #'(lambda (tok time why)
	 (cond
	  
	  ;; Initialization code
	  ((and (eq why 'advance) (compare-times time '= start-time))
	   
	   ;; Set the truck's speed and heading, but turn off the
	   ;;  corresponding maintainance conditions first, so we don't
	   ;;  invoke ourselves because of it.
	   
	   (disable-condition tok 'truck-speed)
	   (disable-condition tok 'truck-heading)
	   (setf (truck-heading truck)
	     (direction-choice dir 
			       road-dir
			       (map-direction-opposite road-dir)))
	   (setf (truck-speed truck) speed)
	   (enable-condition tok 'truck-speed)
	   (enable-condition tok 'truck-heading)

	   (setf mpg (percent-noise 
		      (percent-noise (travel-mpg truck)
				     (query road 'fuel-drag))
		      (mpg-adjust (truck-tires truck))
		      :lo 0.001))
	   (setf mph (percent-noise 
		      (percent-noise (travel-mph truck)
				     (query road 'speed-drag))
		      (mph-adjust (truck-tires truck))
		      :lo 0.001))

	   ;; Set the system up to compute accident positions
	   (initialize-accident-counter truck road)

	   ;; Declare that the truck is now traveling on the road
	   (truck-is-on-road road truck)
	   
	   ;; Find the first interesting thing to happen to us.
	   (multiple-value-setq (next-interest next-location next-time)
	     (next-senic-turnout (actual-time *the-world*)
				 road truck start dir mpg mph))
	   

	   ;; If the road is bumpy, shuffle the contents of the truck
	   (when (query road 'bumpy)
	     (mapc #'(lambda (x)
		       (jolt-object x *default-bumpy-jolt-intensity*))
		   (bays truck)))
		 
	   ;; make sure the process gets updated when the next
	   ;; interesting time comes around
	   (kick-process tok next-time))
	  
	  ;; NORMAL ADVANCEMENT
	  ((eq why 'advance)

	   ;; Compute time and distance travelled
	   (let* ((deltat (sub-times time last-time))
		  ;; If arriving at interesting point, use that distance,
		  ;; Else, compute from time and speed
		  (deltad (if (compare-times time '= next-time)
			      (abs (- next-location last-loc))
			    (* deltat (/ mph (time-scale truck))))))
	     
	     ;; Update time and place
	     (setf last-loc (+ last-loc (* dir deltad)))
	     (setf last-time time)
	     (setf new-node (node-at road (round last-loc)))

	     ;; Update probability that we could have gotten
	     ;; this far without having an accident
	     
	     (update-accident-counter truck deltad)
	     
	     ;; Burn fuel
	     (pour-out (fuel-tank truck) (* 1.0 (/ deltad mpg)))
	     
	     ;; Crank odometer
	     (setf (truck-odometer truck) (+ (truck-odometer truck) deltad))
	     
	     ;; If we actually moved far enough to get to a new node,
	     ;; move there.
	     
	     (when (not (eq new-node (query truck 'container)))
	       (take-out (query truck 'container) truck)
	       (put-in new-node truck)
	       ;; If the truck has reached a real node, then it has
	       ;; traversed the road, and is off of it.
	       ;; Tell the road this fact.
	       (when (eq (type-of new-node) 'map-node)
		 (truck-is-off-road road truck)))
	     
	     ;; Now, are we at an "interesting" time?
	     ;; (interesting because we reached a node or had an accident)
	     
	     (when (compare-times time '= next-time)
	       
	       ;; Did we have an accident? (i.e. This time is not interesting
	       ;;  because we reached a node, but because something
	       ;;  extraordinary happened).
	       
	       (cond
		((not (typep next-interest 'map-node))
		 (case next-interest
		   
		   ;; Ran out of fuel: Make sure the fuel is gone
		   (OUT-OF-FUEL
		    (pour-out (fuel-tank truck) 
			      (space-full (fuel-tank truck))))
		   
		   ;; Set truck status to specified value
		   (otherwise
		    (setf (truck-status truck) next-interest)))
		 
		 (disable-condition tok 'truck-heading)
		 (setf (truck-heading truck)
		   (map-direction-opposite
		    (direction-choice dir (finish-node-direction road)
				      (start-node-direction road))))
		 (enable-condition tok 'truck-heading)
		 (disable-condition tok 'truck-speed)
		 (setf (truck-speed truck) 'stop)
		 (enable-condition tok 'truck-speed)
		 (stop-process tok))
		  
		;; If we have reached the end of the road...
		((eq (type-of new-node) 'map-node)
		 (disable-condition tok 'truck-heading)
		 (setf (truck-heading truck)
		   (map-direction-opposite
		    (direction-choice dir (finish-node-direction road)
				      (start-node-direction road))))
		 (enable-condition tok 'truck-heading)
		 (disable-condition tok 'truck-speed)
		 (setf (truck-speed truck) 'stop)
		 (enable-condition tok 'truck-speed)
		 (stop-process tok))
		
		;; If we didn't have an accident, and we aren't at the end
		;; of the road, then compute the
		;; next interesting event, and move the stop-event
		;; of the process forward in time to match that event.
		;; Why is reaching a sub-node interesting? because
		;; the truck must at least "pass-through" the node
		;; as it drives, so that passive sensors are given
		;; a chance to check the place out.
		
		(t
		 (multiple-value-setq
		     (next-interest next-location next-time)
		   (next-senic-turnout time road truck last-loc dir 
				       mpg mph))
		 (kick-process tok next-time))))))

	  ;; SOMETHING HAPPENED TO AFFECT THE TRUCK-MOVE
	  
	  ((eq why 'condition)
;;	   (format t "Conditional ~S while moving~%" (name why))
	   (case (name why)

	     (TRUCK-SPEED-SETTING	; Desired speed has changed
	      (setf (truck-speed truck) (truck-speed-setting truck)))
	     
	     (TRUCK-HEADING-SETTING	; Desired heading has changed
	      (setf (truck-heading truck) (truck-heading-setting truck)))
	     
	     (TRUCK-SPEED		; Speed has changed
	      (case (truck-speed truck)
		
		(STOP			; Truck has stopped
		 (stop-process tok))
		
		(otherwise		; Truck has changed speed
		 (setf mpg (percent-noise 
			    (percent-noise (travel-mpg truck)
					   (query road 'fuel-drag))
			    (mpg-adjust (truck-tires truck))
			    :lo 0.001))
		 (setf mph (percent-noise 
			    (percent-noise (travel-mph truck)
					   (query road 'speed-drag))
			    (mph-adjust (truck-tires truck))
			    :lo 0.001))
		 
		 (multiple-value-setq
		     (next-interest next-location next-time)
		   (next-senic-turnout time road truck last-loc dir 
				       mpg mph))
		 (kick-process tok next-time))))
	     
	     (TRUCK-HEADING		; Truck has turned
	      (if (cond
		   ((eq (truck-heading truck) road-dir)
		    (setf dir 1)
		    t)
		   ((eq (truck-heading truck)
			(map-direction-opposite road-dir))
		    (setf dir -1)
		    t)
		   (t nil))
		  
		  ;; Head toward next thing in new direction
		  (progn
		    (multiple-value-setq
			(next-interest next-location next-time)
		      (next-senic-turnout time road truck last-loc dir 
					  mpg mph))
		    (kick-process tok next-time))
		
		;; Truck can't go off-roading
		(progn
		  ;; Make truck crash: stopping the truck-move process
		  ;; will be handled by the TRUCK-STATUS handler
		  (setf (truck-status truck) 'rolled))))
	     
	     ;; The road condition has changed, or the sun has gone up/down
	     ((ROAD-CONDITION DAYTIME)
	      (multiple-value-setq
		  (next-interest next-location next-time)
		(next-senic-turnout time road truck last-loc dir mpg mph))
	      (kick-process tok next-time))
	     
	     ;; Something has upset the truck. Stop moving.
	     (TRUCK-STATUS
	      (when (not (eq (truck-status truck) 'HAPPY))
		(disable-condition tok 'truck-speed)
		(setf (truck-speed truck) 'stop)
		(enable-condition tok 'truck-speed)
		;; If we crashed at an endpoint, then I guess we made it
		;; off of the road.
		(when (eq (type-of (query truck 'container)) 'map-node)
			  (truck-is-off-road road truck))

		(stop-process tok))))))))))
     
;;
;; NEXT-SENIC-TURNOUT
;;
;; Given the ROAD, TRUCK, WHERE the truck is on the road, DIRECTION (+1/-1)
;; and the actual MPG and MPH of the truck, return
;; the reason, distance, and time to the next interesting event in the
;; truck's travels (it reached a location, an accident, ran out of gas, etc.)
;;

(defun next-senic-turnout (time road truck where direction mpg mph)

  ;; Will the truck crush the road?
  ;;; This only crashes a truck if IT's mass exceeds the max load.
  ;;; But right now, the map link is keeping track of all the trucks
  ;;; on a road, to do global crash detection
  
  ;;;  (when (and (query road 'max-load)
  ;;;	     (>= (query truck 'gross-bigness) (query road 'max-load)))
  ;;;    (values 'DEAD 0 0))
  
  ;; If an accident is going to happen, find out where
  (multiple-value-bind (location reason)
      (distance-to-mishap truck)
    
    ;; Will the truck reach a location before that?
    (multiple-value-bind (distance next-node)
	(find-next-node road where direction)
      (when (<= distance location)
	(setf location distance)
	(setf reason next-node)))
    
    ;; Will the truck run out of gas before that?
    (when (< (* (space-full (fuel-tank truck)) mpg) location)
      (setf location (* (space-full (fuel-tank truck)) mpg))
      (setf reason 'OUT-OF-FUEL))
    
    (values reason (+ where (round (* location direction)))
	    (add-times time (round (* (time-scale truck)
				      (/ location mph)))))))

;; FIND-NEXT-NODE
;;
;; On a road, pointed in direction DIRECTION, sitting at milepost FROM-WHERE,
;; find the nearest existing node on the road (excluding the one at 
;; FROM-WHERE). Return the distance and node found.

(defun find-next-node (road from-where direction)
  (let ((road-vector (sub-nodes road))
	(road-length (query road 'length)))
    
    (do ((milepost (+ (round from-where) direction) (+ milepost direction)))
	((or (<= milepost 0) (>= milepost road-length)
	     (typep (svref road-vector milepost) 'map-link-subnode))
	 (values (abs (- milepost from-where))
		 (cond ((<= milepost 0) (start-node road))
		       ((>= milepost road-length) (finish-node road))
		       (t (svref road-vector milepost))))))))


;; DISTANCE-TO-MISHAP
;;
;; given the road conditions and the truck, determine the location and
;; circumstances of the next mishap.
;;
;; --------------------------------------------------------------------
;; * How far can a truck move without crashing?
;; --------------------------------------------
;;
;; This function figures out how far a truck can move down this type of
;;  road before it encounters some type of mishap.  Defined mishaps
;;  are:
;;    ROLLED, STUCK, DEAD, CAPTURED
;;
;; This function returns multiple values => [distance-to-mishap mishap]
;;


(defclass accident-counter
    ()
    ((road      :accessor road
		:initarg :road)
     (fates     :accessor fates
		:initform `(,(/ (random 1000) 1000)
			    ,(/ (random 1000) 1000)
			    ,(/ (random 1000) 1000)))
     (cum-probs :accessor cum-probs
		:initform '(1.0 1.0 1.0))
     (cum-dist  :accessor cum-dist
		:initform 0)))
	  
(defvar *accident-table* (make-hash-table))
       
(defun initialize-accident-counter (truck road)
  (setf (gethash truck *accident-table*) (make-instance 'accident-counter
					 :road road)))
  
(defun update-accident-counter (truck dist)
  (let ((entry (gethash truck *accident-table*)))
    (setf (cum-dist entry)  (+ (cum-dist entry) dist))
    (setf (cum-probs entry)
      (mapcar #'(lambda (cp p)
		  (* cp (expt (/ p 100) (/ dist 100))))
	      (cum-probs entry)
	      (accident-probability (road entry) truck)))))

(defun distance-to-mishap (truck)
  (let* ((entry (gethash truck *accident-table*))
	 (places
	  (mapcar #'(lambda (cum-prob cur-prob fate)
		      (if (and (> fate 0.0) (< cur-prob 100))
			  (+ (cum-dist entry)
			     (* 100
				(/ (log (/ fate cum-prob))
				   (log (/ cur-prob 100)))))
			999999999))
		  (cum-probs entry)
		  (accident-probability (road entry) truck)
		  (fates entry)))
	 (min-place (apply #'min places)))
    (values (- min-place (cum-dist entry))
	    (cond
	     ((= min-place (first places))  'STUCK)
	     ((= min-place (second places)) 'ROLLED)
	     (t 'DEAD)))))
      
  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(install-command 'TRUCK-MOVE '(0) nil #'truck-move-process)
