;;;
;;; *COMMAND-INFO-TABLE*
;;;
;;; A hash table, keyed by the truck-command name, containing
;;; dotted pairs of The number of arguments for each command,
;;; and the function that implements the command.
;;;

(defvar *command-info-table* (make-hash-table))

;;;
;;; INSTALL-COMMAND
;;;
;;; Installs a truck command into the simulator, with the given
;;; name, number of arguments, syntax-check function (which may be nil)
;;; and the implementing function.  The syntax check function is
;;; applied to a command as soon as it is received, so that
;;; syntax and semantic errors can be flagged immediately.
;;;

(defun install-command (name nargs syntax func)
  (setf (gethash name *command-info-table*) (list nargs syntax func)))

;;;
;;; COMMAND-INFO
;;;
;;; Given the name of the command, returns the number of arguments, syntax
;;; function,
;;; and the implementing function as 3 values.  If not a valid command,
;;; returns NIL.
;;;

(defun command-info (name)
  (let ((ci (gethash name *command-info-table*)))
    (if ci
	(values (first ci) (second ci) (third ci))
      nil)))

;;;
;;; STANDARDIZE-COMMAND
;;;
;;; If the syntax of user-command is correct, will return values
;;;   nil <std-cmd>, where <std-cmd> is a list of the command process
;;;     information function, followed by the arguments to the function
;;;  The standardized arguments to a command function are the truck
;;;  that issued the command, followed by the arguments of the command.
;;;
;;; If syntax is violated,
;;;   <error> user-command  is returned.
;;;


(defun standardize-command (user-command truck)
  (if (not (consp user-command))
      (values 'bad-truck-command user-command)

    (multiple-value-bind (nargs syntax func)
	(command-info (car user-command))
      (cond
       ((null nargs)
	(values 'bad-truck-command user-command))
       ((not (check-args (cdr user-command) syntax nargs))
	(values 'bad-truck-command user-command))
       (t
	(values nil
		(cons func (cons truck (rest user-command)))))))))
  
;;;
;;; CHECK-ARGS
;;;
;;; Given a list of actual arguments, and information about the expected
;;;  formal arguments, will say whether or not the actual arguments
;;;  are legal.
;;;
;;; Formal arg info is of the form:
;;;
;;; ( {T | <integer>}* )
;;;
;;; If the number of actuals matches any of the integers in the list,
;;;   they are correct.
;;; The number of actuals, N, match a T if N is >= than the number before
;;;   the T, and <= than the number after the T, with the beginning
;;;   of the list taken to be 0, and the end of the list taken to be infinity.
;;;
;;; After the number of arguments is checked, the arguments are run
;;; through the syntax function (if there is one), which will
;;; return T if they pass, NIL if they fail.
;;;

(defun check-args (actuals syntax formals)
  (when (not (valid-arglist? actuals))
    (values nil))
  
  (when (let ((valid nil)
	      (len (length actuals)))
	  (do* ((before 0 (car current))
		(current formals (cdr current)))
	      ((null current) valid)
	    (if (or (and (numberp (car current))
			 (= len (car current)))
		    (and (eq (car current) t)
			 (>= len before)
			 (or (null (second current))
			     (<= len (second current)))))
		(setf valid t))))
    (if (functionp syntax)
	(apply syntax actuals)
      t)))
      
(defun valid-arglist? (actuals)
  (cond
   ((null actuals)  t)
   ((consp actuals) (valid-arglist? (cdr actuals)))
   (t nil)))
