
(defun find-arm (truck arm-name)
  (find-if #'(lambda (x)
	       (eq (query x 'id) arm-name))
	   (arms truck)))

(defun find-truck-location (truck thing)
  (find-if #'(lambda (x) (eq thing (query x 'id)))
	   `(,(fuel-tank truck) ,(tire-bay truck) ,(weapon-bay truck)
					 ,@(bays truck))))
      
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; ARM-READ arm-name
;;;
;;; duration: 0
;;; Preconds: The arm-name must name one of the truck's arms and
;;;             The arm must not be folded
;;; effect  : The response from READ-OBJECT for whatever object
;;;           is at the gripper's location is sent back to the truck
;;;           send them over the command channel of the truck in the form:
;;;           (SENSOR sensor-id(if any) result)
;;;

(defun arm-read-command (truck arm-name obj-pos)
  (let ((arm (find-arm truck arm-name))
	(start (actual-time)))
    (cond
     ((null arm)
      (values 'NO-SUCH-ARM nil nil))
     ((and (null obj-pos) (folded? arm))
      (values 'ARM-FOLDED nil nil))
     ((not (busy? arm))
      (values
       nil
       nil
       #'(lambda (tok time why)		; This is the process' update function
	   (arm-is-busy arm)
	   (let ((object (if (numberp obj-pos)
			     (nth-contents arm obj-pos)
			   (nth-contents (query arm 'claw-container) 
					 (query arm 'claw-position)))))
	     (send-command truck `(SENSOR ,(query object 'sensor-id)
					  ,(read-object object)))
	     (arm-is-idle arm)
	     (stop-process tok)))))
     (t
      (values 'ARM-FOLDED-OR-BUSY nil nil)))))
       
(install-command 'ARM-READ '(2) #'(lambda (&rest args)
				    (or (null (second args))
					(integerp (second args))))
		 #'arm-read-command)
 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; ARM-SET arm-name set-value*
;;;
;;; duration: 0
;;; Preconds: The arm-name must name one of the truck's arms and
;;;             The arm must not be folded
;;; effect  : The object at the gripper's location is sent the SET-OBJECT
;;;           message, with all of the SET-VALUEs as arguments
;;;


(defun arm-set-command (truck arm-name obj-pos &rest set-values)
  (let ((arm (find-arm truck arm-name))
	(start (actual-time)))
    (cond
     ((null arm)
      (values 'NO-SUCH-ARM nil nil))
     ((not (busy? arm))
      (values
       nil
       nil
       #'(lambda (tok time why)		; This is the process' update function
	   (arm-is-busy arm)
	   (let ((object (if (numberp obj-pos)
			     (nth-contents arm obj-pos)
			   (nth-contents (query arm 'claw-container) 
					 (query arm 'claw-position)))))
	     (apply #'init-set-object object truck set-values)
	     (arm-is-idle arm)
	     (stop-process tok)))))
     (arm
      (values 'ARM-FOLDED-or-busy nil nil)))))


(install-command 'ARM-SET '(2 t) #'(lambda (&rest args)
				    (or (null (second args))
					(integerp (second args))))
		 #'arm-set-command)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; ARM-POUR arm-name n
;;;
;;; duration: Amount in source vessel * Pour speed of arm
;;; preconds: arm-name names an arm, and arm is not folded
;;; effect  : pours the entire contents of the thing within arm at position
;;;           N, into whatever is at the claw-position.  If that thing
;;;           is a vessel, the contents of the source vessel are transferred
;;;           into the destination vessel.
;;;           If the destination is not a vessel, the fluid is lost
;;;

(defun arm-pour-command (truck arm-name source-pos)
  (let* ((arm (find-arm truck arm-name))
	 (start (actual-time)))
    (cond
     ((null arm)
      (values 'NO-SUCH-ARM nil nil))

     ((not (integerp source-pos))
      (values 'POSITION-NOT-INTEGER nil nil))
     
     ((and (not (folded? arm)) (not (busy? arm)))
      (let* ((ok t)
	     (source (nth-contents arm source-pos))
	     (sink   (nth-contents (query arm 'claw-container)
				   (query arm 'claw-position)))
	     (amount (query source 'amount-held))
	     (pour-speed (add-noise (pour-duration arm)
				    (pour-duration-noise arm)
				    :lo 0
				    :args (list arm source sink)))
	     (stop (add-times start
			      (round (* (time-scale truck)
					pour-speed
					(if amount amount 0))))))
	(values
	 nil
	 nil
	 #'(lambda (tok time why)
	     (when (compare-times time '= start)
	       (cond
		((spout-occupied sink) 
		 (setf ok nil)
		 (stop-process tok))
		(t
		 (arm-is-busy arm)
		 (setf (spout-occupied sink) arm)
		 (stop-process tok :at stop))))
	     
	     (when (and ok (compare-times time '>= stop))
	       (pour-in (nth-contents (query arm 'claw-container)
				      (query arm 'claw-position))
			(query (nth-contents arm source-pos) 'composition)
			(pour-out (nth-contents arm source-pos) t))
	       (setf (spout-occupied sink) nil)
	       (arm-is-idle arm))))))
     (t
      (values 'ARM-FOLDED-or-busy nil nil)))))
       
       
(install-command 'ARM-POUR '(2) #'(lambda (&rest args)
				    (integerp (second args)))
		 #'arm-pour-command)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; ARM-MOVE arm-name container position
;;;
;;; duration: number of positions to move * arm's speed
;;; preconds: arm-name names an arm
;;; effect  : see documentation
;;;

(defun arm-move-command (truck arm-name &optional arg1 arg2)
  (let ((arm (find-arm truck arm-name))
	place)
    
    (cond
     ((null arm)
      (values 'NO-SUCH-ARM nil nil))
     
     ((busy? arm)
      (values 'ARM-BUSY nil nil))
     
     ;; Fold arm
     ((eq arg1 'folded)
      (arm-fold-process truck arm))
     
     ;; Move arm outside current container (if not a map node)
     ((and (not (typep (query arm 'claw-container) 'map-node)) 
	   (eq arg1 'outside))
      (arm-move-process truck
			arm 
			(query (query arm 'claw-container) 'container)
			(query (query arm 'claw-container) 'position)
			1))
     
     ;; Move arm inside truck from folded position
     ((and (eq arg1 'inside) (folded? arm))
      (arm-move-process truck
			arm
			truck
			(if arg2 arg2 0)
			1))

     ;; If folded, only INSIDE and OUTSIDE are valid
     ((folded? arm)
      (values 'ARM-FOLDED nil nil))
     
     ;; Move arm inside a container
     ((eq arg1 'inside)
      (arm-move-process truck
			arm
			(nth-contents (query arm 'claw-container)
				      (query arm 'claw-position))
			(if arg2 arg2 0)
			1))
     
     ;; move to a special place in the truck (can only do if arm is within
     ;;  the truck)
     ((and (symbolp arg1) (eq (my-truck (query arm 'claw-container)) truck)
	   (setf place (find-truck-location truck arg1)))
      (if (eq place (fuel-tank truck))
	  ;; Fuel tank
	  (arm-move-process truck arm truck (query place 'position) 1)
	;; other bay
	(arm-move-process truck
			  arm
			  place
			  (if arg2 arg2 0)
			  1)))
     
     ;; move to specified position
     ((numberp arg1)
      (arm-move-process truck
			arm
			(query arm 'claw-container)
			arg1
			(abs (- (query arm 'claw-position) arg1))))
     
     (t
      (values 'BAD-ARM-MOVE-ARGS nil nil)))))

(defun arm-move-process (truck arm dest-cont dest-pos distance)
  (cond
   ((not (integerp dest-pos))
    (values 'POSITION-NOT-INTEGER nil nil))
     
   ((typep dest-cont 'container)
    (real-arm-move-process truck arm dest-cont dest-pos distance))

   (t
    (values 'ARM-DESTINATION-NOT-CONTAINER nil nil))))

(defun real-arm-move-process (truck arm dest-cont wanted-dest-pos distance)
  (let* ((dest-pos (max 0 (min (1- (max-positions dest-cont))
			       wanted-dest-pos)))
	 (start (actual-time))
	 (move-speed (max 0
			  (+ (arm-speed arm)
			     (process-probability (arm-speed-noise arm)
						  arm))))
	 (stop (add-times start
			  (round (* (time-scale truck) move-speed distance)))))
    (values
     nil
     nil
     #'(lambda (tok time why)
	 (when (compare-times start '= time)
	   (arm-is-busy arm)
	   (stop-process tok :at stop))
	 
	 (when (compare-times time '>= stop)
	   (move-arm arm dest-cont dest-pos)
	   (arm-is-idle arm))))))
	   
(defun arm-fold-process (truck arm)
  (let* ((start (actual-time))
	 (move-speed (max 0
			  (+ (arm-speed arm)
			     (process-probability (arm-speed-noise arm) arm))))
	 (stop (add-times start (round (* (time-scale truck)
					  move-speed)))))
  
    (values
     nil
     nil
     #'(lambda (tok time why)
	 (when (compare-times start '= time)
	   (arm-is-busy arm)
	   (stop-process tok :at stop))
	 
	 (when (compare-times time '>= stop)
	   (fold-arm arm)
	   (arm-is-idle arm))))))
	    
(install-command 'ARM-MOVE '(1 2 3) nil #'arm-move-command)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; ARM-GRASP arm-name
;;;
;;; duration: depends on arm
;;; preconds: arm-name names an arm, and arm is not folded
;;;

(defun arm-grasp-command (truck arm-name)
  (let* ((arm (find-arm truck arm-name)))
    (cond
     ((null arm)
      (values 'no-such-arm nil nil))
     
     ((and (not (folded? arm)) (not (busy? arm)))
      (let* ((thing (thing-at-arm arm))
	     (start (actual-time))
	     (lifting-force (query arm 'capacity))
	     (grasp-speed (max 0
			       (+ (grasp-duration arm)
				  (process-probability (grasp-duration-noise arm)
						       arm thing))))
	     (stop (add-times start (round (* (time-scale truck) grasp-speed)))))
    
	(values
	 nil
	 nil
	 #'(lambda (tok time why)
	     (when (compare-times start '= time)
	       (arm-is-busy arm)
	       (if thing
		   (setp thing 'bigness (- (query thing 'bigness) lifting-force))
		 (setf stop start))

	       (stop-process tok :at stop))
	     
	     (when (compare-times time '>= stop)
	       (when thing
		 (setp thing 'bigness (+ (query thing 'bigness) lifting-force))
		 (when (eq thing (thing-at-arm arm))
		   (if (random-chance? 
			(+ (clumsiness arm)
			   (process-probability (clumsiness-noise arm) thing)))
		       (pick-up arm thing)
		     (jolt-object thing *default-arm-jolt-intensity*))))
	       (arm-is-idle arm))))))
     
     (t
      (values 'arm-folded-or-busy nil nil)))))
      

(install-command 'ARM-GRASP '(1) nil #'arm-grasp-command)
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; ARM-UNGRASP arm-name n
;;;
;;; duration: depends on arm
;;; preconds: arm-name names an arm, and arm is not folded
;;;

(defun arm-ungrasp-command (truck arm-name pos)
  (let* ((arm (find-arm truck arm-name))
	 (start (actual-time)))
    
    (cond
     ((null arm)
      (values 'no-such-arm nil nil))
     
     ((not (integerp pos))
      (values 'POSITION-NOT-INTEGER nil nil))
     
     ((and (not (folded? arm)) (not (busy? arm)))
      (let* ((thing (nth-contents arm pos))
	     (grasp-speed (max 0
			       (+ (grasp-duration arm)
				  (process-probability (grasp-duration-noise arm)
						       arm thing))))
	     (stop (add-times start (round (* (time-scale truck) grasp-speed)))))
	(values
	 nil
	 nil
	 #'(lambda (tok time why)
	     (when (compare-times start '= time)
	       (arm-is-busy arm)
	       (stop-process tok :at stop))
	     
	     (when (compare-times time '>= stop)
	       (put-down arm (nth-contents arm pos))
	       (arm-is-idle arm))))))
      
     (t
      (values 'arm-folded-or-busy nil nil)))))
      
(install-command 'ARM-UNGRASP '(2) #'(lambda (&rest args)
				       (integerp (second args)))
		 #'arm-ungrasp-command)



