
(defobject arm (container)
  (claw-container nil)
  (claw-position  nil)
  (:slot moveable? :initform t)
  (:slot arm-speed  :accessor arm-speed
	 :initform 1)
  (:slot arm-speed-noise :accessor arm-speed-noise
	 :initform 0)
  (:slot pour-duration :accessor pour-duration
	 :initform 1)
  (:slot pour-duration-noise :accessor pour-duration-noise
	 :initform 0)
  (:slot grasp-duration :accessor grasp-duration
	 :initform 1)
  (:slot grasp-duration-noise :accessor grasp-duration-noise
	 :initform 0)
  (:slot sensor :accessor sensor)
  (:slot busy? :accessor busy?
	 :initform       nil)
  (:slot clumsiness :accessor clumsiness
	 :initform 100)
  (:slot clumsiness-noise 
	 :accessor clumsiness-noise :initform 0)
  (:slot anchor :accessor anchor
	 :initform nil))


;;; CONSTRUCTORS **************************************************

(defun make-arm (arm-name &key (bigness *default-arm-bigness*)
			       (capacity *default-arm-capacity*)
			       (clumsiness *default-arm-clumsiness*)
			       (clumsiness-noise 
				*default-arm-clumsiness-noise*)
			       (speed *default-arm-speed*)
			       (speed-noise *default-arm-speed-noise*)
			       (pour-duration *default-arm-pour-duration*)
			       (pour-duration-noise 
				*default-arm-pour-duration-noise*)
			       (grasp-duration *default-arm-grasp-duration*)
			       (grasp-duration-noise 
				*default-arm-grasp-duration-noise*)
			       (sensing-method *default-sensing-method*)
			       (sensed-properties 
				*default-arm-sensed-properties*)
			       (sensing-duration *default-sensing-duration*)
			       (sensing-duration-noise *default-sensing-duration-noise*))
  (let* ((new-arm (make-sim-object 'arm
				   'id arm-name
				   'arm-speed speed
				   'arm-speed-noise speed-noise
				   'pour-duration pour-duration
				   'pour-duration-noise pour-duration-noise
				   'clumsiness clumsiness
				   'clumsiness-noise clumsiness-noise
				   'bigness bigness
				   'capacity capacity
				   'max-positions *max-arm-displayer-size*
				   )))
    (setf (sensor new-arm) (make-arm-sensor new-arm
					    sensing-method
					    sensed-properties
					    sensing-duration
					    sensing-duration-noise))
    new-arm))

(defun make-arm-sensor (arm method properties dur dur-noise)
  (let ((new-sensor
	 (make-sensor :sensing-scope      #'(lambda (sensor) (holdings arm))
		      :sensing-method     method
		      :sensed-properties  properties
		      :sensing-duration   dur
		      :sensing-duration-noise dur-noise)))
    (setp new-sensor 'id (intern (format nil "~A-SENSOR" (query arm 'id))))
    new-sensor))
				 


;;; OPERATORS **************************************************


(defun install-arm (arm truck)
  (put-in truck arm t)
  (setf (anchor arm) truck)
  (setp arm 'claw-container truck)
  (setp arm 'claw-position 'folded))
  
(defun fold-arm (arm)
  (setp arm 'claw-container (anchor arm))
  (setp arm 'claw-position  'folded)
  (send-movement-message arm)
  (unless *update-often*
    (update-displayers arm))		;**UPDATE**
  )

(defun folded? (arm)
  (eq (query arm 'claw-position) 'folded))

(defun arm-is-busy (self)
  (setf (busy? self) t))

(defun arm-is-idle (self)
  (setf (busy? self) nil))

(defun thing-at-arm (self)
  (let ((cont (query self 'claw-container))
	(pos  (query self 'claw-position)))
    (if (and (typep cont 'container) (integerp pos))
	(nth-contents cont pos)
      nil)))

;;;
;;; PICK-UP arm object &opt force-it
;;;
;;; This puts an object in the arm's grasp, if the arm has the capacity
;;; to do so, or if force-it is non-nil.  Attempts to grasp the arm's anchor
;;; will fail. The object must be moveable.
;;; Returns nil for failure, non-nil for success.
;;;

(defun pick-up (self object &optional force-it)
  (cond
   ((eq object (anchor self)) nil)
   ((or force-it (will-hold? self object))
    (when (take-out (query object 'container) object)
      (put-in self object force-it)))))

;;;
;;; PUT-DOWN arm object
;;;
;;; This removes the object from the arm's grasp, and places it
;;; at the arm's gripper position and container.  Success/failure
;;; is dependent on whether or not the destination container can hold
;;; the object at the arm's position.
;;; If FORCE-IT is non-nil, the object will be forced into the destination
;;; container.  The put-down can still fail if there is already an object
;;; at that position.
;;; 

(defun put-down (self object &optional force-it)
  (when (and (not (nth-contents (query self 'claw-container)
				(query self 'claw-position)))
	     (or force-it
		 (will-hold? (query self 'claw-container) object)))
    (take-out self object)
    (put-in-at-position (query self 'claw-container)
			(query self 'claw-position) object force-it)))


(defun move-arm (self container position)
  (cond
   ((class? container 'container)
    (setp self 'claw-position position)
    (setp self 'claw-container container)
    (send-movement-message self)
    (unless *update-often*
      (update-displayers self)	)	;**UPDATE**
    )
   (t nil)))



