
;;;
;;; An ACTIVE sensor, like a RECORDINg-SENSOR, is activated whenever
;;; world state changes.  It, however, reports its findings to the truck
;;; who set it.

;;;  Active sensors need (in addition to normal sensor info):
;;;
;;;  - Function that describes the object(s) whose state change will
;;;    trigger the sensor
;;;    NOTE: Right now, only 1 such object is allowed.
;;;  - The property(ies) in above object that triggers the sensor
;;;  - The object to report it's findings to (must accept send-command method,
;;;    like trucks and channels).
;;;
;;;  Each activated active sensor uses a process to watch the objects
;;;

(defobject active-sensor (recording-sensor)
  (:slot watched-objects     :accessor watched-objects)
  (:slot current-watched     :accessor current-watched)
  (:slot watched-properties  :accessor watched-properties)
  (:slot watching-process    :accessor watching-process :initform nil)
  (:slot report-to           :accessor report-to))


;;;
;;; DEFASENSOR
;;;
;;; Allows the user to create new types of active sensors.  The new sensor type
;;; may be customized through the keyword arguments.  If a particular
;;; keyword argument is not supplied, then that value is inherited
;;; from class ACTIVE-SENSOR.
;;;
;;; The argument :EXTRA-PROPERTIES can be a list of property
;;;   definitions as specified by DEFOBJECT.  This is good for
;;;   adding properties which the sensor itself will be sensing.
;;;   (like a sensor that has a SOUND property, and will generate
;;;    a report when it changes as a result of some other object
;;;    broadcasting some SOUND value)

(defmacro defasensor (type &key extra-properties
				(parent-sensor 'active-sensor)
				(sensing-method nil smp)
				(sensed-properties nil spp)
				(sensing-scope nil ssp)
				(watched-objects nil wop)
				(watched-properties nil wpp)
				(sensing-duration nil sdp)
				(sensing-duration-noise nil sdnp))
  `(defobject ,type (,parent-sensor)
     ,@extra-properties
     ,@(create-initforms '(sense-method
			   sense-info
			   visibles
			   watched-objects
			   watched-properties
			   set-duration
			   set-duration-noise)
			 (list sensing-method sensed-properties sensing-scope
			       watched-objects watched-properties
			       sensing-duration sensing-duration-noise)
			 (list smp spp ssp wop wpp sdp sdnp))))


;;;
;;; MAKE-ACTIVE-SENSOR
;;;

(defun make-active-sensor (&key (sensor-id (make-sensor-id))
				(sensing-scope *default-sensing-scope*)
				(sensing-method *default-sensing-method*)
				(sensed-properties *default-sensed-properties*)
				(sensing-duration 0)
				(sensing-duration-noise 0)
				(watched-objects *default-watched-objects*)
				(watched-properties *default-watched-properties*))
  (let ((new-sensor (make-sim-object 'active-sensor
				     'sensor-id sensor-id
				     'sense-method sensing-method
				     'sense-info   sensed-properties
				     'watched-objects watched-objects
				     'watched-properties watched-properties
				     'set-duration sensing-duration
				     'set-duration-noise sensing-duration-noise
				     'visibles  sensing-scope)))
    new-sensor))



;;;
;;; When an active sensor is SET by someone, it remembers who did it, so that
;;; it can send them reports later.
;;;

(defmethod set-object :after ((sensor active-sensor) (setter t) &rest args)
  (setf (report-to sensor) setter))

;;;
;;; After a report has been generated, an active-sensor will
;;; send it to the truck.
;;;

(defmethod perform-sensor-specific-action ((sensor active-sensor))
  (send-command (report-to sensor)
		`(SENSOR ,(query sensor 'sensor-id) ,(read-object sensor))))
   

