;;;;;;;
;;;;;;;
;;;;;;;

;;;
;;; For automatic compilation, set *fasl-default-type* to the proper
;;; extension.
;;;

(defvar *fasl-default-type* "sfasl4")

(defvar *lisp-default-type* "lisp")

;;;
;;; *TRUCKWORLD-DIR* tells the system where the truckworld files
;;; reside.
;;;

(defvar *truckworld-dir* "/projects/ai/truckworld/")


;;;
;;; RUN-DEMO
;;;
;;; After loading the integrated simulator (see LOAD-INTEGRATED-SIMULATOR,
;;; below), RUN-DEMO will start a session that allows truck commands to be
;;; entered interactively.
;;;
;;; RUN-DEMO accepts the key argument :DISPLAY, specifying the X-display to
;;; use for the truck displayer.  This argument is a string that names
;;; the X-server.  Put no display or screen spec in this string.  Examples:
;;;
;;;   "deer"  NOT    "deer:0.0" or "deer:0"
;;;   "goby.cs.washington.edu"   NOT  "goby.cs.washington.edu:0.0"
;;;
;;; If no display argument is given, no truckworld display will be used
;;;
;;;        Insert description of interactive client here
;;;


(defun run-demo (&rest keys &key (world "demo-world") display)
  (apply #'start-interactive-simulator 
	 :world world
	 :truck-form '(make-truck :truck-id demo-truck)
	 keys))
			       
  


;;;
;;; LOAD-SERVER
;;;
;;; This loads the truckworld simulator
;;; If COMPILE is non-nil, then changed lisp files involved are compiled
;;;

(defun load-server (&optional compile)
  (setf *compile-truckworld* compile)
  (load-control)
  (load-domain)
  (load-display)
  (setf *compile-truckworld* nil))

;;;
;;; LOAD-CLIENT
;;;
;;; This loads the client interface, which client applications use
;;; to communicate with the server
;;; If COMPILE is non-nil, then changed lisp files involved are compiled
;;;

(defun load-client (&optional compile)
  (setf *compile-truckworld* compile)
  (load-list *ipc-dir* *ipc-files*)
  (load-list *client-dir* *client-files*)
  (setf *compile-truckworld* nil))

;;;
;;; LOAD-INTEGRATED-SIMULATOR
;;;
;;; This loads both the server and client interface code into the same
;;; lisp process.  Aside from a few initialization differences, the
;;; client application should be able to run exactly the same as in
;;; the client/server model, with a few restrictions:
;;;
;;;  - The real-time mode of the simulator is not supported, as this
;;;    requires that code for the client and code for the server be running
;;;    "concurrently".  "Concurrently" can either mean doing some clever
;;;    control flow juggling between client and server (more trouble than
;;;    it's worth), or using the multiprocessing capabilities of the
;;;    Common Lisp implementation (non-portable, and you might as well
;;;    just use the client/server mode).
;;;
;;; Anyway, the entire system was designed around a separate client and
;;; server, running in separate lisp jobs, using IPC to communicate.
;;; The integrated simulator should only be used for playing with
;;; truckworld, or demonstrating it, or when the implementation of lisp
;;; doesn't support IPC communication.  For serious applications,
;;; use the client/server mode.
;;;
;;; If COMPILE is non-nil, then changed lisp files involved are compiled
;;;

(defun load-integrated-simulator (&optional compile)
  (setf *compile-truckworld* compile)
  ;; This bit is common to client and server
  (load-list *ipc-dir* *ipc-files*)
  
  ;; These bits are for the server
  (load-list *control-dir* *control-files*)
  (load-list *domain-dir* *domain-files*)
  (load-list *display-dir* *display-files*)
  
  ;; These bits are for the client
  (load-list *client-dir* *client-files*)
  
  ;; This is the patch file that allows client and server to run
  ;; together
  (load-list *integrated-dir* *integrated-files*)
  
  (setf *compile-truckworld* nil))


;;;
;;; LOAD-WORLD world &optional force-load
;;;
;;; World is a string representing the world file (default directory
;;; is the world directory) to load.  The file is not loaded if it was
;;; the most recent world file to be loaded.  If FORCE-LOAD is non-nil, then
;;; the world is loaded regardless.
;;;

(defvar *current-world-file* nil)
(defvar *current-world-date* 0)
(defvar *force-world-load* nil)

(defun load-world (world &optional force-load)
  (let* ((world-file (merge-pathnames world *world-dir*))
	 (world-date (file-write-date world-file)))
    (when (or force-load
	      *force-world-load*
	      (not (equal world-file *current-world-file*)))
      (load world-file)
      (setf *current-world-date* world-date)
      (setf *current-world-file* world-file)))
  *current-world-file*)

;;;
;;; LOAD-EXTENSION extension &optional force-load
;;;
;;; Insures that certain definitions contained in a file are loaded
;;; into the system.
;;;
;;; extension is a lisp file that contains some extension to truckworld.
;;; The default directory is the extension directory.  A list of
;;; loaded extension files is kept in memory, and an extension file will
;;; only be loaded if it does not appear in the list.  If FORCE-LOAD is
;;; non-nil, the extension file will be loaded regardless
;;;

(defvar *current-extension-files* nil)

(defun load-extension (extension &optional force-load)
  (let* ((ext-file (merge-pathnames extension *extension-dir*)))
    (when (or force-load
	      *force-world-load*
	      (not (member ext-file *current-extension-files* :test #'equal)))
      (load ext-file)
      (push ext-file *current-extension-files*))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; File definitions for the simulator
;;;

(defparameter *domain-dir* (merge-pathnames "simulator/" *truckworld-dir*))
(defparameter *domain-files*
    '("rectangles" "lisp-additions" "lowlevel" "locality"
      "defaults" "probability"
      "sim-object" "thingoid" "controller" "container" "vessel" "shuffle"
      "world" "truck" "communication"
      "map" "map-sector" "map-node" "map-link" 
      "map-link-subnode" "transport" "messages"
      "arm" "sensor" "recording-sensor" "active-sensor" "misc-objects" 
      "truck-support" "enemy-unit"
      "cmd-install" "cmd-sensors" "cmd-arms"  "cmd-move"
      "server-stuff" "server-only"
      "realtime" "command-loop" "command-execution"))

(defparameter *control-dir* (merge-pathnames "control/" *truckworld-dir*))
(defparameter *control-files*
    '("support" "errors"
      "token-table" "event" "event-queue" "process-def"
      "process" "properties" "handler-info" "handler-queue"
      "simulation-loop" "initialize" "exogeny-manager"
      "generator" "standard-events"))


(defparameter *display-dir* (merge-pathnames "displayer/graphic/" 
					     *truckworld-dir*))
(defparameter *display-files*
    '("primitives" "interface" "drawing" "method-defs" 
      "arms" "bays" "loc" "truck" "gauges" "maps" "toplevel"))

;;; This is required to resolve some name conflicts between the XLIB package
;;; and the displayer code.

(shadow '(color display))

(defparameter *ipc-dir* (merge-pathnames "ipc/" *truckworld-dir*))
(defparameter *ipc-files*
    '("connection-man" "channel" "client" "server"))

(defparameter *client-dir* (merge-pathnames "client-server/" *truckworld-dir*))
(defparameter *client-files* 
    '("client-only" "sample-clients"))

(defparameter *integrated-dir* (merge-pathnames "client-server/" *truckworld-dir*))
(defparameter *integrated-files* '("integrated"))


(defparameter *world-dir* (merge-pathnames "worlds/" *truckworld-dir*))

(defvar *compile-truckworld* nil)

(defun source-newer? (path)
  (let* ((source   (merge-pathnames (make-pathname :type *lisp-default-type*)
				    path))
	 (compiled (merge-pathnames (make-pathname :type *fasl-default-type*)
				    path))
	 (sdate (file-write-date source))
	 (cdate (file-write-date compiled)))
    (cond
     
     ;; Where are the files?
     ((and (null sdate) (null cdate))
      (error "Source and Compiled files missing: ~S" path))
     
     ;; No compiled file? the compile it!
     ((null cdate)    t)

     ;; No source file, but a compiled one? Strange, but OK, load compiled
     ((null sdate)   nil)
     
     (t  (> (file-write-date source) (file-write-date compiled))))))


(defun load-list (dir files)
    
  (mapcar #'(lambda (file)
	      (let ((fullpath (merge-pathnames file dir)))
		(if (and *compile-truckworld* 
			 (source-newer? fullpath))
		    (compile-file fullpath))
		(load fullpath)))
	  files))

(defun load-control ()
  (load-list *control-dir* *control-files*))

(defun load-domain ()
  (load-list *ipc-dir* *ipc-files*)
  (load-list *domain-dir* *domain-files*))

(defun load-display ()
  (load-list *display-dir* *display-files*))


(defparameter *truckworld-bitmap-directory*
    (merge-pathnames "bitmaps/" *display-dir*))

(defparameter *extension-dir* (merge-pathnames "extensions/" *truckworld-dir*))
